﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/Math.hpp>

//------------------------------------------------------------------------------
namespace lib {

//@{
/// 乱数クラス。
class Random
{
public:

    ///@name シードの型。
    //@{
    struct Seed
    {
        /// シードのデフォルト値。
        static const Seed Default();
        uint32_t x;
        uint32_t y;
        uint32_t z;
        uint32_t w;
    };
    //@}

    /// @name 生成・破棄
    //@{
    /// シードを指定して乱数を作成する。
    explicit Random(const Seed& aSeed = Seed::Default());
    //@}

    // シード取得
    const Seed randSeed();
    // シード設定
    const Seed randSeed(const Seed& aSeed);

    // @name 乱数
    // 内部で発生する乱数を参考にmin以上とterm未満(maxの場合はmax以下)の乱数を取得する。
    // 例：min=0,term=6 -> 0,1,2,3,4,5を返す
    //     min=1.0f,max=2.0f -> 1.0f以上、2.0f未満の値を返す
    //
    //@{
    bool randBool(float aPrTrue);          ///< bool版(prTrue の確率で true を返す)
    int rand(int aTerm);                   ///< int版。aMin=0。
    int rand(int aMin, int aTerm);         ///< int版
    float randF(float aMin, float aMax);   ///< float版
    uint32_t randU32(uint32_t aMax);                 ///< uint32_t版。aMin=0。
    uint32_t randU32(uint32_t aMin, uint32_t aTerm);      ///< uint32_t版
    float randNF();                           ///< 0.0f <= val <= 1.0fの乱数。
    float randAF();                           ///< -1.0f <= val <= 1.0fの乱数。
    int operator()(int aTerm);             ///< (0, aTerm)のintを返す。std::random_shuffle適合版。

    /// @brief Enum版
    ///
    /// @return 0から順に終端までが定義されたEnum型に対し、
    /// 開始値から終端の1つ前の値までのうち1つを与えた型でランダムに返す。
    ///
    /// <pre>
    /// <code>
    /// enum Test { Test_A, Test_B, ... Test_Z, Test_End };
    /// </code>
    /// に対し、randEnum&lt;Test&gt;(Test_End) は、Test_A から Test_Z のうち
    /// 任意の1つの値を、Test型で返す。
    /// </pre>
    template <class TEnumType> TEnumType randEnum(TEnumType end) { return static_cast<TEnumType>(rand(end));  }

    /// @brief 配列の任意の要素を取得する。
    ///
    /// @return 配列のうち任意の値。参照ではなく値を返すことに注意。
    template <class TArray, int TCount> TArray randArrayVal(const TArray (&ary)[TCount]) { return ary[rand(TCount)]; }
    //@}

private:
    uint32_t randCoreU32();                      ///< 内部で発生する乱数
    float randCoreF32();                      ///< 0.0f <= val <= 1.0f に正規化された乱数

    Seed mSeed;
};
//@}

} // namespace
// EOF
