﻿// 文字コード:UTF-8
/// @file
#include <lib/Random.hpp>

//------------------------------------------------------------------------------
#include <lib/debug/Assert.hpp>

//------------------------------------------------------------------------------
namespace lib {

//------------------------------------------------------------------------------
const Random::Seed Random::Seed::Default()
{
    Seed obj = {};
    obj.x = 123456789;
    obj.y = 362436069;
    obj.z = 521288629;
    obj.w = 88675123;
    return obj;
}

//------------------------------------------------------------------------------
Random::Random(const Seed& aSeed)
: mSeed(aSeed)
{
}

//------------------------------------------------------------------------------
const Random::Seed Random::randSeed()
{
    return mSeed;
}

//------------------------------------------------------------------------------
const Random::Seed Random::randSeed(const Seed& aSeed)
{
    mSeed = aSeed;
    return mSeed;
}

//------------------------------------------------------------------------------
bool Random::randBool(const float aPrTrue)
{
    if (aPrTrue == 0) {
        return false;
    }
    return randNF() <= aPrTrue;
}

//------------------------------------------------------------------------------
int Random::rand(const int aTerm)
{
    LIB_ASSERT_NOT_EQUAL(aTerm, 0);
    return int(randCoreU32() & 0x7FFFFFFF) % aTerm;
}

//------------------------------------------------------------------------------
int Random::rand(const int aMin, const int aTerm)
{
    LIB_ASSERT(aMin <= aTerm);
    return aMin + rand(aTerm - aMin);
}

//------------------------------------------------------------------------------
float Random::randF(const float aMin, const float aMax)
{
    LIB_ASSERT(aMin <= aMax);
    return aMin + randCoreF32() * (aMax - aMin);
}

//------------------------------------------------------------------------------
uint32_t Random::randU32(const uint32_t aTerm)
{
    LIB_ASSERT_NOT_EQUAL(aTerm, 0);
    return randCoreU32() % aTerm;
}

//------------------------------------------------------------------------------
uint32_t Random::randU32(const uint32_t aMin, const uint32_t aTerm)
{
    LIB_ASSERT(aMin <= aTerm);
    return aMin + randU32(aTerm - aMin);
}

//------------------------------------------------------------------------------
float Random::randNF()
{
    return randCoreF32();
}

//------------------------------------------------------------------------------
float Random::randAF()
{
    return randF(-1.0f, 1.0f);
}

//------------------------------------------------------------------------------
int Random::operator() (const int aMax)
{
    return rand(aMax);
}

//------------------------------------------------------------------------------
uint32_t Random::randCoreU32()
{
    uint32_t t = uint32_t();
    t = (mSeed.x ^ (mSeed.x << 11));
    mSeed.x = mSeed.y;
    mSeed.y = mSeed.z;
    mSeed.z = mSeed.w;
    mSeed.w = (mSeed.w ^ (mSeed.w >> 19)) ^ (t ^ (t >> 8));
    return mSeed.w;
}

//------------------------------------------------------------------------------
float Random::randCoreF32()
{
    return float(randCoreU32() % 0x10000) / float(0x10000);
}

} // namespace
// EOF
