﻿// 文字コード:UTF-8
/// @file
#pragma once

#include "lib/Vector3.hpp"

LIB_FORWARD_DECLARE_2(lib, class Matrix4x4);

namespace lib {

class Matrix4x3
{
public:
    inline static Matrix4x3 Identity();
    inline static Matrix4x3 Translate(const Vector3& aVec);
    inline static Matrix4x3 Scale(const Vector3& aVec);
    // ・単位行列で生成される
    inline Matrix4x3();
    inline Matrix4x3(
        float a00, float a01, float a02,
        float a10, float a11, float a12,
        float a20, float a21, float a22,
        float a30, float a31, float a32
        );
    ~Matrix4x3() {}
    inline Matrix4x3& operator*=(const Matrix4x3& aMtx);
    inline const Matrix4x3 operator*(const Matrix4x3& aMtx) const;
    void set(int aCol, int aRow, float aVal) { m[aCol][aRow] = aVal; }
    void setT(const Vector3& aVec) { m[3][0] = aVec.x; m[3][1] = aVec.y; m[3][2] = aVec.z; }
    Vector3 getT() const { return Vector3(m[3][0], m[3][1], m[3][2]); }
    Matrix4x4 toMatrix4x4() const;
    inline bool isTranslateOnly() const;
    float at(int aCol, int aRow) const { return m[aCol][aRow]; }
private:
    float m[4][3];
};

//------------------------------------------------------------------------------
Matrix4x3 Matrix4x3::Identity()
{
    return Matrix4x3(
        1.0f, 0.0f, 0.0f,
        0.0f, 1.0f, 0.0f,
        0.0f, 0.0f, 1.0f,
        0.0f, 0.0f, 0.0f);
}

//------------------------------------------------------------------------------
Matrix4x3 Matrix4x3::Translate(const Vector3& aVec)
{
    return Matrix4x3(
        1.0f, 0.0f, 0.0f,
        0.0f, 1.0f, 0.0f,
        0.0f, 0.0f, 1.0f,
        aVec.x, aVec.y, aVec.z);
}

//------------------------------------------------------------------------------
Matrix4x3 Matrix4x3::Scale(const Vector3& aVec)
{
    return Matrix4x3(
        aVec.x, 0.0f, 0.0f,
        0.0f, aVec.y, 0.0f,
        0.0f, 0.0f, aVec.z,
        0.0f, 0.0f, 0.0f);
}

//------------------------------------------------------------------------------
Matrix4x3::Matrix4x3()
{
    *this = Identity();
}

//------------------------------------------------------------------------------
Matrix4x3::Matrix4x3(
    float a00, float a01, float a02,
    float a10, float a11, float a12,
    float a20, float a21, float a22,
    float a30, float a31, float a32)
{
    m[0][0] = a00; m[0][1] = a01; m[0][2] = a02;
    m[1][0] = a10; m[1][1] = a11; m[1][2] = a12;
    m[2][0] = a20; m[2][1] = a21; m[2][2] = a22;
    m[3][0] = a30; m[3][1] = a31; m[3][2] = a32;
}

//------------------------------------------------------------------------------
Matrix4x3& Matrix4x3::operator*=(const Matrix4x3& aMtx)
{
    *this = *this * aMtx;
    return *this;
}

//------------------------------------------------------------------------------
const Matrix4x3 Matrix4x3::operator*(const Matrix4x3& aMtx) const
{
    Matrix4x3 mtx;
    mtx.m[0][0] = m[0][0] * aMtx.m[0][0] + m[0][1] * aMtx.m[1][0] + m[0][2] * aMtx.m[2][0];
    mtx.m[0][1] = m[0][0] * aMtx.m[0][1] + m[0][1] * aMtx.m[1][1] + m[0][2] * aMtx.m[2][1];
    mtx.m[0][2] = m[0][0] * aMtx.m[0][2] + m[0][1] * aMtx.m[1][2] + m[0][2] * aMtx.m[2][2];

    mtx.m[1][0] = m[1][0] * aMtx.m[0][0] + m[1][1] * aMtx.m[1][0] + m[1][2] * aMtx.m[2][0];
    mtx.m[1][1] = m[1][0] * aMtx.m[0][1] + m[1][1] * aMtx.m[1][1] + m[1][2] * aMtx.m[2][1];
    mtx.m[1][2] = m[1][0] * aMtx.m[0][2] + m[1][1] * aMtx.m[1][2] + m[1][2] * aMtx.m[2][2];

    mtx.m[2][0] = m[2][0] * aMtx.m[0][0] + m[2][1] * aMtx.m[1][0] + m[2][2] * aMtx.m[2][0];
    mtx.m[2][1] = m[2][0] * aMtx.m[0][1] + m[2][1] * aMtx.m[1][1] + m[2][2] * aMtx.m[2][1];
    mtx.m[2][2] = m[2][0] * aMtx.m[0][2] + m[2][1] * aMtx.m[1][2] + m[2][2] * aMtx.m[2][2];

    mtx.m[3][0] = m[3][0] * aMtx.m[0][0] + m[3][1] * aMtx.m[1][0] + m[3][2] * aMtx.m[2][0] + aMtx.m[3][0];
    mtx.m[3][1] = m[3][0] * aMtx.m[0][1] + m[3][1] * aMtx.m[1][1] + m[3][2] * aMtx.m[2][1] + aMtx.m[3][1];
    mtx.m[3][2] = m[3][0] * aMtx.m[0][2] + m[3][1] * aMtx.m[1][2] + m[3][2] * aMtx.m[2][2] + aMtx.m[3][2];
    return mtx;
}

//------------------------------------------------------------------------------
bool Matrix4x3::isTranslateOnly() const
{
    return
        m[0][0] == 1.0f && m[0][1] == 0.0f && m[0][2] == 0.0f &&
        m[1][0] == 0.0f && m[1][1] == 1.0f && m[1][2] == 0.0f &&
        m[2][0] == 0.0f && m[2][1] == 0.0f && m[2][2] == 1.0f;
}

} // namespace

// EOF

