﻿// 文字コード:UTF-8
/// @file
#pragma once

LIB_FORWARD_DECLARE_2(lib, template<class TType> class IntrusiveList);

//------------------------------------------------------------------------------
namespace lib {

/// 侵入リスト「IntrusiveList」の、リスト化されるものの内部にリストにされるための構造
template<class TType>
class IntrusiveListNode
{
public:
    typedef IntrusiveListNode<TType> This;
    typedef IntrusiveList<TType> Container;
    // 通常構築
    IntrusiveListNode(TType* aOwner);
    // 末端ノード用特殊構築(内部利用)
    IntrusiveListNode(Container* aContainer);
    ~IntrusiveListNode() { removeSelfIfPossible(); }
    TType& owner() { SYS_ASSERT(mOwner); return *mOwner; }
    const TType& owner() const { SYS_ASSERT(mOwner); return *mOwner; }
    bool isLinked() const { return mContainer != nullptr; }
    bool hasNext() const { SYS_ASSERT(mContainer); return mNext != &mContainer->terminal(); }
    This& next() { SYS_ASSERT(hasNext()); return *mNext; }
    const This& next() const { return const_cast<This*>(this)->next(); }
    This* nextPtr() { return (hasNext() ? &next() : 0); }
    const This* nextPtr() const { return (hasNext() ? &next() : 0); }
    bool hasPrevious() const { SYS_ASSERT(mContainer); return mPrev != &mContainer->terminal(); }
    This& previous() { SYS_ASSERT(hasPrevious()); return *mPrev; }
    const This& previous() const { return const_cast<This*>(this)->previous(); }
    This* previousPtr() { return (hasPrevious() ? &previous() : 0); }
    const This* previousPtr() const { return (hasPrevious() ? &previous() : 0); }
    void insertAfterSelf(This* aNode);
    void insertBeforeSelf(This* aNode);
    void removeSelf();
    void removeSelfIfPossible();
private:
    TType* mOwner;
    Container* mContainer;
    This* mNext;
    This* mPrev;
};

//------------------------------------------------------------------------------
template<class TType>
IntrusiveListNode<TType>::IntrusiveListNode(TType* aOwner)
: mOwner(aOwner)
, mContainer(0)
, mNext(this)
, mPrev(this)
{
    SYS_ASSERT_POINTER(aOwner);
}

//------------------------------------------------------------------------------
template<class TType>
IntrusiveListNode<TType>::IntrusiveListNode(Container* aContainer)
: mOwner(0)
, mContainer(aContainer)
, mNext(this)
, mPrev(this)
{
    SYS_ASSERT_POINTER(aContainer);
}

//------------------------------------------------------------------------------
template<class TType>
void IntrusiveListNode<TType>::insertAfterSelf(This* aNode)
{
    SYS_ASSERT_POINTER(aNode);
    SYS_ASSERT(!aNode->isLinked());
    SYS_ASSERT(!aNode->mContainer);
    SYS_ASSERT(mContainer);
    aNode->mContainer = mContainer;
    aNode->mPrev = this;
    aNode->mNext = mNext;
    mNext->mPrev = aNode;
    mNext = aNode;
}

//------------------------------------------------------------------------------
template<class TType>
void IntrusiveListNode<TType>::insertBeforeSelf(This* aNode)
{
    SYS_ASSERT_POINTER(aNode);
    SYS_ASSERT(!aNode->isLinked());
    SYS_ASSERT(!aNode->mContainer);
    SYS_ASSERT(mContainer);
    aNode->mContainer = mContainer;
    aNode->mPrev = mPrev;
    aNode->mNext = this;
    mPrev->mNext = aNode;
    mPrev = aNode;
}

//------------------------------------------------------------------------------
template<class TType>
void IntrusiveListNode<TType>::removeSelf()
{
    SYS_ASSERT(isLinked());
    SYS_ASSERT(mContainer);
    mNext->mPrev = mPrev;
    mPrev->mNext = mNext;
    mPrev = this;
    mNext = this;
    mContainer = 0;
}

//------------------------------------------------------------------------------
template<class TType>
void IntrusiveListNode<TType>::removeSelfIfPossible()
{
    if (isLinked()) {
        removeSelf();
    }
}

} // namespace
// EOF
