﻿// 文字コード:UTF-8
/// @file
#include "SplitCommand.hpp"

//------------------------------------------------------------------------------
namespace {

// 有効とするコマンド
// @ref ROWSER1-1340
// @{

static const ::lib::ShortString48 fCommandList[] = {
    "--help",                             // Help表示
    "web",                                // WebApplet呼び出します。
    "offline",                            // WebOfflineApplet呼び出します。
    "--callback-url",                     // コールバックURL
    "--callbackable-url",                 // コールバック可能URL
    "--left-stick-mode",                  // 左スティックモードを指定します。
    "--user-index",                       // 作成済みのユーザー番号を指定します。
    "--background-kind",                  // 背景の種類を指定します。
    "--boot-display-kind",                // 起動時の画面の種類を指定します。
    "--display-url-kind",                 // 画面に表示される URL の表記形式を指定します。
    "--footer-enabled",                   // フッターを利用しません
    "--pointer-enabled",                  // 左スティックオードのポインター切り替えを不許可にします。
    "--boot-as-media-player",             // 動画専用起動モードにします。
    "--js-extension-enabled",             // JS拡張を利用可能にします。
    "--key-repeat-delay-frame",           // キーリピート開始までのフレーム数を指定します。
    "--key-repeat-pulse-frame",           // キーリピート間隔フレーム数を指定します。
    "--user-agent-additional-string",     // ユーザーエージェントに追加する文字列を指定します。
    "--touch-enabled-on-contents",        // コンテンツのタッチ操作を有効にするかどうか指定します。
    "--media-player-auto-close-enabled",  // 動画が最後まで再生された際に、動画プレイヤーを自動的に終了するかどうかを設定します。
    "--web-audio-enabled",                // Web Audioの利用設定
    "--page-cache-enabled",               // ページキャッシュを有効にするかどうか指定します。
    "--footer-fixed-kind",                // フッター表示のデフォルトの制御方法を指定します。
    "--page-fade-enabled",                // ページ間フェードを有効にするかどうかを指定します。
    "--boot-loading-icon-enabled",        // 起動時にローディングアイコンを表示するかどうかを指定します。
    "--page-scroll-indicator-enabled",    // ページスクロールインジケータの表示を有効にするかどうかを指定します。
    "--media-player-speed-control-enabled",// 動画再生時、再生速度変更を有効にするかどうか指定します。
};
LIB_ASSERT_ARRAY_NUM(fCommandList, int(SplitCommand::CommandKind::TERM));

// @}

}

//------------------------------------------------------------------------------
SplitCommand::SplitCommand(int aArgCount, char** aArgValue)
: mItemArray()
, mRequiresHelp()
{
    resetImpl(aArgCount, aArgValue);
}

//------------------------------------------------------------------------------
SplitCommand::~SplitCommand()
{
}

//------------------------------------------------------------------------------
#if DEBUG_IS_ENABLED
void SplitCommand::reset(int aArgCount, char** aArgValue)
{
    resetImpl(aArgCount, aArgValue);
}
#endif

//------------------------------------------------------------------------------
bool SplitCommand::hasKey(const char* aKey) const
{
    SYS_ASSERT_POINTER(aKey);
    return getItemIndex(aKey) > -1;
}

//------------------------------------------------------------------------------
bool SplitCommand::hasKey(CommandKind aKind) const
{
    return getItemIndex(aKind) > -1;
}

//------------------------------------------------------------------------------
int SplitCommand::getInt(const char* aKey) const
{
    SYS_ASSERT_POINTER(aKey);
    auto i = getItemIndex(aKey);
    SYS_ASSERT(i > -1);
    return atoi(mItemArray[i].mValue.readPtr());
}

//------------------------------------------------------------------------------
int SplitCommand::getInt(CommandKind aKind) const
{
    return getInt(ToReadPtr(aKind));
}

//------------------------------------------------------------------------------
int SplitCommand::getIntOrDefaultValue(const char* aKey, int aDefaultValue) const
{
    SYS_ASSERT_POINTER(aKey);
    SYS_ASSERT_POINTER(aDefaultValue);
    auto i = getItemIndex(aKey);
    if (i > -1) {
        return atoi(mItemArray[i].mValue.readPtr());
    }
    return aDefaultValue;
}

//------------------------------------------------------------------------------
int SplitCommand::getIntOrDefaultValue(CommandKind aKind, int aDefaultValue) const
{
    return getIntOrDefaultValue(ToReadPtr(aKind), aDefaultValue);
}

//------------------------------------------------------------------------------
const char* SplitCommand::getString(const char* aKey) const
{
    SYS_ASSERT_POINTER(aKey);
    auto i = getItemIndex(aKey);
    SYS_ASSERT(i > -1);
    return mItemArray[i].mValue.readPtr();
}

//------------------------------------------------------------------------------
const char* SplitCommand::getString(CommandKind aKind) const
{
    return getString(ToReadPtr(aKind));
}

//------------------------------------------------------------------------------
const char* SplitCommand::getStringOrDefaultValue(const char* aKey, const char* aDefaultValue) const
{
    SYS_ASSERT_POINTER(aKey);
    SYS_ASSERT_POINTER(aDefaultValue);
    auto i = getItemIndex(aKey);
    if (i > -1) {
        return mItemArray[i].mValue.readPtr();
    }
    return aDefaultValue;
}

//------------------------------------------------------------------------------
const char* SplitCommand::getStringOrDefaultValue(CommandKind aKind, const char* aDefaultValue) const
{
    return getStringOrDefaultValue(ToReadPtr(aKind), aDefaultValue);
}

//------------------------------------------------------------------------------
bool SplitCommand::getBool(const char* aKey) const
{
    SYS_ASSERT_POINTER(aKey);
    auto i = getItemIndex(aKey);
    SYS_ASSERT(i > -1);
    ::lib::ShortString32 trueStr("true");
    return trueStr.isEquals(mItemArray[i].mValue.readPtr());
}

//------------------------------------------------------------------------------
bool SplitCommand::getBool(CommandKind aKind) const
{
    return getBool(ToReadPtr(aKind));
}

//------------------------------------------------------------------------------
bool SplitCommand::getBoolOrDefaultValue(const char* aKey, bool aDefaultValue) const
{
    SYS_ASSERT_POINTER(aKey);
    auto i = getItemIndex(aKey);
    ::lib::ShortString32 trueStr("true");
    if (i > -1) {
        return trueStr.isEquals(mItemArray[i].mValue.readPtr());
    }
    return aDefaultValue;
}

//------------------------------------------------------------------------------
bool SplitCommand::getBoolOrDefaultValue(CommandKind aKind, bool aDefaultValue) const
{
    return getBoolOrDefaultValue(ToReadPtr(aKind), aDefaultValue);
}

//------------------------------------------------------------------------------
::lib::ShortString48 SplitCommand::ToString(CommandKind aKind)
{
    SYS_ASSERT(aKind <= CommandKind::TERM);
    return fCommandList[int(aKind)];
}

//------------------------------------------------------------------------------
const char* SplitCommand::ToReadPtr(CommandKind aKind)
{
    SYS_ASSERT(aKind <= CommandKind::TERM);
    return fCommandList[int(aKind)].readPtr();
}

//------------------------------------------------------------------------------
int SplitCommand::getItemIndex(const char* aKey) const
{
    SYS_ASSERT_POINTER(aKey);
    for (int i = 0; i < mItemArray.count(); ++i) {
        if (mItemArray.at(i).mKey.isEquals(aKey)) {
            return i;
        }
    }
    return -1;
}

//------------------------------------------------------------------------------
int SplitCommand::getItemIndex(CommandKind aKind) const
{
    return getItemIndex(ToReadPtr(aKind));
}

//------------------------------------------------------------------------------
void SplitCommand::dump()
{
#if DEBUG_IS_ENABLED
    if (mItemArray.count()) {
        DEBUG_LOG("\n--- [SplitCommand] Dump ---\n");
        for (const auto* item : mItemArray) {
            DEBUG_LOG("%s: %s\n", item->mKey.readPtr(), item->mValue.readPtr());
        }
        DEBUG_LOG("\n");
    }
#endif
}

//------------------------------------------------------------------------------
void SplitCommand::resetImpl(int aArgCount, char** aArgValue)
{
    mItemArray.clear();

    // 引数をItemにセットします。
    for (auto i = 1; i < aArgCount; ++i) {
        bool isSuccess = false;

        // --help は 値を持たない為 別枠で拾います。
        if (ToString(CommandKind::Help).isEquals(aArgValue[i])) {
            mItemArray.add(new Item(::lib::ShortString48(aArgValue[i]), ValueType()));
            isSuccess = true;
            continue;
        }

        for (auto cmd : fCommandList) {
            if (cmd.isEquals(aArgValue[i])) {
                if (i + 1 < aArgCount){
                    // web --help || offline --help のパターンは別コマンドとして扱います。
                    if (needsDisplayHelp(aArgValue[i], aArgValue[i + 1])) {
                        mItemArray.add(new Item(::lib::ShortString48(aArgValue[i]), ValueType()));
                        isSuccess = true;
                        break;
                    }
                    else {
                        mItemArray.add(new Item(::lib::ShortString48(aArgValue[i]), ValueType(aArgValue[i + 1])));
                        ++i;
                        isSuccess = true;
                        break;
                    }
                }
            }
        }
        if (isSuccess) {
            continue;
        }
    }

    // web, offline のみ指定した場合はヘルプを表示します。
    if (mItemArray.count() == 0 && aArgCount == 2 && needsDisplayHelp(aArgValue[1])) {
        mItemArray.add(new Item(::lib::ShortString48(aArgValue[1]), ValueType()));
        mItemArray.add(new Item(ToString(CommandKind::Help), ValueType()));
        mRequiresHelp = true;
    }

    DEBUG_EVAL(dump());
}

//------------------------------------------------------------------------------
bool SplitCommand::needsDisplayHelp(const char* aCurArg)
{
    SYS_ASSERT_POINTER(aCurArg);
    return ToString(CommandKind::Web).isEquals(aCurArg) ||
        ToString(CommandKind::Offline).isEquals(aCurArg);
}

//------------------------------------------------------------------------------
bool SplitCommand::needsDisplayHelp(const char* aCurArg, const char* aNextArg)
{
    SYS_ASSERT_POINTER(aCurArg);
    SYS_ASSERT_POINTER(aNextArg);
    return (ToString(CommandKind::Web).isEquals(aCurArg) ||
        ToString(CommandKind::Offline).isEquals(aCurArg)) &&
        ToString(CommandKind::Help).isEquals(aNextArg);
}

//------------------------------------------------------------------------------
// EOF
