﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/rwlock.h>
#include <sys/lock.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>

static MALLOC_DEFINE(M_RWLOCK, "irwlock", "RW lock");

struct rwlock_internal {
    nn::os::MutexType mutex;
    nn::os::EventType reader_unlocked;
    volatile int reader_count;
    nn::os::ThreadType *owner;    // Note, this is invalid for checking read/shared owner
};

static void lock_rw(struct lock_object *lock, uintptr_t how)
{
    if (how) {
        rw_rlock((struct rwlock*)lock);
    } else {
        rw_wlock((struct rwlock*)lock);
    }
}

static uintptr_t unlock_rw(struct lock_object *lock)
{
    struct rwlock* rw = (struct rwlock*)lock;
    if (((struct rwlock_internal*)rw->siglo_ptr)->reader_count > 0) {
        rw_runlock(rw);
        return 1;
    } else {
        rw_wunlock(rw);
    }
    return 0;
}

struct lock_class lock_class_rw = {
    .lc_name     = "rw",
    .lc_flags    = 0,
    .lc_assert   = NULL,
    .lc_ddb_show = NULL,
    .lc_lock     = lock_rw,
    .lc_owner    = NULL,
    .lc_unlock   = unlock_rw
};

void rw_init_l(struct rwlock *rw, const char *name, int opts)
{
    struct rwlock_internal* lock =
        (struct rwlock_internal*)malloc(
                sizeof(struct rwlock_internal),
                M_RWLOCK,
                M_ZERO | M_WAITOK);
    NN_ABORT_UNLESS(lock != NULL);
    nn::os::InitializeMutex(&lock->mutex, opts & RW_RECURSE ? true : false, 0);
    nn::os::InitializeEvent(&lock->reader_unlocked, false, nn::os::EventClearMode_AutoClear);
    rw->siglo_ptr = (void*)lock;
    lock->owner = NULL;
    lock_init(&rw->lock_object, &lock_class_rw, name, NULL, 0);
}

void rw_init_flags_l(struct rwlock *rw, const char *name, int opts)
{
    rw_init_l(rw, name, opts);
}

void rw_destroy_l(struct rwlock *rw)
{
    struct rwlock_internal* lock = (struct rwlock_internal*)(rw->siglo_ptr);
    lock->owner = NULL;
    nn::os::FinalizeMutex(&lock->mutex);
    nn::os::FinalizeEvent(&lock->reader_unlocked);
    free((void*)rw->siglo_ptr, M_RWLOCK);
}

void rw_rlock_l(struct rwlock *rw)
{
    struct rwlock_internal* lock = (struct rwlock_internal*)(rw->siglo_ptr);
    nn::os::LockMutex(&lock->mutex);
    __atomic_add_fetch(&lock->reader_count, 1, __ATOMIC_RELAXED);
    lock->owner = NULL;
    nn::os::UnlockMutex(&lock->mutex);
}

void rw_runlock_l(struct rwlock *rw)
{
    struct rwlock_internal* lock = (struct rwlock_internal*)(rw->siglo_ptr);
    __atomic_sub_fetch(&lock->reader_count, 1, __ATOMIC_RELAXED);
    nn::os::SignalEvent(&lock->reader_unlocked);
}

void rw_wlock_l(struct rwlock *rw)
{
    struct rwlock_internal* lock = (struct rwlock_internal*)(rw->siglo_ptr);
    nn::os::LockMutex(&lock->mutex);
    while (lock->reader_count > 0) {
        nn::os::WaitEvent(&lock->reader_unlocked);
    }
    lock->owner = nn::os::GetCurrentThread();
}

void rw_wunlock_l(struct rwlock *rw)
{
    struct rwlock_internal* lock = (struct rwlock_internal*)(rw->siglo_ptr);
    if (lock->mutex._nestCount < 2)
        lock->owner = NULL;
    nn::os::UnlockMutex(&lock->mutex);
}

int rw_try_wlock_l(struct rwlock *rw)
{
    struct rwlock_internal* lock = (struct rwlock_internal*)(rw->siglo_ptr);
    if (nn::os::TryLockMutex(&lock->mutex)) {
        if (lock->reader_count == 0) {
            lock->owner = nn::os::GetCurrentThread();
            return true;
        }
        nn::os::UnlockMutex(&lock->mutex);
    }
    return false;
}

void rw_downgrade_l(struct rwlock *rw)
{
    struct rwlock_internal* lock = (struct rwlock_internal*)(rw->siglo_ptr);
    __atomic_add_fetch(&lock->reader_count, 1, __ATOMIC_RELAXED);
    lock->owner = NULL;
    nn::os::UnlockMutex(&lock->mutex);
}

int rw_initialized_l(const struct rwlock *rw)
{
    struct rwlock_internal* lock = (struct rwlock_internal*)(rw->siglo_ptr);
    return (lock != NULL && lock->mutex._state == nn::os::MutexType::State_Initialized);
}

void rw_sysinit(void *arg)
{
    struct rw_args *rwargs = (struct rw_args*)arg;
    rw_init((struct rwlock*)rwargs->ra_rw, rwargs->ra_desc);
}

void rw_sysinit_flags_l(void *arg)
{
    struct rw_args_flags *rwargs = (struct rw_args_flags*)arg;
    rw_init_flags((struct rwlock*)rwargs->ra_rw, rwargs->ra_desc, rwargs->ra_flags);
}

#ifdef INVARIANTS
void _rw_assert(const struct rwlock *c, int what, const char *file, int line)
{
    if (what & (RA_RECURSED | RA_NOTRECURSED)) {
        panic("%s: RA_RECURSED and RA_NOTRECURED not supported", __FUNCTION__);
    }

    struct rwlock_internal* lock = (struct rwlock_internal*)(rw->siglo_ptr);
    nn::os::ThreadType *curr = nn::os::GetCurrentThread();

    switch (what) {
    case RA_LOCKED:
        if ((lock->reader_count == 0) && (lock->owner == NULL)) {
            panic("Lock %s not locked @ %s:%d\n",
                  rw->lock_object.lo_name, file, line);
        }
        break;
    case RA_RLOCKED:
        if (lock->reader_count == 0 || (lock->owner != NULL)) {
            panic("Lock %s not read locked @ %s:%d\n",
                  rw->lock_object.lo_name, file, line);
        }
        break;
    case RA_WLOCKED:
        if (lock->owner != curr) {
            panic("Lock %s not write locked @ %s:%d\n",
                  rw->lock_object.lo_name, file, line);
        }
        break;
    case RA_UNLOCKED:
        // Can't reliably check to see if we are read locked, so we do not try to.
        if (lock->owner == curr) {
            panic("Lock %s is write locked @ %s:%d\n",
                rw->lock_object.lo_name, file, line);
        }
        break;
    default:
        panic("%s(%d) not implemented lock %s @ %s:%d\n",
              __FUNCTION__, what, rw->lock_object.lo_name, file, line);
    }

}
#endif

}

