﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#define _RLIMIT_IDENT
#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/resourcevar.h>

/*
 * Change the total socket buffer size a user has used.
 */
int chgsbsize(struct uidinfo *uip, u_int *hiwat, u_int to, rlim_t max)
{
    int diff = to - *hiwat;
    if (diff > 0) {
        if (atomic_fetchadd_long((u_long*)&uip->ui_sbsize, (u_long)diff) + diff > max) {
            atomic_subtract_long((u_long*)&uip->ui_sbsize, (u_long)diff);
            return 0;
        }
    } else {
        atomic_add_long((u_long*)&uip->ui_sbsize, (u_long)diff);
        if (uip->ui_sbsize < 0) {
            printf("negative sbsize for uid = %d\n", uip->ui_uid);
        }
    }
    *hiwat = to;
    return 1;
}

void movesbsize(struct uidinfo *to, struct uidinfo *from, u_int sbsize)
{
    atomic_fetchadd_long((u_long*)&to->ui_sbsize,   (u_long)sbsize);
    atomic_subtract_long((u_long*)&from->ui_sbsize, (u_long)sbsize);
}

/*
 * Return the current (soft) limit for a particular system resource.
 * The which parameter which specifies the index into the rlimit array
 */
rlim_t lim_cur(struct proc *p, int which)
{
    rlim_t r = 0;

    if(p == NULL){
        return r;
    }

    switch (which) {
    case RLIMIT_SBSIZE:
    case RLIMIT_TCP_SEND_SBSIZE:
    case RLIMIT_TCP_RECEIVE_SBSIZE:
    case RLIMIT_UDP_SEND_SBSIZE:
    case RLIMIT_UDP_RECEIVE_SBSIZE:
    case RLIMIT_SB_EFFICIENCY:
    case RLIMIT_LIB_COMPATIBILITY:
        r = p->p_limit->pl_rlimit[which].rlim_cur;
        break;
    default:
        printf("rlim_cur %s is not supported\n", rlimit_ident[which]);
        break;
    }
    return r;
}


rlim_t lim_max(struct proc *p, int which)
{
    rlim_t r = 0;

    if(p == NULL){
        return r;
    }

    switch (which) {
    case RLIMIT_TCP_SEND_SBSIZE:
    case RLIMIT_TCP_RECEIVE_SBSIZE:
    case RLIMIT_UDP_SEND_SBSIZE:
    case RLIMIT_UDP_RECEIVE_SBSIZE:
    case RLIMIT_SB_EFFICIENCY:
    case RLIMIT_LIB_COMPATIBILITY:
        r = p->p_limit->pl_rlimit[which].rlim_max;
        break;
    default:
        printf("rlim_max %s is not supported\n", rlimit_ident[which]);
        break;
    }
    return r;
}

}

