﻿/*-
 * Copyright (c) 1982, 1986, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)time.h	8.5 (Berkeley) 5/4/95
 * $FreeBSD$
 */

#ifndef _SYS_BSDTIME_H_
#define    _SYS_BSDTIME_H_

#include_next <sys/time.h>

#ifdef _KERNEL

#include <sys/types.h>

struct bintime {
    time_t    sec;
    uint64_t frac;
};

static __inline void
bintime_addx(struct bintime *_bt, uint64_t _x)
{
    uint64_t _u;

    _u = _bt->frac;
    _bt->frac += _x;
    if (_u > _bt->frac)
        _bt->sec++;
}

static __inline void
bintime_add(struct bintime *_bt, const struct bintime *_bt2)
{
    uint64_t _u;

    _u = _bt->frac;
    _bt->frac += _bt2->frac;
    if (_u > _bt->frac)
        _bt->sec++;
    _bt->sec += _bt2->sec;
}

static __inline void
bintime_sub(struct bintime *_bt, const struct bintime *_bt2)
{
    uint64_t _u;

    _u = _bt->frac;
    _bt->frac -= _bt2->frac;
    if (_u < _bt->frac)
        _bt->sec--;
    _bt->sec -= _bt2->sec;
}

static __inline void
bintime_mul(struct bintime *_bt, u_int _x)
{
    uint64_t _p1, _p2;

    _p1 = (_bt->frac & 0xffffffffull) * _x;
    _p2 = (_bt->frac >> 32) * _x + (_p1 >> 32);
    _bt->sec *= _x;
    _bt->sec += (_p2 >> 32);
    _bt->frac = (_p2 << 32) | (_p1 & 0xffffffffull);
}

static __inline void
bintime_shift(struct bintime *_bt, int _exp)
{

    if (_exp > 0) {
        _bt->sec <<= _exp;
        _bt->sec |= _bt->frac >> (64 - _exp);
        _bt->frac <<= _exp;
    } else if (_exp < 0) {
        _bt->frac >>= -_exp;
        _bt->frac |= (uint64_t)_bt->sec << (64 + _exp);
        _bt->sec >>= -_exp;
    }
}

#define bintime_clear(a)    ((a)->sec = (a)->frac = 0)
#define bintime_isset(a)    ((a)->sec || (a)->frac)
#define bintime_cmp(a, b, cmp)                          \
    (((a)->sec == (b)->sec) ?                           \
        ((a)->frac cmp (b)->frac) :                     \
        ((a)->sec cmp (b)->sec))

#define SBT_1S      ((sbintime_t)1 << 32)
#define SBT_1M      (SBT_1S * 60)
#define SBT_1MS     (SBT_1S / 1000)
#define SBT_1US     (SBT_1S / 1000000)
#define SBT_1NS     (SBT_1S / 1000000000)
#define SBT_MAX     0x7fffffffffffffff

static __inline void
bintime2timespec(const struct bintime *_bt, struct timespec *_ts)
{
    _ts->tv_sec = _bt->sec;
    _ts->tv_nsec = ((uint64_t)1000000000 * (uint32_t)(_bt->frac >> 32)) >> 32;
}

#define FREQ2BT(freq, bt)                                           \
{                                                                   \
    (bt)->sec = 0;                                                  \
    (bt)->frac = ((uint64_t)0x8000000000000000  / (freq)) << 1;     \
}

static __inline sbintime_t
bttosbt(const struct bintime _bt)
{
    return (((sbintime_t)_bt.sec << 32) + (_bt.frac >> 32));
}
static __inline void
bintime2timeval(const struct bintime *_bt, struct timeval *_tv)
{
    _tv->tv_sec = _bt->sec;
    _tv->tv_usec = ((uint64_t)1000000 * (uint32_t)(_bt->frac >> 32)) >> 32;
}
static __inline struct timeval
sbttotv(sbintime_t _sbt)
{
    struct timeval _tv;

    _tv.tv_sec = _sbt >> 32;
    _tv.tv_usec = ((uint64_t)1000000 * (uint32_t)_sbt) >> 32;
    return (_tv);
}

static __inline sbintime_t
tvtosbt(struct timeval _tv)
{
    return (((sbintime_t)_tv.tv_sec << 32) +
        (_tv.tv_usec * (((uint64_t)1 << 63) / 500000) >> 32));
}

static __inline sbintime_t
tstosbt(struct timespec _ts)
{
    return (((sbintime_t)_ts.tv_sec << 32) +
        (_ts.tv_nsec * (((uint64_t)1 << 63) / 500000000) >> 32));
}

#define timevalclear(tvp)        ((tvp)->tv_sec = (tvp)->tv_usec = 0)
#define timevalisset(tvp)        ((tvp)->tv_sec || (tvp)->tv_usec)
#define timevalcmp(tvp, uvp, cmp)                               \
            (((tvp)->tv_sec == (uvp)->tv_sec) ?                 \
            ((tvp)->tv_usec cmp (uvp)->tv_usec) :               \
            ((tvp)->tv_sec cmp (uvp)->tv_sec))

#ifdef __SigloBSD__
extern volatile time_t get_current_milliseconds();
#define                time_second ((get_current_milliseconds() + 500)/1000)
#define                time_uptime (time_second - boottimebin.sec)
#else  // __SigloBSD__
extern volatile time_t time_second;
extern volatile time_t time_uptime;
#endif // __SigloBSD__
extern struct bintime  boottimebin;
extern sbintime_t tick_sbt;
void bintime(struct bintime *bt);
void microtime(struct timeval *tvp);

void getbinuptime(struct bintime *bt);
#define binuptime(bt)   getbinuptime(bt)

void getmicrouptime(struct timeval *tvp);
static __inline sbintime_t
getsbinuptime(void)
{
    struct bintime _bt;
    getbinuptime(&_bt);
    return (bttosbt(_bt));
}
void getmicrotime(struct timeval *tvp);
int  ppsratecheck(struct timeval *, int *, int);
int  ratecheck(struct timeval *, const struct timeval *);
void timevaladd(struct timeval *t1, const struct timeval *t2);
void timevalsub(struct timeval *t1, const struct timeval *t2);
uint64_t tvtohz(struct timeval *tv);
sbintime_t microseconds_to_sbt(int64_t us);
int64_t    sbt_to_microseconds (sbintime_t sbt);
uint64_t cputicks_to_usec(int64_t cpu_ticks);
int64_t  sbintime_to_cputicks(sbintime_t time);
int  itimerfix(struct timeval *tv);

#endif /* _KERNEL */

#endif /* !_SYS_BSDTIME_H_ */
