﻿/*-
 * Copyright (c) 1986, 1989, 1991, 1993
 *    The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *    @(#)proc.h    8.15 (Berkeley) 5/19/95
 * $FreeBSD$
 */

#ifndef _SYS_PROC_H_
#define    _SYS_PROC_H_

#ifdef _KERNEL

#include <sys/event.h>            /* For struct klist. */
#include <sys/_lock.h>
#include <sys/_mutex.h>
#include <sys/resource.h>
#include <sys/sigio.h>
#include <sys/signal.h>
#include <sys/pcpu.h>
#include <sys/ucred.h>
#include <vm/uma.h>

/*-
 * Description of a process.
 *
 * This structure contains the information needed to manage a thread of
 * control, known in UN*X as a process; it has references to substructures
 * containing descriptions of things that the process uses, but may share
 * with related processes.  The process structure and the substructures
 * are always addressable except for those marked "(CPU)" below,
 * which might be addressable only on a processor on which the process
 * is running.
 *
 * Below is a key of locks used to protect each member of struct proc.  The
 * lock is indicated by a reference to a specific character in parens in the
 * associated comment.
 *      * - not yet protected
 *      a - only touched by curproc or parent during fork/wait
 *      b - created at fork, never changes
 *        (exception aiods switch vmspaces, but they are also
 *        marked 'P_SYSTEM' so hopefully it will be left alone)
 *      c - locked by proc mtx
 *      d - locked by allproc_lock lock
 *      e - locked by proctree_lock lock
 *      f - session mtx
 *      g - process group mtx
 *      h - callout_lock mtx
 *      i - by curproc or the master session mtx
 *      j - locked by proc slock
 *      k - only accessed by curthread
 *    k*- only accessed by curthread and from an interrupt
 *      l - the attaching proc or attaching proc parent
 *      m - Giant
 *      n - not locked, lazy
 *      o - ktrace lock
 *      q - td_contested lock
 *      r - p_peers lock
 *      t - thread lock
 *      x - created at fork, only changes during single threading in exec
 *      y - created at first aio, doesn't change until exit or exec at which
 *          point we are single-threaded and only curthread changes it
 *      z - zombie threads lock
 *
 * If the locking key specifies two identifiers (for example, p_pptr) then
 * either lock is sufficient for read access, but both locks must be held
 * for write access.
 */
struct proc;
struct thread;

/*
 * Kernel runnable context (thread).
 * This is what is put to sleep and reactivated.
 * Thread context.  Processes may have multiple threads.
 */
struct thread {
    struct proc     *td_proc;           /* (*) Associated process. */
    struct ucred    *td_ucred;          /* (k) Reference to credentials. */
    struct rusage    td_ru;             /* (t) rusage information. */
    struct seltd    *td_sel;            /* Select queue/channel. */
    char             td_name[MAXCOMLEN + 1]; /* (*) Thread name. */
    register_t       td_retval[2];      /* (k) Syscall aux returns. */
    volatile u_int   td_generation;     /* (k) For detection of preemption */
    struct file     *td_fpop;           /* (k) file referencing cdev under op */
    uint64_t         td_pid;            /* (k) process identifier */
    int              td_errno;          /* (k) thread's errno */
};

/*
 * Process structure.
 */
struct proc {
    struct filedesc    *p_fd;           /* (b) Open files. */
    struct plimit      *p_limit;        /* (c) Process limits. */
    pid_t               p_pid;          /* (b) Process identifier. */
    struct mtx          p_mtx;          /* (n) Lock for this struct. */
    int                 p_fibnum;       /* in this routing domain XXX MRT */
    struct malloc_type *p_malloc;       /* Allocator for this process */
    struct uma_client_zones p_zones;    /* Client's UMA zones */
};

#define PROC_LOCK(p)        mtx_lock(&(p)->p_mtx)
#define PROC_UNLOCK(p)      mtx_unlock(&(p)->p_mtx)
#define thread_lock(p)      (void)(0)
#define thread_unlock(p)    (void)(0)

extern struct proc*  proc0;          /* Startup process */
extern struct thread thread0;        /* Primary thread in proc0. */

int securelevel_ge(struct ucred *cr, int level);
int securelevel_gt(struct ucred *cr, int level);

#endif /* _KERNEL */
#endif /* !_SYS_PROC_H_ */
