﻿/*-
 * Copyright (c) 1982, 1986, 1989, 1993
 *    The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *    @(#)file.h    8.3 (Berkeley) 1/9/95
 * $FreeBSD$
 */

#ifndef _SYS_FILE_H_
#define    _SYS_FILE_H_

#ifndef _KERNEL
#include <sys/types.h> /* XXX */
#include <sys/fcntl.h>
#include <sys/unistd.h>
#else
#include <sys/queue.h>
#include <sys/refcount.h>
#include <sys/_lock.h>
#include <sys/_mutex.h>
#include <sys/socket.h>

struct stat;
struct thread;
struct uio;
struct knote;
struct vnode;
struct socket;

#endif /* _KERNEL */

#define DTYPE_VNODE     1    /* file */
#define DTYPE_SOCKET    2    /* communications endpoint */

#ifdef _KERNEL

struct file;
struct filecaps;
struct ucred;

typedef int fo_rdwr_t(struct file *fp, struct uio *uio,
            struct ucred *active_cred, int flags,
            struct thread *td);
typedef int fo_truncate_t(struct file *fp, off_t length,
            struct ucred *active_cred, struct thread *td);
typedef int fo_ioctl_t(struct file *fp, u_long com, void *data,
            struct ucred *active_cred, struct thread *td);
typedef int fo_poll_t(struct file *fp, int events,
            struct ucred *active_cred, struct thread *td);
typedef int fo_kqfilter_t(struct file *fp, struct knote *kn);
typedef int fo_stat_t(struct file *fp, struct stat *sb,
            struct ucred *active_cred, struct thread *td);
typedef int fo_close_t(struct file *fp, struct thread *td);
typedef int fo_chmod_t(struct file *fp, mode_t mode,
            struct ucred *active_cred, struct thread *td);
typedef int fo_chown_t(struct file *fp, uid_t uid, gid_t gid,
            struct ucred *active_cred, struct thread *td);
typedef int fo_sendfile_t(struct file *fp, int sockfd, struct uio *hdr_uio,
            struct uio *trl_uio, off_t offset, size_t nbytes,
            off_t *sent, int flags, int kflags, struct thread *td);
typedef int fo_seek_t(struct file *fp, off_t offset, int whence,
            struct thread *td);
typedef int fo_flags_t;

struct fileops {
    fo_rdwr_t       *fo_read;
    fo_rdwr_t       *fo_write;
    fo_truncate_t   *fo_truncate;
    fo_ioctl_t      *fo_ioctl;
    fo_poll_t       *fo_poll;
    fo_kqfilter_t   *fo_kqfilter;
    fo_stat_t       *fo_stat;
    fo_close_t      *fo_close;
    fo_chmod_t      *fo_chmod;
    fo_chown_t      *fo_chown;
    fo_sendfile_t   *fo_sendfile;
    fo_seek_t       *fo_seek;
    fo_flags_t       fo_flags;    /* DFLAG_* below */
};

#define DFLAG_PASSABLE    0x01    /* may be passed via unix sockets. */
#endif /* _KERNEL */

#if defined(_KERNEL) || defined(_WANT_FILE)
/*
 * Kernel descriptor table.
 * One entry for each open kernel vnode and socket.
 *
 * Below is the list of locks that protects members in struct file.
 *
 * (a) f_vnode lock required (shared allows both reads and writes)
 * (f) protected with mtx_lock(mtx_pool_find(fp))
 * (d) cdevpriv_mtx
 * none    not locked
 */

struct file {
    void                 *f_data;     /* file descriptor specific data */
    struct fileops       *f_ops;      /* File operations */
    struct ucred         *f_cred;     /* associated credentials. */
    short                 f_type;     /* descriptor type */
    volatile u_int        f_flag;     /* see fcntl.h */
    volatile u_int        f_count;    /* ref counter */
    void                 *f_cdevpriv; /* (d) Private data for the cdev. */
};

#endif /* _KERNEL || _WANT_FILE */

#ifdef _KERNEL

 /* per process limit on number of open sockets */
static const int maxfilesperproc = MAX_SOCKETS_PER_CLIENT;

extern struct fileops badfileops;
extern struct fileops socketops;
extern int maxfiles;                  /* kernel limit on number of open files */

/*
 * The socket operations are used a couple of places.
 * XXX: This is wrong, they should go through the operations vector for
 * XXX: sockets instead of going directly for the individual functions. /phk
 */
fo_rdwr_t       soo_read;
fo_rdwr_t       soo_write;
fo_truncate_t   soo_truncate;
fo_ioctl_t      soo_ioctl;
fo_poll_t       soo_poll;
fo_kqfilter_t   soo_kqfilter;
fo_stat_t       soo_stat;
fo_close_t      soo_close;

fo_chmod_t      invfo_chmod;
fo_chown_t      invfo_chown;
fo_sendfile_t   invfo_sendfile;

fo_sendfile_t   vn_sendfile;
fo_seek_t       vn_seek;

void finit(struct file *, u_int, short, void *, struct fileops *);
int  fdrop(struct file *fp, struct thread *td);
void fhold(struct file *fp);
#ifdef __SigloBSD__
int  fdup(struct proc *owner, int fd, int *fd_out);
#endif

static __inline fo_rdwr_t       fo_read;
static __inline fo_rdwr_t       fo_write;
static __inline fo_truncate_t   fo_truncate;
static __inline fo_ioctl_t      fo_ioctl;
static __inline fo_poll_t       fo_poll;
static __inline fo_kqfilter_t   fo_kqfilter;
static __inline fo_stat_t       fo_stat;
static __inline fo_close_t      fo_close;
static __inline fo_chmod_t      fo_chmod;
static __inline fo_chown_t      fo_chown;
static __inline fo_sendfile_t   fo_sendfile;

static __inline int
fo_read(struct file *fp, struct uio *uio, struct ucred *active_cred,
    int flags, struct thread *td)
{
    return ((*fp->f_ops->fo_read)(fp, uio, active_cred, flags, td));
}

static __inline int
fo_write(struct file *fp, struct uio *uio, struct ucred *active_cred,
    int flags, struct thread *td)
{
    return ((*fp->f_ops->fo_write)(fp, uio, active_cred, flags, td));
}

static __inline int
fo_truncate(struct file *fp, off_t length, struct ucred *active_cred,
    struct thread *td)
{
    return ((*fp->f_ops->fo_truncate)(fp, length, active_cred, td));
}

static __inline int
fo_ioctl(struct file *fp, u_long com, void *data, struct ucred *active_cred,
    struct thread *td)
{
    return ((*fp->f_ops->fo_ioctl)(fp, com, data, active_cred, td));
}

static __inline int
fo_poll(struct file *fp, int events, struct ucred *active_cred,
    struct thread *td)
{
    return ((*fp->f_ops->fo_poll)(fp, events, active_cred, td));
}

static __inline int
fo_stat(struct file *fp, struct stat *sb, struct ucred *active_cred,
    struct thread *td)
{
    return ((*fp->f_ops->fo_stat)(fp, sb, active_cred, td));
}

static __inline int
fo_close(struct file *fp, struct thread *td)
{
    return ((*fp->f_ops->fo_close)(fp, td));
}

static __inline int
fo_kqfilter(struct file *fp, struct knote *kn)
{
    return ((*fp->f_ops->fo_kqfilter)(fp, kn));
}

static __inline int
fo_chmod(struct file *fp, mode_t mode, struct ucred *active_cred,
    struct thread *td)
{
    return ((*fp->f_ops->fo_chmod)(fp, mode, active_cred, td));
}

static __inline int
fo_chown(struct file *fp, uid_t uid, gid_t gid, struct ucred *active_cred,
    struct thread *td)
{
    return ((*fp->f_ops->fo_chown)(fp, uid, gid, active_cred, td));
}

static __inline int
fo_sendfile(struct file *fp, int sockfd, struct uio *hdr_uio,
    struct uio *trl_uio, off_t offset, size_t nbytes, off_t *sent, int flags,
    int kflags, struct thread *td)
{
    return ((*fp->f_ops->fo_sendfile)(fp, sockfd, hdr_uio, trl_uio, offset,
        nbytes, sent, flags, kflags, td));
}

static __inline int
fo_seek(struct file *fp, off_t offset, int whence, struct thread *td)
{
    return ((*fp->f_ops->fo_seek)(fp, offset, whence, td));
}

#endif /* _KERNEL */

#endif /* !SYS_FILE_H */
