﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
*/
#include <pthread.h>
#include <unistd.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
  #include "ntd-tests/ntd-tests.h"
#ifdef __cplusplus
}
#endif /* __cplusplus */

/* For safe condition variable usage, must use a boolean predicate and  */
/* a mutex with the condition.                                          */
static int                 workToDo = 0;
#if SUPPORT_STATIC_INITALIZATION
static pthread_cond_t      cond  = PTHREAD_COND_INITIALIZER;
static pthread_mutex_t     mutex = PTHREAD_MUTEX_INITIALIZER;
#else
static pthread_cond_t      cond;
static pthread_mutex_t     mutex;
#endif

#define NTHREADS      2

typedef struct thread_info {
    pthread_t           threadid;
    char                name[16];
} thread_info;
static thread_info      threads[NTHREADS];

static void *threadfunc(void *parm)
{
    int           rc, i;
    pthread_t     me = pthread_self();
    char          *name = (char *)"Can not find thread";

    for (i = 0; i < NTHREADS; i++) {
        if (threads[i].threadid == me) {
            name = threads[i].name;
            break;
        }
    }

    while (1) {
        /* Usually worker threads will loop on these operations */
        rc = pthread_mutex_lock(&mutex);
        TESTCASE_MESSAGE(rc == 0, "pthread_mutex_lock() returned %d expected 0", rc);

        while (!workToDo) {
            TESTWARN("%s blocked\n", name);
            rc = pthread_cond_wait(&cond, &mutex);
            TESTCASE_MESSAGE(rc == 0, "pthread_cond_wait() returned %d expected 0", rc);
        }
        TESTWARN("%s awake, finish work!\n", name);

        /* Under protection of the lock, complete or remove the work     */
        /* from whatever worker queue we have. Here it is simply a flag  */
        workToDo = 0;

        rc = pthread_mutex_unlock(&mutex);
        TESTCASE_MESSAGE(rc == 0, "pthread_mutex_lock() returned %d expected 0", rc);
    }
    return NULL;
}

void test_pthread_cond_signal()
{
  int                   rc=0;
  int                   i;

  NTD_TEST_GROUP_START("cond_signal", 3);

#if !SUPPORT_STATIC_INITALIZATION
  rc = pthread_cond_init(&cond, NULL);
  TESTCASE_MESSAGE(rc == 0, "pthread_cond_init() returned %d expected 0", rc);

  rc = pthread_mutex_init(&mutex, NULL);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_init(NULL) returned %d expected 0", rc);
#endif

  TESTWARN("Create %d threads\n", NTHREADS);
  for(i=0; i<NTHREADS; ++i) {
    rc = pthread_create(&(threads[i].threadid), NULL, threadfunc, NULL);
    sprintf(threads[i].name, "thread %d", i);
    TESTCASE_MESSAGE(rc == 0, "pthread_create() returned %d expected 0", rc);
  }

  usleep(5000);  /* Sleep is not a very robust way to serialize threads   */

  for(i=0; i<5; ++i) {
    TESTWARN("Wake up a worker, work to do...\n");

    rc = pthread_mutex_lock(&mutex);
    TESTCASE_MESSAGE(rc == 0, "pthread_mutex_lock() returned %d expected 0", rc);

    /* In the real world, all the threads might be busy, and        */
    /* we would add work to a queue instead of simply using a flag  */
    /* In that case the boolean predicate might be some boolean     */
    /* statement like: if (the-queue-contains-work)                 */
    if (workToDo) {
       TESTWARN("Work already present, likely threads are busy\n");
    }
    workToDo = 1;
    rc = pthread_cond_signal(&cond);
    TESTCASE_MESSAGE(rc == 0, "pthread_cond_signal() returned %d expected 0", rc);

    rc = pthread_mutex_unlock(&mutex);
    TESTCASE_MESSAGE(rc == 0, "pthread_mutex_unlock() returned %d expected 0", rc);
    usleep(5000);   /* Sleep is not a very robust way to serialize threads */
  }

#if 0
  /* mea added call to pthread_join */
  TESTWARN("Wait for threads and cleanup\n");
  for (i=0; i<NTHREADS; ++i) {
    rc = pthread_join(threads[i].threadid, NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_join() returned %d expected 0", rc);
  }
#endif
  /* mea added call to pthread_cond_destroy */
  pthread_cond_destroy(&cond);
  pthread_mutex_destroy(&mutex);

  NTD_TEST_GROUP_END("cond_signal", 3);
}

#if 0
Debug output:

Enter Testcase - QP0WTEST/TPCOS0
Create 2 threads
Thread blocked
Thread blocked
Wake up a worker, work to do...
Thread awake, finish work!
Thread blocked
Wake up a worker, work to do...
Thread awake, finish work!
Thread blocked
Wake up a worker, work to do...
Thread awake, finish work!
Thread blocked
Wake up a worker, work to do...
Thread awake, finish work!
Thread blocked
Wake up a worker, work to do...
Thread awake, finish work!
Thread blocked
Main completed
#endif
