﻿/*---------------------------------------------------------------------------*
  Copyright (C)2014 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

/**
 * @examplesource{OsMemoryHeap.cpp,PageSampleOsMemoryHeap}
 *
 * @brief
 *  メモリヒープ機能のサンプルプログラム
 */

/**
 * @page PageSampleOsMemoryHeap MemoryHeap サンプルプログラム
 * @tableofcontents
 *
 * @brief
 *  メモリヒープ機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsMemoryHeap_SectionBrief 概要
 *  ここでは、メモリヒープ機能を使ったサンプルプログラムの説明をします。
 *
 *  メモリヒープ機能の使い方については、メモリヒープ機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsMemoryHeap_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsMemoryHeap Samples/Sources/Applications/OsMemoryHeap @endlink 以下にあります。
 *
 * @section PageSampleOsMemoryHeap_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsMemoryHeap_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsMemoryHeap_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsMemoryHeap_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsMemoryHeap_SectionDetail 解説
 *
 * @subsection PageSampleOsMemoryHeap_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsMemoryHeap.cpp
 *  @includelineno OsMemoryHeap.cpp
 *
 * @subsection PageSampleOsMemoryHeap_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて SetMemoryHeapSize() でメモリヒープを確保
 *  - nnMain() にて確保したメモリヒープのアドレスとサイズ情報をログ出力
 *  - MemorySample() にて AllocateMemoryBlock() を発行
 *  - メモリが獲得できなくなるまで上記の処理を繰り返す
 *  - メモリヒープ全体のメモリ領域を返却
 *  - nnMain() に戻って終了
 *
 *  まず、nnMain() 関数にてメモリヒープのサイズを設定し、メモリヒープを利用
 *  できる環境を整えます。@n通常、この処理はアプリケーション起動時に一度だけ
 *  実行することが望ましいです。
 *
 *  MemorySample() では、メモリが獲得できなくなるまで AllocateMemoryBlock() を
 *  繰り返します。 AllocateMemoryBlock() では指定したサイズを超えて
 *  メモリヒープを消費することはありません。そのため、フラグメント等が起きて
 *  いない状況ならば、メモリヒープを 1byte も無駄にすることなく
 *  AllocateMemoryBlock() で使い切ることが出来ます。
 *
 *  また、仮にフラグメントが起きた状態であっても、メモリブロックの最小単位
 *  である nn::os::MemoryBlockUnitSize で AllocateMemoryBlock() を行なうことで、
 *  やはりメモリヒープを無駄なく使い切ることが出来ます。
 *
 *  最後にメモリヒープを完全に獲得し切った後に、メモリヒープ全体の領域を
 *  FreeMemoryBlock() しています。通常、メモリの返却は獲得した領域ごとに
 *  返却しますが、今回のように獲得した領域をまとめて返却することも出来ます。
 *  ただし、返却する領域に未獲得領域が含まれていてはいけません。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *  なお、実行結果のうち、メモリのアドレスについては実行環境によって異なります。
 *
 *  @verbinclude  OsMemoryHeap_OutputExample.txt
 *
 *  ログ出力結果を見て分かる通り、最初に確保したメモリヒープのサイズと、
 *  最終的に AllocateMemoryBlock() で確保したメモリの合計値が一致しているのが
 *  分かります。
 *
 */

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

extern "C" {
  #include "ntd-tests/ntd-tests.h"
  #include "malloc_test_helper_funcs.h"
}


// Need to extern this since it's implemented in a '.c' file
//extern "C" void NTD_TESTCASE(bool passed, const char *file, int line, const char *expression);


namespace nn { namespace os {
    uintptr_t   GetMemoryHeapAddress();
    size_t      GetMemoryHeapSize();
}}

//-----------------------------------------------------------------------------


//
//  サンプルプログラムの本体
//
static void MemorySample()
{
    size_t     size   = nn::os::MemoryBlockUnitSize;
    char *ptr1;
    char *ptr2;
    char *ptr3;
    char *ptr4;
    char *ptr5;

    ptr1 = (char*)malloc( size - 0x5000 );
    //printf("malloc address 0x%08x; size 0x%08x\n", ptr1, size - 0x5000);
    TESTCASE( ptr1 != 0 );
    ptr2 = (char*)malloc( size );
    //printf("malloc address 0x%08x; size 0x%08x\n", ptr2, size);
    TESTCASE( ptr2 != 0 );

    TESTCASE( ptr2 - ptr1 == size );

    ptr3 = (char*)malloc( size * 2 );
    //printf("malloc address 0x%08x; size 0x%08x\n", ptr3, size * 2);
    TESTCASE( ptr3 != 0 );

    TESTCASE( ptr3 - ptr2 == (size*2) );

    size_t small_alloc_size = 101;
    ptr4 = (char*)malloc( small_alloc_size );
    //printf("malloc address 0x%08x; size 0x%08x\n", ptr4, small_alloc_size);
    TESTCASE( ptr4 != 0 );

    // Since 'ptr4' is a smaller allocation, it shouldn't be placed within the
    // contiguous heap
    TESTCASE( ptr4 - ptr3 != 100 );

    //printf("freeing address 0x%08x; size 0x%08x\n", ptr1, size - 0x5000);
    free (ptr1);
    //printf("free finish\n");
    ptr5 = (char*)malloc( size - 0x5000 );
    //printf("malloc address 0x%08x; size 0x%08x\n", ptr5, size - 0x5000);
    TESTCASE( ptr5 != 0 );

    // 'ptr5' should have the same address that 'ptr1' had
    TESTCASE( ptr2 - ptr5 == size );
}


//
//  メイン関数です。
//
///extern "C" void test_OSMemoryHeap(void);

extern "C" int test_OSMemoryHeap()
{
    // init a contiguous empty heap
    // the size of the heap must be a multiple of nn::os::MemoryBlockUnitSize (2 MB)
    // MemorySample() requires a heap of 8 MB
    size_t size = (nn::os::MemoryBlockUnitSize * 10);
    nn::Result result = nn::os::SetMemoryHeapSize(size);

    if (result.IsFailure()) {
        printf("SetMemoryHeapSize() failed\n");
    } else {
      MemorySample();
    }

  return NTD_TEST_RESULTS;
}
