﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <stdlib.h> /* malloc() */
#include <stdio.h> /* printf() */

#include <errno.h> /* errno */


#include "ntd-tests/ntd-tests.h"

#include "malloc_test_helper_funcs.h"

/*{{{{------------------------------------------------------------------------*/
/* Check that the actual sized of the pointer returned by malloc() is larger
   than the size passed in (should contain header information) */
void test_returned_size( void )
{
  // Choose a size that should get allocated without error. As long as the size
  // isn't greater than (PTRDIFF_MAX - SIZE_ALIGN - PAGE_SIZE + 1), it's fine
  int size = 1000;
  int chunk_size = -1;
  char* ptr = 0;
  struct chunk* chunk_ptr = 0;

  ptr = (char*)malloc(size);
  chunk_ptr = MEM_TO_CHUNK(ptr);
  chunk_size = CHUNK_SIZE(chunk_ptr);

  TESTCASE( chunk_size > size );

  free(ptr);
}
/*------------------------------------------------------------------------}}}}*/

/*{{{{------------------------------------------------------------------------*/
/* Check that adjacent freed chunks will be merged into one freed chunk */
void test_merge_free_chunks()
{
  char* pointer1;
  char* pointer2;
  char* pointer3; // To ensure pointer2 doesn't get merged with memory after it
  struct chunk* chunk_ptr1;
  struct chunk* chunk_ptr2;
  struct chunk* chunk_ptr3;
  struct chunk* next_chunk; // To check the chunk after the merged chunk
  int initial_num_freelist_elements;
  int returned_size1; // Adjusted size returned from malloc() for pointer1
  int returned_size2; // Adjusted size returned from malloc() for pointer2
  int returned_size3; // Adjusted size returned from malloc() for pointer3
  int alloc_size = 1000;

  pointer1 = (char*)malloc( alloc_size );
  pointer2 = (char*)malloc( alloc_size );
  pointer3 = (char*)malloc( alloc_size * 4 );
  if (ERROR_IF_NULL(pointer1, "pointer1")) return;
  if (ERROR_IF_NULL(pointer2, "pointer2")) return;
  if (ERROR_IF_NULL(pointer3, "pointer3")) return;

  chunk_ptr1 = MEM_TO_CHUNK(pointer1);
  chunk_ptr2 = MEM_TO_CHUNK(pointer2);
  chunk_ptr3 = MEM_TO_CHUNK(pointer3);
  if (ERROR_IF_NULL(chunk_ptr1, "chunk_ptr1")) return;
  if (ERROR_IF_NULL(chunk_ptr2, "chunk_ptr2")) return;
  if (ERROR_IF_NULL(chunk_ptr3, "chunk_ptr3")) return;

  returned_size1 = CHUNK_SIZE(chunk_ptr1);
  returned_size2 = CHUNK_SIZE(chunk_ptr2);
  returned_size3 = CHUNK_SIZE(chunk_ptr3);

  // Capture the current number of freed chunks before we free anything
  initial_num_freelist_elements = num_freelist_chunks();

  free(pointer2);

  // A pointer was freed, so there should be a new element in the freelist
  TESTCASE( num_freelist_chunks() == initial_num_freelist_elements + 1 );
  TESTCASE( is_ptr_in_freelist(chunk_ptr1) == false );
  TESTCASE( is_ptr_in_freelist(chunk_ptr2) == true );
  TESTCASE( is_ptr_in_freelist(chunk_ptr3) == false );

  free(pointer1);

  // The freed chunks should have been merged into one, so there should still
  // be only one additional element in the freelist rather than two
  TESTCASE( num_freelist_chunks() == initial_num_freelist_elements + 1 );

  // The sizes of the combined chunks should have been combined, too
  TESTCASE( chunk_ptr1->csize == returned_size1 + returned_size2 );

  // Verify that the chunk located directly after the newly merged chunk is
  // still 'pointer3'
  next_chunk = MEM_TO_CHUNK( pointer1 + CHUNK_SIZE(chunk_ptr1) );
  if (ERROR_IF_NULL(next_chunk, "next_chunk")) return;
  TESTCASE( CHUNK_TO_MEM(next_chunk) == pointer3 );
  TESTCASE( returned_size3 == CHUNK_SIZE(next_chunk) );
  TESTCASE( (next_chunk->csize & C_INUSE) == 1 );
  TESTCASE( is_ptr_in_freelist(chunk_ptr1) == true );
  TESTCASE( is_ptr_in_freelist(chunk_ptr2) == false );
  TESTCASE( is_ptr_in_freelist(chunk_ptr3) == false );

  free(pointer3);

  // We should be back to the original number of chunks again
  TESTCASE( num_freelist_chunks() == initial_num_freelist_elements );
  TESTCASE( is_ptr_in_freelist(chunk_ptr1) == true );
  TESTCASE( is_ptr_in_freelist(chunk_ptr2) == false );
  TESTCASE( is_ptr_in_freelist(chunk_ptr3) == false );
}
/*------------------------------------------------------------------------}}}}*/


/*{{{{------------------------------------------------------------------------*/
/* Checks that the first allocation gets assigned to where the heap pointer is
   pointing */
void test_initial_allocation_at_heap()
{
  char* pointer;
  struct chunk* chunk_ptr;
  struct chunk* heap_chunk_ptr;
  int alloc_size = 1500;

  heap_chunk_ptr = MEM_TO_CHUNK(mal.heap);
  if (ERROR_IF_NULL(heap_chunk_ptr, "heap_chunk_ptr")) return;

  if ((heap_chunk_ptr->csize & C_INUSE) == 0)
  {
    pointer = (char*)malloc( alloc_size );
    if (ERROR_IF_NULL(pointer, "pointer")) return;
    TESTCASE( MEM_TO_CHUNK(pointer) == heap_chunk_ptr );
    free(pointer);
    TESTCASE( is_ptr_in_freelist(heap_chunk_ptr) == true );
  }
}
/*------------------------------------------------------------------------}}}}*/

/*{{{{------------------------------------------------------------------------*/
/* Checks trying to allocate more than MMAP_THRESHOLD
   (should be about 114688 bytes)*/
void test_mmap_threshold()
{
  char* pointer;
  struct chunk* chunk_ptr;
  //struct chunk* chunk_ptr;
  int alloc_size = MMAP_THRESHOLD + 1;

  pointer = (char*)malloc( alloc_size );
  if (ERROR_IF_NULL(pointer, "pointer")) return;
  chunk_ptr = MEM_TO_CHUNK(pointer);
  // Larger allocations shouldn't have the INUSE bit set since they shouldn't
  // be added to the freelist when freed
  TESTCASE( (chunk_ptr->csize & C_INUSE) == 0 );

  free(pointer);

  // If the memory was allocated with __mmap() (because the requested size was
  // so big), then it shouldn't appear in the freelist when freed
  TESTCASE( is_ptr_in_freelist(pointer) == false );
}
/*------------------------------------------------------------------------}}}}*/


/*{{{{------------------------------------------------------------------------*/
/* Tests trying to allocate the maximum size possible
   (should fail with ENOMEM) */
void test_alloc_max_size()
{
  char* pointer;
  int saved_errno; // Save the value of errno right after an error should happen
  int alloc_size = PTRDIFF_MAX; // Should equal INT_MAX, but malloc.c checks
                                // specifically for PTRDIFF_MAX

  // The malloc() call should fail and return null and set errno to ENOMEM
  pointer = (char*)malloc( alloc_size );
  saved_errno = errno;
  TESTCASE( pointer == 0 );
  TESTCASE( errno == ENOMEM);

  // The free() function should succeed even if given a null pointer
  // (it'll just do nothing)
  free(pointer);
}
/*------------------------------------------------------------------------}}}}*/

/*{{{{------------------------------------------------------------------------*/
/* The standard specifies that freeing a null pointer should have no effect.
   So make sure that doing it won't cause a crash or change the pointer */
void test_free_null_ptr()
{
  char* pointer = 0;
  free(pointer);
  TESTCASE( pointer == 0 );
}
/*------------------------------------------------------------------------}}}}*/

/*{{{{------------------------------------------------------------------------*/
/* Test that allocations within a space of memory will be adjacent to each other
   and that we can walk the list of allocations based on their sizes */
void test_walk_addresses()
{
  const int num_allocs = 10;
  char* alloc_array[num_allocs];
  struct chunk* chunk_array[num_allocs];
  int i; // Loop counter
  int alloc_size = 1000;
  unsigned int index_to_free = 4; // Which index in the array to free during the test
  TESTCASE( index_to_free < num_allocs ); // In case one var is changed and not the other
  TESTCASE( index_to_free+1 < num_allocs); // So we can free an adjacent chunk

  // Allocate all the pointers
  for (i = 0; i < num_allocs; ++i)
  {
    alloc_array[i] = (char*)malloc( alloc_size );
    // If a pointer is returned as null, then clean up and exit the test,
    // because something went wrong
    if (ERROR_IF_NULL(alloc_array[i], "alloc_array[i]"))
    {
      printf("In test_walk_addresses(). 'alloc_array[%d]' is null\n", i);
      int j;
      for (j = 0; j < i; ++j)
        free(alloc_array[j]);
      return;
    }
    chunk_array[i] = MEM_TO_CHUNK(alloc_array[i]);
  }

  struct chunk* walk_chunk_ptr;
  char* walk_ptr = alloc_array[0];

  // We should be able to walk through all the allocated chunks by moving the
  // pointer by the size of each chunk
  for (i = 1; i < num_allocs; ++i)
  {
    walk_chunk_ptr = MEM_TO_CHUNK(walk_ptr);
    TESTCASE( (walk_chunk_ptr->csize & C_INUSE) == 1 );
    walk_ptr += CHUNK_SIZE(walk_chunk_ptr);
    TESTCASE( walk_ptr == alloc_array[i] );
    if (walk_ptr != alloc_array[i])
    {
      // Print a little more info to go along with the TESTCASE error message
      printf("FAIL: walk_ptr (%p)  !=  alloc_array[%d] (%p)\n",
             walk_ptr, i, alloc_array[i]);
    }
  }

  free(alloc_array[index_to_free]);

  // Now that one element is free, we should see that only one chunk in the
  // array has its INUSE bit deactivated
  walk_ptr = alloc_array[0];
  walk_chunk_ptr = MEM_TO_CHUNK(walk_ptr);
  for (i = 1; i < num_allocs; ++i)
  {
    walk_ptr += CHUNK_SIZE(walk_chunk_ptr);
    TESTCASE( walk_ptr == alloc_array[i] );
    if (walk_ptr != alloc_array[i])
    {
      // Print a little more info to go along with the TESTCASE error message
      printf("FAIL: walk_ptr (%p)  !=  alloc_array[%d] (%p)\n",
             walk_ptr, i, alloc_array[i]);
    }
    walk_chunk_ptr = MEM_TO_CHUNK(walk_ptr);

    if (i == index_to_free)
      TESTCASE( (walk_chunk_ptr->csize & C_INUSE) == 0 );
    else
      TESTCASE( (walk_chunk_ptr->csize & C_INUSE) == 1 );
  }

  walk_chunk_ptr = MEM_TO_CHUNK(alloc_array[index_to_free]);
  int combined_size = CHUNK_SIZE(walk_chunk_ptr);
  combined_size *= 2;
  free(alloc_array[index_to_free+1]);

  // The two adjacent freed chunks should have been combined, so when the walker
  // pointer steps over the first freed chunk, it should skip over the other
  // freed chunk since they were combined together into a single, large chunk
  walk_ptr = alloc_array[0];
  walk_chunk_ptr = MEM_TO_CHUNK(walk_ptr);
  for (i = 1; i < num_allocs; ++i)
  {
    walk_ptr += CHUNK_SIZE(walk_chunk_ptr);
    TESTCASE( walk_ptr == alloc_array[i] );
    if (walk_ptr != alloc_array[i])
    {
      // Print a little more info to go along with the TESTCASE error message
      printf("FAIL: walk_ptr (%p)  !=  alloc_array[%d] (%p)\n",
             walk_ptr, i, alloc_array[i]);
    }
    walk_chunk_ptr = MEM_TO_CHUNK(walk_ptr);

    if (i == index_to_free)
    {
      TESTCASE( (walk_chunk_ptr->csize & C_INUSE) == 0 );
      ++i; // Chunks got combined, so incrementing walk_ptr should skip an 'i'
    }
    else
      TESTCASE( (walk_chunk_ptr->csize & C_INUSE) == 1 );
  }

  // Free anything left unfreed
  for (i = 0; i < num_allocs; ++i)
  {
    if (i == index_to_free || i == index_to_free+1)
      continue;
    free(alloc_array[i]);
  }
}
/*------------------------------------------------------------------------}}}}*/

/*{{{{------------------------------------------------------------------------*/
/* Test trying to get malloc() to place a new allocation in a freed spot that is
   the same size as the allocation. Malloc() should place the new allocation at
   a different location, however, since the idea is that it might expand */
void test_fitted_chunk()
{
  // Allocate 3 pointers, free the middle one, then reallocate the middle one
  // with the same size and check that malloc() puts it at a different address
  const int num_allocs = 3; // This should always be '3'. Less code with loops
  char* allocs[num_allocs];
  struct chunk* chunks[num_allocs]; // Chunks corresponding to 'allocs'
  char* saved_ptr; // Store address of middle pointer
  int i; // Loop counter
  int returned_size_before; // Actual size of allocation returned by malloc()
  int returned_size_after;  // Actual size of allocation returned by malloc()
  int alloc_size = 2000;

  // Allocate the pointers
  for(i = 0; i < num_allocs; ++i)
  {
    allocs[i] = (char*)malloc(alloc_size);
    // Cleanup and exit if an allocation failed
    if (ERROR_IF_NULL(allocs[i], "allocs[i]"))
    {
      printf("ERROR: allocs[%d] is null\n", i);
      int j;
      for (j = 0; j < i; ++j)
        free(allocs[j]);
    }

    chunks[i] = MEM_TO_CHUNK(allocs[i]);
    TESTCASE( (chunks[i]->csize & C_INUSE) == 1 );
  }

  returned_size_before = CHUNK_SIZE(chunks[1]);

  // Free the middle pointer
  saved_ptr = allocs[1];
  free(allocs[1]);
  TESTCASE( (chunks[1]->csize & C_INUSE) == 0 );

  // Re-allocate the middle pointer
  allocs[1] = (char*)malloc(alloc_size);
  TESTCASE( allocs[1] != saved_ptr );
  chunks[1] = MEM_TO_CHUNK(allocs[1]);

  returned_size_after = CHUNK_SIZE(chunks[1]);
  TESTCASE( returned_size_before == returned_size_after );

  for (i = 0; i < num_allocs; ++i)
    free(allocs[i]);
}
/*------------------------------------------------------------------------}}}}*/

/*{{{{------------------------------------------------------------------------*/
/* Test calling realloc on a chunk that has no room to grow and also on a
   chunk that does have room */
void test_realloc_no_room()
{
  // Allocate 3 pointers, realloc the middle one which will be sandwiched
  // between the other pointers and initially not have room to expand
  const int num_allocs = 3; // This should always be '3'. Less code with loops
  char* allocs[num_allocs];
  struct chunk* chunks[num_allocs]; // Chunks corresponding to 'allocs'
  char* saved_ptr; // Store address of middle pointer before realloc()
  char* saved_ptr2; // Store address of the third pointer before realloc()
  char* realloc_ptr; // Return value of realloc()
  struct chunk* realloc_chunk_ptr;
  char* realloc_ptr2;
  struct chunk* realloc_chunk_ptr2;
  char* next_realloc_ptr; // The 'next' chunk after the 2nd realloc'd pointer
  struct chunk* next_realloc_chunk_ptr;
  int i; // Loop counter
  int returned_size; // Actual size of allocation returned by malloc()
  const int alloc_size = 2000;
  const int realloc_size  = alloc_size + 1000; // Larger realloc size
  const int realloc_size2 = alloc_size - 1000; // Smaller realloc size

  // Allocate the pointers
  for(i = 0; i < num_allocs; ++i)
  {
    allocs[i] = (char*)malloc(alloc_size);
    // Cleanup and exit if an allocation failed
    if (ERROR_IF_NULL(allocs[i], "allocs[i]"))
    {
      printf("ERROR: allocs[%d] is null\n", i);
      int j;
      for (j = 0; j < i; ++j)
        free(allocs[j]);
    }

    chunks[i] = MEM_TO_CHUNK(allocs[i]);
    TESTCASE( (chunks[i]->csize & C_INUSE) == 1 );
  }

  saved_ptr = allocs[1];
  realloc_ptr = (char*)realloc(allocs[1], realloc_size);
  // If something goes wrong with the realloc, then cleanup and leave
  if (ERROR_IF_NULL(realloc_ptr, "realloc_ptr"))
  {
    free(allocs[0]);
    // The realloc() may have freed the old memory block, so double-check first
    // to avoid potential double-free
    if ( (chunks[1]->csize & C_INUSE) == 1 )
      free(allocs[1]);
    free(allocs[2]);
    return;
  }
  realloc_chunk_ptr = MEM_TO_CHUNK(realloc_ptr);
  TESTCASE( (realloc_chunk_ptr->csize & C_INUSE) == 1 );
  TESTCASE( realloc_ptr != saved_ptr );
  // Realloc is supposed to free() the old block of memory if it has to move
  // the realloc'd memory to a new location
  TESTCASE( (chunks[1]->csize & C_INUSE) == 0 );
  // The new realloc'd location should have been placed after the 3rd allocation
  TESTCASE( (allocs[2] + CHUNK_SIZE(chunks[2])) == realloc_ptr );

  saved_ptr2 = allocs[2];
  realloc_ptr2 = (char*)realloc(allocs[2], realloc_size2);
  // If something goes wrong with the realloc, then cleanup and leave
  if (ERROR_IF_NULL(realloc_ptr2, "realloc_ptr2"))
  {
    // Shouldn't need to free allocs[1] if the previous realloc() worked right
    free(allocs[0]);
    free(realloc_ptr);
    if ( (chunks[2]->csize & C_INUSE) == 1 )
      free(allocs[2]);
    return;
  }
  realloc_chunk_ptr2 = MEM_TO_CHUNK(realloc_ptr2);
  TESTCASE( (realloc_chunk_ptr2->csize & C_INUSE) == 1 );
  // It was a smaller realloc(), so the returned address should be the same
  TESTCASE( realloc_ptr2 == saved_ptr2 );
  // The original address should still be marked as IN-USE
  TESTCASE( (chunks[2]->csize & C_INUSE) == 1 );
  // The size of the 2nd realloc'd chunk should be smaller than it used to be
  TESTCASE( chunks[2]->csize < alloc_size );
  next_realloc_ptr = allocs[2] + CHUNK_SIZE(chunks[2]);
  next_realloc_chunk_ptr = MEM_TO_CHUNK(next_realloc_ptr);
  // The original 'allocs[2]' chunk should have been split in half, which means
  // they should be adjacent to each other
  TESTCASE( next_realloc_chunk_ptr->psize == chunks[2]->csize );
  // The second half of 'allocs[2]' should have been freed
  TESTCASE( (next_realloc_chunk_ptr->csize & C_INUSE) == 0 );

  // Need to make sure to free the pointer returned by realloc() since it should
  // be a new address. Also, realloc() should have freed the old block during
  // the reallocation, so we shouldn't have to free allocs[1]
  free(allocs[0]);
  free(allocs[2]);
  free(realloc_ptr);
}
/*------------------------------------------------------------------------}}}}*/

/*{{{{------------------------------------------------------------------------*/
/* Test that 'mal.binmap' gets updated after an allocation */
void test_binmap_updated()
{
  uint64_t saved_binmap; // Save value of binmap for comparison
  char* ptr;
  int alloc_size = 1000;

  saved_binmap = mal.binmap;
  ptr = (char*)malloc(alloc_size);
  if (ERROR_IF_NULL(ptr, "ptr")) return;
  TESTCASE( mal.binmap != saved_binmap );

  saved_binmap = mal.binmap;
  free(ptr);
  TESTCASE( mal.binmap != saved_binmap );
}
/*------------------------------------------------------------------------}}}}*/

int test_MiscTests( void )
{
    int freelist_chunks_before_tests = num_freelist_chunks();
    int freelist_chunks_after_tests;

    test_returned_size();
    test_merge_free_chunks();
    test_initial_allocation_at_heap();
    test_mmap_threshold();
    test_alloc_max_size();
    test_free_null_ptr();
    test_walk_addresses();
    test_fitted_chunk();
    test_realloc_no_room();
    //test_binmap_updated(); // Might not be a guarantee, after all, that
                             // 'binmap' gets updated after a memory operation.
                             // Commenting test out for now

    // Check for any unfreed chunks
    freelist_chunks_after_tests = num_freelist_chunks();
    // If these were the first tests ran in the testsuite, then the number of
    // freed chunks will have increased rather than being the same as it was at
    // the start because the freelist will have been null and empty at the start
    if (freelist_chunks_before_tests == 0)
      TESTCASE( freelist_chunks_before_tests+1 == freelist_chunks_after_tests );
    else
      TESTCASE( freelist_chunks_before_tests == freelist_chunks_after_tests );

    return NTD_TEST_RESULTS;
}
