﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <stdlib.h> /* malloc() */
#include <stdio.h> /* printf() */
#include <stdint.h> /* uint64_t */


#include "ntd-tests/ntd-tests.h"
#include "malloc_test_helper_funcs.h"
mal_info_struct_type mal; //<-- Defined in stdlib.h, used in malloc.c

void print_malloc_info()
{
    printf("vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv print_malloc_info() mal.heap = %p\n", mal.heap);
    printf("Printing bin array:\n");
    printf("mal.heap: %%p = %p, 0x%%08x = 0x%08x\n", mal.heap, mal.heap);
    printf("mal.brk  = %p\n", mal.brk);
    printf("mal.binmap = %llu\n", mal.binmap);
    printf("mal.brk_lock[0] = %d\n", mal.brk_lock[0]);
    printf("mal.brk_lock[1] = %d\n", mal.brk_lock[1]);
    printf("mal.free_lock[0] = %d\n", mal.free_lock[0]);
    printf("mal.free_lock[1] = %d\n", mal.free_lock[1]);
    printf("mal.mmap_step = %u\n", mal.mmap_step);
    int i;
    for (i=0; i<64; i++)
    {
        struct chunk* ptr = mal.bins[i].head;
        if (ptr != 0)
        {
          printf("/\\/\\bin %d: ptr = %p\n", i, ptr);

          printf("  ptr->csize = %d\n", ptr->csize);
          printf("  ptr->psize = %d\n", ptr->psize);
          printf("  ptr->next  = %p\n", ptr->next);
          printf("  ptr->prev  = %p\n", ptr->prev);

          if (ptr->next != 0)
          {
            struct chunk* next_ptr = ptr->next;
            printf("    next_ptr->csize = %d\n", next_ptr->csize);
            printf("    next_ptr->psize = %d\n", next_ptr->psize);
            printf("    next_ptr->next  = %p\n", next_ptr->next);
            printf("    next_ptr->prev  = %p\n", next_ptr->prev);

            if (next_ptr->next != 0)
            {
              struct chunk* next_next_ptr = next_ptr->next;
              printf("      next_next_ptr->csize = %d\n", next_next_ptr->csize);
              printf("      next_next_ptr->psize = %d\n", next_next_ptr->psize);
              printf("      next_next_ptr->next  = %p\n", next_next_ptr->next);
              printf("      next_next_ptr->prev  = %p\n", next_next_ptr->prev);
            }
          }
        }
    }
    printf("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ print_malloc_info() mal.heap = %p\n", mal.heap);
}

// Recursively print the binary version of an integer number
void print_binary(int number)
{
  if (number > 0)
  {
    print_binary(number >> 1);
    putc((number & 1) ? '1' : '0', stdout);
  }
}
// Recursively print the binary version of an integer number
void print_binary_uint64(uint64_t number)
{
  if (number > 0)
  {
    print_binary_uint64(number >> 1);
    putc((number & 1) ? '1' : '0', stdout);
  }
}

void print_ptr_info(char* allocated_mem_ptr)
{
  printf(" vvv Printing data for pointer '%p' vvv\n", allocated_mem_ptr);
  if (allocated_mem_ptr == 0)
  {
    printf("ptr '%p' is null\n", allocated_mem_ptr);
  }
  else
  {
    struct chunk* chunk_ptr = MEM_TO_CHUNK(allocated_mem_ptr);
    printf("ptr       = %p\n", allocated_mem_ptr);
    print_chunk_info(chunk_ptr, false);

    // 'next' and 'prev' pointers are technically undefined when the chunk of
    // memory is in use. Only when the chunk is freed do the pointers work
    if ( chunk_ptr->next != chunk_ptr && (chunk_ptr->csize & C_INUSE == 0) )
    {
      printf(" --- chunk_ptr->next (%p):\n", chunk_ptr->next);
      print_chunk_info(chunk_ptr->next, false);
    }

    if (CHUNK_SIZE(chunk_ptr) > 0)
    {
      struct chunk* chunk_ptr2 = MEM_TO_CHUNK( allocated_mem_ptr + CHUNK_SIZE(chunk_ptr) );
      printf(" --- ptr (%p) + CHUNK_SIZE(chunk_ptr) (%d)  :  (%p):\n",
             allocated_mem_ptr, CHUNK_SIZE(chunk_ptr), allocated_mem_ptr + CHUNK_SIZE(chunk_ptr));

      // If the chunk is freed but its 'next' pointer is null, then something
      // weird is up with the chunk, so we shouldn't try to access too much
      // info when printing the 'next' chunk when obtained in this way rather
      // than obtaining it by doing '->next'
      if (chunk_ptr->next == 0 && ((chunk_ptr->csize & C_INUSE) == 0))
        print_chunk_info(chunk_ptr2, true);
      else
        print_chunk_info(chunk_ptr2, false);
    }

    if (CHUNK_PSIZE(chunk_ptr) == 1)
      printf(" --- CHUNK_PSIZE(chunk_ptr) = 1\n");
    if (CHUNK_PSIZE(chunk_ptr) > 1)
    {
      struct chunk* chunk_ptr2 = MEM_TO_CHUNK( allocated_mem_ptr - CHUNK_PSIZE(chunk_ptr) );
      printf(" --- ptr (%p) - CHUNK_PSIZE(chunk_ptr) (%d)  :  (%p):\n",
             allocated_mem_ptr, CHUNK_PSIZE(chunk_ptr), allocated_mem_ptr - CHUNK_PSIZE(chunk_ptr));

      // If the chunk is freed but its 'prev' pointer is null, then something
      // weird is up with the chunk, so we shouldn't try to access too much
      // info when printing the 'prev' chunk when obtained in this way rather
      // than obtaining it by doing '->prev'
      if (chunk_ptr->prev == 0 && ((chunk_ptr->csize & C_INUSE) == 0))
        print_chunk_info(chunk_ptr2, true);
      else
        print_chunk_info(chunk_ptr2, false);
    }

  }
  printf(" ^^^ Done printing data for pointer '%p' ^^^\n", allocated_mem_ptr);
}

void print_chunk_info(void* void_chunk_ptr, bool print_minimum)
{
  if (!void_chunk_ptr)
  {
    printf("void_chunk_ptr (%p) is null\n", void_chunk_ptr);
    return;
  }
  struct chunk* chunk_ptr = (struct chunk*)void_chunk_ptr;
  if (!chunk_ptr)
  {
    printf("chunk_ptr (%p) is null\n", chunk_ptr);
    return;
  }
  printf("chunk_ptr = %p\n", chunk_ptr);

  // Print only a safe amount of data about the pointer if there's a suspicion
  // that it might be an invalid address
  if (print_minimum == true)
    return;


  printf("chunk_ptr->csize (binary) = '");  print_binary(chunk_ptr->csize);  printf("'\n");
  printf("C_INUSE = '%d'\n", C_INUSE);
  printf("chunk_ptr->csize & C_INUSE = '%d'\n", chunk_ptr->csize & (C_INUSE));
  printf("IS_MMAPPED = '%d'\n", IS_MMAPPED(chunk_ptr));
  printf("CHUNK_SIZE = '%d'\n", CHUNK_SIZE(chunk_ptr));
  printf("chunk_ptr->csize = %d\n", chunk_ptr->csize);
  printf("chunk_ptr->psize = %d,  CHUNK_PSIZE(chunk_ptr) = %d\n", chunk_ptr->psize, CHUNK_PSIZE(chunk_ptr));
  printf("chunk_ptr->psize & 1 = %d\n", chunk_ptr->psize & 1);
  printf("chunk_ptr->next  = %p\n", chunk_ptr->next);
  printf("chunk_ptr->prev  = %p\n", chunk_ptr->prev);
}


/*{{{{------------------------------------------------------------------------*/
/* Call this to easily check for and display error about a null pointer */
bool null_ptr_error(void* ptr, const char* ptr_var_name,
                    const char* filename, int line_num)
{
  if (ptr == 0)
  {
    printf("Pointer '%s' is null\n", ptr_var_name);
    printf(" (in file '%s', line %d)\n", filename, line_num);
    return true;
  }

  return false;
}
/*------------------------------------------------------------------------}}}}*/



/*{{{{------------------------------------------------------------------------*/
/* Returns the number of freed chunks within the free list */
int num_freelist_chunks()
{
  struct chunk* chunk_ptr;
  int i;
  int num_freed_chunks = 0; // The return value
  int mal_bins_size = sizeof( mal.bins ) / sizeof( mal.bins[0] );

  for (i = 0; i < mal_bins_size; ++i)
  {
    chunk_ptr = mal.bins[i].head;
    if (!chunk_ptr)
      continue;

    // Check that the list contains more than just one element
    if (chunk_ptr->next != chunk_ptr)
    {
      // Skip to the pointer within the doubly linked list of 'bins[i]' that's
      // within the address space of the 'bins' array
      // (i.e., a pointer between 'bins[0]' and 'bins[<size of bins array>]')
      // since it's not a guarantee that 'bins[i]' will be pointing to that
      // one. We're treating it as the head of the doubly-linked list, so we'd
      // like to start at the head each time to be consistent
      while(chunk_ptr->csize != 0) chunk_ptr = chunk_ptr->next;

      // Now we can walk through each node in this linked list
      do
      {
        if (chunk_ptr->csize != 0)
          ++num_freed_chunks;
        chunk_ptr = chunk_ptr->next;
      } while(chunk_ptr->csize != 0);
    }
  }

  return num_freed_chunks;
}

// Returns false if the passed-in pointer is *not* in the freelist
// Returns true  if the passed-in pointer *is* in the freelist
bool is_ptr_in_freelist(struct chunk* chunk_to_find)
{
  struct chunk* chunk_ptr;
  int i;
  int num_freed_chunks = 0; // The return value
  int mal_bins_size = sizeof( mal.bins ) / sizeof( mal.bins[0] );

  for (i = 0; i < mal_bins_size; ++i)
  {
    chunk_ptr = mal.bins[i].head;
    if (!chunk_ptr)
      continue;

    // Check that the list contains more than just one element
    if (chunk_ptr->next != chunk_ptr)
    {
      // Skip to the pointer within the doubly linked list of 'bins[i]' that's
      // within the address space of the 'bins' array
      // (i.e., a pointer between 'bins[0]' and 'bins[<size of bins array>]')
      // since it's not a guarantee that 'bins[i]' will be pointing to that
      // one. We're treating it as the head of the doubly-linked list, so we'd
      // like to start at the head each time to be consistent
      while(chunk_ptr->csize != 0) chunk_ptr = chunk_ptr->next;

      // Now we can walk through each node in this linked list
      do
      {
        if (chunk_ptr == chunk_to_find)
          return true;
        chunk_ptr = chunk_ptr->next;
      } while(chunk_ptr->csize != 0);
    }
  }

  return false;
}

void walk_free_list()
{
  struct chunk* chunk_ptr;
  int i;
  int mal_bins_size = sizeof( mal.bins ) / sizeof( mal.bins[0] );

  for (i = 0; i < mal_bins_size; ++i)
  {
    chunk_ptr = mal.bins[i].head;
    if (!chunk_ptr)
      continue;

    // Check that the list contains more than just one element
    if (chunk_ptr->next != chunk_ptr)
    {
      // Skip to the pointer within the doubly linked list of 'bins[i]' that's
      // within the address space of the 'bins' array
      // (i.e., a pointer between 'bins[0]' and 'bins[<size of bins array>]')
      // since it's not a guarantee that 'bins[i]' will be pointing to that
      // one. We're treating it as the head of the doubly-linked list, so we'd
      // like to start at the head each time to be consistent
      while(chunk_ptr->csize != 0) chunk_ptr = chunk_ptr->next;

      // Now we can walk through each node in this linked list
      do
      {
        printf("%p -> ", chunk_ptr);
        chunk_ptr = chunk_ptr->next;
      } while(chunk_ptr->csize != 0);
    }
  }

  printf("done\n");
}
/*------------------------------------------------------------------------}}}}*/

