﻿/*---------------------------------------------------------------------------*
  Copyright (C) 2016 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "jira-regression-tests.h"
#include "ntd-test-libc.h"
#include <stdlib.h>
#include <pthread.h>
#include <unistd.h>

struct Node
{
    void* p;
    struct Node* next;
};

// child needs to wait until parent reads info before existing
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

void* test_pthread_create_thread(void* arg)
{
    int ret;
    ret = pthread_mutex_lock(&mutex);
    TESTCASE_MESSAGE(ret == 0, "pthread_mutex_lock(%p) failed result = %d\n", &mutex, ret);
    ret = pthread_mutex_unlock(&mutex);
    TESTCASE_MESSAGE(ret == 0, "pthread_mutex_unlock(%p) failed result = %d\n", &mutex, ret);
    return 0;
}

void cleanup_list(struct Node* head)
{
    while (head)
    {
        free(head->p);
        struct Node* next = head->next;
        free(head);
        head = next;
    }
}

void rynda_579(int argc, char** argv)
{
    NTD_TEST_GROUP_START("RYNDA-579", 1);

    int ret; // Store return values
    // only run the memory consumption test on horizon
#ifdef __NX__
    struct Node* head = (struct Node*)malloc(sizeof(struct Node));
    TESTCASE_MESSAGE(head, "Couldn't allocate memory for linked list\n");
    head->next = NULL;
    struct Node* walker = head;

    void* bytes520 = malloc(520);
    TESTCASE_MESSAGE(bytes520, "Couldn't allocate memory for bytes520\n");
    void* bytes1072 = malloc(1072);
    TESTCASE_MESSAGE(bytes1072, "Couldn't allocate memory for bytes1072\n");
    void* bytes90112 = malloc(90112);
    TESTCASE_MESSAGE(bytes90112, "Couldn't allocate memory for bytes90112\n");

    // Allocate all possible memory from malloc
    int stop = 16 * 1024;
    for (int i = 1024 * 1024 * 1024; i > stop; i /= 2)
    {
        void* p = NULL;
        do
        {
            p = malloc(i);
            walker->p = p;
            walker->next = (struct Node*)malloc(sizeof(struct Node));
            walker = walker->next;
            if (walker == NULL)
            {
                i = stop; // Break out of for-loop, too
                break;
            }
            walker->next = NULL;
        } while(p);
    }

    for (int i = 16 * 1024; i > 0 && walker; i -= 16)
    {
        void* p = NULL;
        do
        {
            p = malloc(i);
            walker->p = p;
            walker->next = (struct Node*)malloc(sizeof(struct Node));
            walker = walker->next;
            if (walker == NULL)
            {
                // Break out of for-loop, too, because walker is NULL
                break;
            }
            walker->next = NULL;
        } while(p);
    }


    free(bytes90112);
    free(bytes1072);
    free(bytes520);
#endif

    pthread_t thread;
    int arg;
    ret = pthread_mutex_lock(&mutex);
    TESTCASE_MESSAGE(ret == 0, "pthread_mutex_lock(%p) failed result = %d\n", &mutex, ret);
    ret = pthread_create(&thread, NULL, test_pthread_create_thread, &arg);
    TESTCASE_MESSAGE(ret == 0, "pthread_getschedparam() failed\n");

    // pthread_getschedparam() used to rely on the extra 520 bytes mentioned in
    // RYNDA-579. It shouldn't need that memory any more, so make sure it works
    int policy = -1;
    struct sched_param param;
    param.sched_priority = -1;
    ret = pthread_getschedparam(thread, &policy, &param);
    TESTCASE_MESSAGE(ret == 0, "pthread_getschedparam() failed\n");
    TESTCASE_MESSAGE(policy != -1, "Expected 'policy' to be a valid value. Got %d\n", policy);
    TESTCASE_MESSAGE(param.sched_priority != -1, "Expected 'param.sched_priority' to be a valid value. Got %d\n", param.sched_priority);
    ret = pthread_mutex_unlock(&mutex);
    TESTCASE_MESSAGE(ret == 0, "pthread_mutex_unlock(%p) failed result = %d\n", &mutex, ret);

    ret = pthread_join(thread, NULL);
    TESTCASE_MESSAGE(ret == 0, "pthread_join() failed\n");

#if __NX__
    printf("Finished pthread_join()\n");
    bytes90112 = malloc(90112);
    TESTCASE_MESSAGE(bytes90112, "Couldn't allocate memory for bytes90112\n");
    bytes1072 = malloc(1072);
    TESTCASE_MESSAGE(bytes1072, "Couldn't allocate memory for bytes1072\n");
    bytes520 = malloc(520);

    printf("Finished allocating bytes again\n");
    printf("  bytes520 = %p\n", bytes520);
    printf("  bytes1072 = %p\n", bytes1072);
    printf("  bytes90112 = %p\n", bytes90112);

    TESTCASE_MESSAGE(bytes520, "Expected 'bytes520' to be a valid pointer. Got %p\n", bytes520);

    free(bytes90112);
    free(bytes1072);
    free(bytes520);
    cleanup_list(head);
#endif

    NTD_TEST_GROUP_END("RYNDA-579", 1);
}
