﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#ifdef _WIN32
#include <cstdlib>
#else
#include <nn/os/os_SdkProcessControl.h>
#endif

#include <nnt/gtest/detail/gtest-climits.h>
#include <nnt/gtest/detail/gtest-cstdlib.h>
#include <nnt/gtest/detail/gtest-cstring.h>

namespace nnt { namespace testing { namespace detail {

namespace
{

bool IsHexDigit(char c)
{
    return ('0' <= c && c <= '9') ||
           ('A' <= c && c <= 'F') ||
           ('a' <= c && c <= 'f');
}

} // namespace

const int CStdlib::ExitFailure = 1;

int32_t CStdlib::Strtol(const char* str, char** endptr, int base) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(str);
    NN_SDK_REQUIRES(base == 0 || (2 <= base && base <= 36));

    const char* p = str;

    bool isNegative = false;
    if (*p == '+' || *p == '-')
    {
        if (*p == '-') { isNegative = true; }
        ++p;
    }

    if (base == 0 || base == 16)
    {
        if (*p == '0')
        {
            ++p;
            if (*p == 'X' || *p == 'x')
            {
                ++p;
                if (IsHexDigit(*p))
                {
                    if (base == 0) { base = 16; }
                }
                else
                {
                    if (endptr) { *endptr = const_cast<char*>(str); }
                    return 0;
                }
            }
            else
            {
                if (base == 0) { base = 8; }
            }
        }
        else
        {
            if (base == 0) { base = 10; }
        }
    }

    int32_t result = 0;

    while (*p)
    {
        int num = 0;
        if ('0' <= *p && *p <= '9')
        {
            num = *p - '0';
        }
        else if ('A' <= *p && *p <= 'Z')
        {
            num = *p - 'A' + 10;
        }
        else if ('a' <= *p && *p <= 'z')
        {
            num = *p - 'a' + 10;
        }
        else
        {
            break;
        }

        if (num >= base) { break; }

        ++p;

        if (!isNegative)
        {
            if (result > LongMax / base || num > LongMax - result * base)
            {
                result = LongMax;
            }
            else
            {
                result = result * base + num;
            }
        }
        else
        {
            if (result < LongMin / base || -num < LongMin - result * base)
            {
                result = LongMin;
            }
            else
            {
                result = result * base - num;
            }
        }
    }

    if (endptr) { *endptr = const_cast<char*>(p); }

    return result;
}

void CStdlib::Exit(int status) NN_NOEXCEPT
{
#ifdef _WIN32
    ::std::exit(status);
#else
    NN_ABORT_UNLESS_EQUAL(status, 0);
    nnosExitProcess();
#endif
}

char* CStdlib::Getenv(const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    if (CString::Strcmp(name, "TERM") == 0)
    {
        return const_cast<char*>("xterm");
    }

    return nullptr;
}

}}} // namespace nnt::testing::detail
