﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nnt { namespace testing { namespace detail {

class Algorithm final
{
public:
    template <typename T>
    static const T& Min(const T& lhs, const T& rhs) NN_NOEXCEPT
    {
        return lhs < rhs ? lhs : rhs;
    }

    template<typename Iterator, typename T>
    static Iterator Find(
        Iterator begin, Iterator end, const T& needle) NN_NOEXCEPT
    {
        for (Iterator it = begin; it != end; ++it)
        {
            if (*it == needle)
            {
                return it;
            }
        }
        return end;
    }

    template<typename Iterator, typename Functor>
    static Functor For_each(
        Iterator begin, Iterator end, Functor functor) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(functor);
        for (Iterator it = begin; it != end; ++it)
        {
            functor(*it);
        }
        return functor;
    }

    template<typename Iterator>
    static void Reverse(Iterator begin, Iterator end) NN_NOEXCEPT
    {
        while ((begin != end) && (begin != --end))
        {
            auto value = *begin;
            *begin = *end;
            *end = value;
            ++begin;
        }
    }

    template<typename Iterator, typename Compare>
    static void Sort(Iterator begin, Iterator end, Compare compare) NN_NOEXCEPT
    {
        const ptrdiff_t count = end - begin;

        NN_SDK_ASSERT_LESS_EQUAL(0, count);

        if (count <= 1)
        {
            return;
        }
        else
        {
            Iterator head = begin;
            Iterator body = head + (count / 2);
            Iterator tail = head + (count - 1);

            const typename Iterator::value_type pivot =
                compare(*head, *body)
                    ? (compare(*body, *tail)
                        ? *body : (compare(*tail, *head) ? *head : *tail))
                    : (compare(*tail, *body)
                        ? *body : (compare(*head, *tail) ? *head : *tail));

            int l = 0;
            int r = count - 1;

            while (NN_STATIC_CONDITION(true))
            {
                while (compare(*(head + static_cast<size_t>(l)), pivot))
                {
                    ++l;
                }

                while (compare(pivot, *(head + static_cast<size_t>(r))))
                {
                    --r;
                }

                if (r <= l)
                {
                    break;
                }

                typename Iterator::value_type&
                    lhs = *(head + static_cast<size_t>(l));

                typename Iterator::value_type&
                    rhs = *(head + static_cast<size_t>(r));

                typename Iterator::value_type tmp = lhs;
                lhs = rhs;
                rhs = tmp;

                ++l;
                --r;
            }

            if (1 < l)
            {
                Sort(head, head + static_cast<size_t>(l), compare);
            }

            if (r + 1 < count)
            {
                Sort(head + static_cast<size_t>(r + 1), tail + 1, compare);
            }
        }
    }

private:
    Algorithm();

    NN_DISALLOW_COPY(Algorithm);
    NN_DISALLOW_MOVE(Algorithm);
};

}}} // namespace nnt::testing::detail
