﻿// ========================================================================
// PSC_SleepWake.cpp : Defines the entry point for the console application.
//
// Uses Power State Controller (PSC) to put the system to sleep and then wake it back up.
//
// ========================================================================

// The majority of this code was taken from here:
// - C:\DejaTools\3rdParty\nintendo\Siglo\sdk\Tests\PowerState\Sources\Tests\SleepWakeSequenceTestLow\SleepWakeSequenceTestLow.cpp.
// If this project and/or SleepWakeApp.cpp file fails to compile, try opening the file above.
// If that fails to provide an answer, try opening the solution:
// - C:\DejaTools\3rdParty\nintendo\Siglo\sdk\Tests\PowerState\PowerState-spec.NX.vs2013.sln
//
// Note tha the PSC_SleepWake.desc and PSC_SleepWake.meta are custom, to enable
// permissions for using psc.

#include <nn/dd.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/psc.h>
#include <nn/psc/psc_PmControl.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <nn/svc/svc_Server.h>


#define NNT_POWERSTATE_TEST_ENTER_SC7
#define NNT_POWERSTATE_TEST_SWTICH_LOG_TO_UART

#ifdef NNT_POWERSTATE_TEST_SWTICH_LOG_TO_UART
extern "C" void nndiagStartup()
{
}
#endif

namespace
{

#ifdef NNT_POWERSTATE_TEST_ENTER_SC7
uintptr_t   ClkRstRegisterVirtualAddress;
uintptr_t   I2C5RegisterVirtualAddress;

#define CLK_RST_RST_DEV_H_CLR    (*reinterpret_cast<volatile uint32_t*>(ClkRstRegisterVirtualAddress + 0x8))
#define CLK_RST_CLK_OUT_ENB_H    (*reinterpret_cast<volatile uint32_t*>(ClkRstRegisterVirtualAddress + 0x14))

#define CLK_RST_RST_DEV_U_CLR    (*reinterpret_cast<volatile uint32_t*>(ClkRstRegisterVirtualAddress + 0xc))
#define CLK_RST_CLK_OUT_ENB_U    (*reinterpret_cast<volatile uint32_t*>(ClkRstRegisterVirtualAddress + 0x18))

#define I2C5_CNFG        (*reinterpret_cast<volatile uint32_t*>(I2C5RegisterVirtualAddress + 0x00))
#define I2C5_CMD_ADDR0   (*reinterpret_cast<volatile uint32_t*>(I2C5RegisterVirtualAddress + 0x04))
#define I2C5_CMD_DATA1   (*reinterpret_cast<volatile uint32_t*>(I2C5RegisterVirtualAddress + 0x0c))
#define I2C5_STATUS      (*reinterpret_cast<volatile uint32_t*>(I2C5RegisterVirtualAddress + 0x1c))
#define I2C5_CONFIG_LOAD (*reinterpret_cast<volatile uint32_t*>(I2C5RegisterVirtualAddress + 0x8c))

const nn::Bit32 CLK_ENB_I2C5 = (0x1 << 15);
const nn::Bit32 CLK_RST_I2C5 = (0x1 << 15);
const nn::Bit32 CLK_ENB_SOC_THERM = (0x1 << 14);
const nn::Bit32 CLK_RST_SOC_THERM = (0x1 << 14);

void EnterSc7()
{
    nn::Result result;

    //Initialize clock addrs
    ClkRstRegisterVirtualAddress = nn::dd::QueryIoMappingAddress(0x60006000, 0x1000);
    NN_ABORT_UNLESS(ClkRstRegisterVirtualAddress, "*** Clock Registers not mapped.");

    //Initialize I2C5 addrs to program PMIC for SC7 entry
    I2C5RegisterVirtualAddress = nn::dd::QueryIoMappingAddress(0x7000D000, 0x1000);
    NN_ABORT_UNLESS(I2C5RegisterVirtualAddress, "*** I2C5 Registers not mapped.");


    //Enable i2c5 clocks
    CLK_RST_RST_DEV_H_CLR  |= CLK_RST_I2C5;
    CLK_RST_CLK_OUT_ENB_H  |= CLK_ENB_I2C5;

    //wait 2us for logic to stabalize
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(2));
    CLK_RST_RST_DEV_H_CLR  &= ~CLK_RST_I2C5;

    //Enable SOC_THERM clock for SC7 Entry
    CLK_RST_RST_DEV_U_CLR  |= CLK_RST_SOC_THERM;
    CLK_RST_CLK_OUT_ENB_U  |= CLK_ENB_SOC_THERM;

    //wait 2us for logic to stabalize
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(2));
    CLK_RST_RST_DEV_U_CLR  &= ~CLK_RST_SOC_THERM;
    NN_SDK_LOG("Finished enabling clocks for I2C5 and SOC_THERM\n");


    // TORIAEZU: TODO:
    // 現状、SC7 に入ると正しく復帰できないドライバが多数存在する。
    // settings の設定が false の場合は実際の SC7 には入らない。
    // ※ ここの段階ではみんな寝ていて settings の値を読めないので値は PowerState::Initialize で読んでおく
#if 0
    I2C5_CMD_ADDR0 = (0x3c << 1);          // bus addr for I2C
    I2C5_CMD_DATA1 = (0x41 | (0x64 << 8)); // Reg, Val << 8
    I2C5_CNFG      = (0x2 << 12) | (1 << 11) | (1 << 1);
    I2C5_CONFIG_LOAD = 0x1;

    // Wait for i2c busy
    while (I2C5_CONFIG_LOAD & 0x1) {}

    // i2c send
    I2C5_CNFG |= (1 << 9);

    // Wait for i2c send to complete
    while(I2C5_STATUS & (1 << 8)) {}

    NN_SDK_LOG("PMIC config done for CORE_PWR_REQ\n");
#endif

    nn::dd::EnsureMemoryAccess();

    NN_SDK_LOG("Trigger SystemSuspend SVC\n");

    //
    // Enter SC7
    //
    // TORIAEZU: テストは後で消すことにして一旦 svc の直呼びを消す
}
#endif // NNT_POWERSTATE_TEST_ENTER_SC7

void CheckDispatchResult( nn::psc::PmControl& Control )
{
//NN_SDK_LOG("[%s] Result == Start.\n", __PRETTY_FUNCTION__);
nn::TimeSpan USBWaitTime = nn::TimeSpan::FromSeconds( 3 );
nn::TimeSpan HBXWaitTime = nn::TimeSpan::FromSeconds( 10 );
    if( !Control.GetEventPointer()->TimedWait( HBXWaitTime ) )
    {
        NN_SDK_LOG("Dispatch Request Timed Out!\n");
//NN_SDK_LOG("[%s] Result == A.\n", __PRETTY_FUNCTION__);
        Control.Cancel();
//NN_SDK_LOG("[%s] Result == B.\n", __PRETTY_FUNCTION__);
        Control.GetEventPointer()->Clear(); // Cancel によるシグナルを解除
//NN_SDK_LOG("[%s] Result == C.\n", __PRETTY_FUNCTION__);
        NN_SDK_LOG( "TEST FAILED.\n" );
    }
    else
    {
//NN_SDK_LOG("[%s] Result == D.\n", __PRETTY_FUNCTION__);
        Control.GetEventPointer()->Clear();
    }
//NN_SDK_LOG("[%s] Result == E.\n", __PRETTY_FUNCTION__);

//    Control.PrintModuleInformation();
//NN_SDK_LOG("[%s] Result == F.\n", __PRETTY_FUNCTION__);
    // result from dispatch operation
    const nn::Result nnResult = Control.GetResult();
//NN_SDK_LOG("[%s] Result == Finish: '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");
}

} // anonymous namespace

extern "C" void nnMain()
{
    nn::Result nnResult;
    nn::psc::PmControl Control;
//NN_SDK_LOG("[%s] Result == Start.\n", __PRETTY_FUNCTION__);

    // Sleep and Wake
    nnResult = Control.Initialize( nn::os::EventClearMode_ManualClear );
    NN_ABORT_UNLESS( nnResult.IsSuccess() );
//NN_SDK_LOG("[%s] Result == A '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");

    nnResult = Control.DispatchRequest( nn::psc::PmState_MinimumAwake, nn::psc::MakeNoPmFlags(), nn::psc::PmTransitionOrder_ToLowerPowerState );
//NN_SDK_LOG("[%s] Result == B '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");
    CheckDispatchResult( Control );

    nnResult = Control.DispatchRequest( nn::psc::PmState_SleepReady, nn::psc::MakeNoPmFlags(), nn::psc::PmTransitionOrder_ToLowerPowerState );
//NN_SDK_LOG("[%s] Result == C '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");
    CheckDispatchResult( Control );

    nnResult = Control.DispatchRequest( nn::psc::PmState_EssentialServicesSleepReady, nn::psc::MakeNoPmFlags(), nn::psc::PmTransitionOrder_ToLowerPowerState );
//NN_SDK_LOG("[%s] Result == D '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");
    CheckDispatchResult( Control );

#ifdef NNT_POWERSTATE_TEST_ENTER_SC7
    EnterSc7();
#endif

    nnResult = Control.DispatchRequest( nn::psc::PmState_EssentialServicesAwake, nn::psc::MakeNoPmFlags(), nn::psc::PmTransitionOrder_ToHigherPowerState );
//NN_SDK_LOG("[%s] Result == E '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");
    CheckDispatchResult( Control );

    nnResult = Control.DispatchRequest( nn::psc::PmState_MinimumAwake, nn::psc::MakeNoPmFlags(), nn::psc::PmTransitionOrder_ToHigherPowerState );
//NN_SDK_LOG("[%s] Result == F '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");
    CheckDispatchResult( Control );

    nnResult = Control.DispatchRequest( nn::psc::PmState_FullAwake, nn::psc::MakeNoPmFlags(), nn::psc::PmTransitionOrder_ToHigherPowerState );
//NN_SDK_LOG("[%s] Result == G '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");
    CheckDispatchResult( Control );
//NN_SDK_LOG("[%s] Result == H '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");

    Control.Finalize();
//NN_SDK_LOG("[%s] Result == Finish: '%s'.\n", __PRETTY_FUNCTION__, nnResult.IsSuccess() ? "Success" : "Failed");
}
