﻿#----------------------------------------------------------------------------
# Project:  NintendoWare
#
# Copyright (C)Nintendo All rights reserved.
#
# These coded instructions, statements, and computer programs contain proprietary
# information of Nintendo and/or its licensed developers and are protected by
# national and international copyright laws. They may not be disclosed to third
# parties or copied or duplicated in any form, in whole or in part, without the
# prior written consent of Nintendo.
#
# The content herein is highly confidential and should be handled accordingly.
#----------------------------------------------------------------------------

# 中間ファイルの更新、System.Xml.XmlDocument 型の doc を更新します
def Update(doc):

	# fsd 以外の <nw4f_3dif> version を 2.2.0 に変更します。
	if not doc.SelectSingleNode("/nw4f_3dif/shader_definition"):
		version = doc.SelectSingleNode("/nw4f_3dif/@version")
		version.Value = "2.2.0"

	# <create> に src_path を空文字列で追加します。
	create = doc.SelectSingleNode("/nw4f_3dif/file_info/create")
	if create:
		create.SetAttribute("src_path", "")

	# <optimize_log> を <process_log> に変更します。
	RenameElement(doc, "//optimize_log_array", "process_log_array")
	RenameElement(doc, "//optimize_log", "process_log")

	# モデルの更新
	model = doc.SelectSingleNode("/nw4f_3dif/model")
	if model:
		UpdateModel(doc, model)
		return

	# テクスチャの更新
	texture = doc.SelectSingleNode("/nw4f_3dif/texture")
	if texture:
		UpdateTexture(doc, texture)
		return

	# スケルタルアニメの更新
	skeletal_anim = doc.SelectSingleNode("/nw4f_3dif/skeletal_anim")
	if skeletal_anim:
		UpdateSkeletalAnim(doc, skeletal_anim)
		return

	# シェーダパラメータアニメの更新
	shader_param_anim = doc.SelectSingleNode("/nw4f_3dif/shader_param_anim")
	if shader_param_anim:
		UpdateShaderParamAnim(doc, shader_param_anim)
		return

	# テクスチャパターンアニメの更新
	tex_pattern_anim = doc.SelectSingleNode("/nw4f_3dif/tex_pattern_anim")
	if tex_pattern_anim:
		UpdateAnimInfo(tex_pattern_anim.SelectSingleNode("tex_pattern_anim_info"))
		return

	# ボーンビジビリティアニメの更新
	bone_visibility_anim = doc.SelectSingleNode("/nw4f_3dif/bone_visibility_anim")
	if bone_visibility_anim:
		UpdateAnimInfo(bone_visibility_anim.SelectSingleNode("bone_visibility_anim_info"))
		return

	# マテリアルビジビリティアニメの更新
	mat_visibility_anim = doc.SelectSingleNode("/nw4f_3dif/mat_visibility_anim")
	if mat_visibility_anim:
		UpdateAnimInfo(mat_visibility_anim.SelectSingleNode("mat_visibility_anim_info"))
		return

	# シェイプアニメの更新
	shape_anim = doc.SelectSingleNode("/nw4f_3dif/shape_anim")
	if shape_anim:
		UpdateAnimInfo(shape_anim.SelectSingleNode("shape_anim_info"))
		return

	# シーンアニメの更新
	scene_anim = doc.SelectSingleNode("/nw4f_3dif/scene_anim")
	if scene_anim:
		UpdateSceneAnim(doc, scene_anim)
		return

	# シェーダ設定の更新
	shader_config = doc.SelectSingleNode("/nw4f_3dif/shader_config")
	if shader_config:
		UpdateShaderConfig(doc, shader_config)

#==========================================================
# モデルの更新
def UpdateModel(doc, model):

	# <model> に <model_info> を dcc_magnify="1"、dcc_start_frame="0" で追加します。
	model_info = doc.CreateElement("model_info")
	model_info.SetAttribute("dcc_magnify", "1")
	model_info.SetAttribute("dcc_start_frame", "0")
	process_log_array = model.SelectSingleNode("process_log_array")
	if process_log_array:
		model.InsertAfter(model_info, process_log_array)
	else:
		model.PrependChild(model_info)


	for material in model.SelectNodes("material_array/material"):

		# <material_info> の env_name と pass_name を削除します。
		material_info = material.SelectSingleNode("material_info")
		material_info.RemoveAttribute("env_name")
		material_info.RemoveAttribute("pass_name")

		# <render_info_array> を <shader_assign> の内容に移動します。
		render_info_array = material.SelectSingleNode("render_info_array")
		if render_info_array:
			material.RemoveChild(render_info_array)

		shader_assign = material.SelectSingleNode("shader_assign")
		if not shader_assign:
			continue
		if render_info_array:
			shader_assign.PrependChild(render_info_array)

			# <render_info> に type を string で追加します。
			for render_info in render_info_array.SelectNodes("render_info"):
				render_info.SetAttribute("type", "string")

		# <sampler_assign> の sampler_name が空文字列だった場合に <sampler_assign> を削除します。
		for sampler_assign in shader_assign.SelectNodes("sampler_assign_array/sampler_assign"):
			sampler_name = sampler_assign.GetAttribute("sampler_name")
			if sampler_name == "":
				sampler_assign.ParentNode.RemoveChild(sampler_assign)

		# <shader_param> type の texsrt_maya、texsrt_3dsmax、texsrt_softimage を texsrt に、
		# texsrt_ex_maya、texsrt_ex_3dsmax、texsrt_ex_softimage を texsrt_ex に変更します。
		# 元の設定に合わせてモード値を追加します。
		for shader_param in shader_assign.SelectNodes("shader_param_array/shader_param"):
			type = shader_param.GetAttribute("type")
			mode = ""
			if type == "texsrt_maya":
				type = "texsrt"
				mode = "0 "
			elif type == "texsrt_3dsmax":
				type = "texsrt"
				mode = "1 "
			elif type == "texsrt_softimage":
				type = "texsrt"
				mode = "2 "
			elif type == "texsrt_ex_maya":
				type = "texsrt_ex"
				mode = "0 "
			elif type == "texsrt_ex_3dsmax":
				type = "texsrt_ex"
				mode = "1 "
			elif type == "texsrt_ex_softimage":
				type = "texsrt_ex"
				mode = "2 "
			if mode != "":
				shader_param.SetAttribute("type", type)
				shader_param.InnerText = mode + shader_param.InnerText

		# <attrib_assign> の attrib_name が空文字列だった場合に <attrib_assign> を削除します。
		for attrib_assign in shader_assign.SelectNodes("attrib_assign_array/attrib_assign"):
			attrib_name = attrib_assign.GetAttribute("attrib_name")
			if attrib_name == "":
				attrib_assign.ParentNode.RemoveChild(attrib_assign)

	# <bone> billboard の world、screen、yaxis を
	# world_viewvector、screen_viewvector、yaxis_viewvector に変更します。
	for bone in model.SelectNodes("skeleton/bone_array/bone"):
		billboard = bone.GetAttribute("billboard")
		if billboard == "world":
			bone.SetAttribute("billboard", "world_viewvector")
		elif billboard == "screen":
			bone.SetAttribute("billboard", "screen_viewvector")
		elif billboard == "yaxis":
			bone.SetAttribute("billboard", "yaxis_viewvector")

	for shape in model.SelectNodes("shape_array/shape"):

		# <shape_info> の optimize_primitive_hash を original_shape_hash に変更します。
		shape_info = shape.SelectSingleNode("shape_info")
		RenameAttribute(shape_info, "@optimize_primitive_hash", "original_shape_hash")

		# <shape_info> に original_shape_count を 0 で、divide_submesh_mode を空文字列で追加します。
		shape_info.SetAttribute("original_shape_count", "0")
		shape_info.SetAttribute("divide_submesh_mode", "")

		for submesh in shape.SelectNodes("mesh/submesh_array/submesh"):

			# <submesh> の index を submesh_index に変更します。
			RenameAttribute(submesh, "@index", "submesh_index")

			# <submesh> の base_vertex を削除します。
			submesh.RemoveAttribute("base_vertex")

#==========================================================
# テクスチャの更新
def UpdateTexture(doc, texture):

	# <texture_info> に hint、linear、initial_swizzle を追加します。
	texture_info = texture.SelectSingleNode("texture_info")

	# hint は quantize_type が snorm_bc5 であれば "normal" に、それ以外であれば空文字列で追加します。
	quantize_type = texture_info.GetAttribute("quantize_type")
	if quantize_type == "snorm_bc5":
		texture_info.SetAttribute("hint", "normal")
	else:
		texture_info.SetAttribute("hint", "")

	# linear は "false false false false" で、initial_swizzle は 0 で追加します。
	texture_info.SetAttribute("linear", "false false false false")
	texture_info.SetAttribute("initial_swizzle", "0")

#==========================================================
# スケルタルアニメの更新
def UpdateSkeletalAnim(doc, skeletal_anim):

	skeletal_anim_info = skeletal_anim.SelectSingleNode("skeletal_anim_info")
	UpdateAnimInfo(skeletal_anim_info)

	# <skeletal_anim_info> に dcc_magnify を 1 で追加します。
	skeletal_anim_info.SetAttribute("dcc_magnify", "1")

#==========================================================
# シェーダパラメータアニメの更新
def UpdateShaderParamAnim(doc, shader_param_anim):

	UpdateAnimInfo(shader_param_anim.SelectSingleNode("shader_param_anim_info"))

	# <param_anim> type の texsrt_maya、texsrt_3dsmax、texsrt_softimage を texsrt に、
	# texsrt_ex_maya、texsrt_ex_3dsmax、texsrt_ex_softimage を texsrt_ex に変更します。
	for param_anim in shader_param_anim.SelectNodes(
		"shader_param_mat_anim_array/shader_param_mat_anim/param_anim_array/param_anim"):
		type = param_anim.GetAttribute("type")
		mode = ""
		if type == "texsrt_maya":
			type = "texsrt"
			mode = "0"
		elif type == "texsrt_3dsmax":
			type = "texsrt"
			mode = "1"
		elif type == "texsrt_softimage":
			type = "texsrt"
			mode = "2"
		elif type == "texsrt_ex_maya":
			type = "texsrt_ex"
			mode = "0"
		elif type == "texsrt_ex_3dsmax":
			type = "texsrt_ex"
			mode = "1"
		elif type == "texsrt_ex_softimage":
			type = "texsrt_ex"
			mode = "2"
		if mode != "":
			param_anim.SetAttribute("type", type)

			# component_index をインクリメントします。
			for param_anim_target in param_anim.SelectNodes("param_anim_target"):
				component_index = int(param_anim_target.GetAttribute("component_index")) + 1
				param_anim_target.SetAttribute("component_index", str(component_index))

			# 元の設定に合わせてモード値の <param_anim_target> を追加します。
			mode_target = doc.CreateElement("param_anim_target")
			mode_target.SetAttribute("component_index", "0")
			mode_target.SetAttribute("base_value", mode)
			param_anim.PrependChild(mode_target)

#==========================================================
# シーンアニメの更新
def UpdateSceneAnim(doc, scene_anim):

	scene_anim_info = scene_anim.SelectSingleNode("scene_anim_info")
	UpdateAnimInfo(scene_anim_info)

	# <scene_anim_info> に dcc_magnify を 1 で追加します。
	scene_anim_info.SetAttribute("dcc_magnify", "1")

#==========================================================
# シェーダ設定の更新
def UpdateShaderConfig(doc, shader_config):

	# <variation> に branch を false で追加します。
	for variation in shader_config.SelectNodes(
		"shader_array/shader/variation_array/variation"):
		variation.SetAttribute("branch", "false")

#==========================================================
# アニメ情報の更新
def UpdateAnimInfo(anim_info):
	# dcc_start_frame を 0 で、dcc_end_frame を frame_count で、dcc_fps を 60 で追加します。
	anim_info.SetAttribute("dcc_start_frame", "0")
	frame_count = anim_info.GetAttribute("frame_count")
	if frame_count:
		anim_info.SetAttribute("dcc_end_frame", frame_count)
	else:
		# ただし、<scene_anim_info> dcc_end_frame は 60 で追加します。
		anim_info.SetAttribute("dcc_end_frame", "60")
	anim_info.SetAttribute("dcc_fps", "60")

	# quantize_tolerance_frame を削除します。
	anim_info.RemoveAttribute("quantize_tolerance_frame")

#----------------------------------------------------------
# 要素のリネーム
def RenameElement(doc, xPath, newName):
	for oldElement in doc.SelectNodes(xPath):
		newElement = doc.CreateElement(newName)

		# 属性の移植
		for attribute in oldElement.Attributes:
			newElement.SetAttribute(attribute.Name, attribute.Value)

		# 子要素の移植
		while oldElement.HasChildNodes:
			newElement.AppendChild(oldElement.FirstChild)

		# 要素の差し替え
		oldElement.ParentNode.ReplaceChild(newElement, oldElement)

#----------------------------------------------------------
# 属性のリネーム
def RenameAttribute(target, xPath, newName):
	for oldAttribute in target.SelectNodes(xPath):
		ownerElement = oldAttribute.OwnerElement
		ownerElement.RemoveAttributeNode(oldAttribute)
		ownerElement.SetAttribute(newName, oldAttribute.Value)

#==========================================================
