﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MemoryManagement.h"

#include <cstdlib>
#include <nn/mem.h>
#include <nv/nv_MemoryManagement.h>

namespace {
    static const size_t DonateMemorySize = 128 * 1024;
    static const size_t HeapMemorySize   = 256 * 1024;

    bool g_IsMemoryInitialized = false;

    NN_ALIGNAS(4096) char g_HeapMemory[HeapMemorySize];

    nn::mem::StandardAllocator g_Allocator;


    void* GraphicsAllocate(size_t size, size_t alignment, void*)
    {
        //NN_LOG("gr alloc\n");
        return aligned_alloc(alignment, size);
    }
    void GraphicsFree(void* p, void*)
    {
        //NN_LOG("gr free\n");
        free(p);
    }
    void* GraphicsReallocate(void* p, size_t size, void*)
    {
        //NN_LOG("gr realloc\n");
        return realloc(p, size);
    }
}

void SetupMemory() NN_NOEXCEPT
{
    if(g_IsMemoryInitialized)
    {
        return;
    }

    g_Allocator.Initialize(g_HeapMemory, sizeof(g_HeapMemory));

    nv::SetGraphicsAllocator(GraphicsAllocate, GraphicsFree, GraphicsReallocate, nullptr);
    nv::SetGraphicsDevtoolsAllocator(GraphicsAllocate, GraphicsFree, GraphicsReallocate, nullptr);
    nv::InitializeGraphics(aligned_alloc(4 * 1024, DonateMemorySize), DonateMemorySize);

    g_IsMemoryInitialized = true;
}

void ReportMemoryUsage() NN_NOEXCEPT
{
    //static size_t g_AllocatedSizeMax = 0;
    //size_t s = HeapMemorySize - g_Allocator.GetTotalFreeSize();
    //g_AllocatedSizeMax = std::max(g_AllocatedSizeMax, s);
    //NN_SDK_LOG("usage: %lld (max %lld) B\n", static_cast<int64_t>(s), static_cast<int64_t>(g_AllocatedSizeMax));
}

extern "C" void* malloc(size_t size)
{
    auto p = g_Allocator.Allocate(size);
    ReportMemoryUsage();
    return p;
}

extern "C" void free(void* p)
{
    if(p)
    {
        g_Allocator.Free(p);
    }
    ReportMemoryUsage();
}

extern "C" void* calloc(size_t num, size_t size)
{
    size_t sum = num * size;
    void*  p   = malloc(sum);
    if (p)
    {
        std::memset(p, 0, sum);
    }
    return p;
}

extern "C" void* realloc(void* p, size_t newSize)
{
    void* q = g_Allocator.Reallocate(p, newSize);
    ReportMemoryUsage();
    return q;
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    void* p = g_Allocator.Allocate(size, alignment);
    ReportMemoryUsage();
    return p;
}

