﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include "settings/visrv_DisplayController.h"
#include "settings/visrv_InterfaceType.h"
#include "SettingsFixture.h"
#include "SettingsStub.h"

typedef SettingsFixture DisplayController;

TEST_F(DisplayController, Ctor)
{
    nn::visrv::settings::DisplayController dc;
    EXPECT_EQ(-1, dc.priority);
    EXPECT_EQ(nn::visrv::settings::InterfaceType::Null, dc.interface);
    EXPECT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
    }
}

TEST_F(DisplayController, Read)
{
    const int Priority = 1;
    const nn::visrv::settings::InterfaceType Interface = nn::visrv::settings::InterfaceType::Hdmi;
    const int ModeCount = 1;

    const int ModeWidth = 1920;
    const int ModeHeight = 1080;
    const int ModeRefresh = 60000;
    const nn::vi::StereoMode ModeStereo = nn::vi::StereoMode_SideBySide;
    const nn::vi::AspectRatio ModeAspectRatio = nn::vi::AspectRatio_16_9;

    nnt::settings::WriteValue(Priority, "default_display_dc_0_priority");
    nnt::settings::WriteValue(Interface, "dc_0_interface");
    nnt::settings::WriteValue(ModeCount, "dc_0_mode_count");
    nnt::settings::WriteValue(ModeWidth, "dc_0_mode_0_width");
    nnt::settings::WriteValue(ModeHeight, "dc_0_mode_0_height");
    nnt::settings::WriteValue(ModeRefresh, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(ModeStereo, "dc_0_mode_0_stereo_mode");
    nnt::settings::WriteValue(ModeAspectRatio, "dc_0_mode_0_aspect_ratio");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    EXPECT_EQ(Priority, dc.priority);
    EXPECT_EQ(Interface, dc.interface);
    ASSERT_EQ(ModeCount, dc.modeCount);
    EXPECT_EQ(ModeWidth, dc.modes[0].clientInfo.width);
    EXPECT_EQ(ModeHeight, dc.modes[0].clientInfo.height);
    EXPECT_FLOAT_EQ(60.f, dc.modes[0].clientInfo.refreshRate);
    EXPECT_EQ(ModeStereo, dc.modes[0].clientInfo.mode);
    EXPECT_EQ(ModeAspectRatio, dc.modes[0].ratio);

    for( int i = 1; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_NoPriority)
{
    nn::visrv::settings::DisplayController dc;
    dc.Read(0);
    EXPECT_LT(dc.priority, 0);
}

TEST_F(DisplayController, Read_NoInterface)
{
    nn::visrv::settings::DisplayController dc;
    dc.Read(0);
    EXPECT_EQ(nn::visrv::settings::InterfaceType::Null, dc.interface);
}

TEST_F(DisplayController, Read_NoModes)
{
    nn::visrv::settings::DisplayController dc;
    dc.Read(0);
    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_InvalidWidth)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(-1, "dc_0_mode_0_width");
    nnt::settings::WriteValue(1080, "dc_0_mode_0_height");
    nnt::settings::WriteValue(60000, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(nn::vi::StereoMode_None, "dc_0_mode_0_stereo_mode");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_InvalidHeight)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(1920, "dc_0_mode_0_width");
    nnt::settings::WriteValue(-1, "dc_0_mode_0_height");
    nnt::settings::WriteValue(60000, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(nn::vi::StereoMode_None, "dc_0_mode_0_stereo_mode");
    nnt::settings::WriteValue(nn::vi::AspectRatio_16_9, "dc_0_mode_0_aspect_ratio");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_InvalidRefreshRate)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(1920, "dc_0_mode_0_width");
    nnt::settings::WriteValue(1080, "dc_0_mode_0_height");
    nnt::settings::WriteValue(-1, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(nn::vi::StereoMode_None, "dc_0_mode_0_stereo_mode");
    nnt::settings::WriteValue(nn::vi::AspectRatio_16_9, "dc_0_mode_0_aspect_ratio");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_InvalidStereoMode)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(1920, "dc_0_mode_0_width");
    nnt::settings::WriteValue(1080, "dc_0_mode_0_height");
    nnt::settings::WriteValue(60000, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(-1, "dc_0_mode_0_stereo_mode");
    nnt::settings::WriteValue(nn::vi::AspectRatio_16_9, "dc_0_mode_0_aspect_ratio");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }

    nnt::settings::WriteValue(std::numeric_limits<std::underlying_type<nn::vi::StereoMode>::type>::max(), "dc_0_mode_0_stereo_mode");
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_InvalidAspectRatio)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(1920, "dc_0_mode_0_width");
    nnt::settings::WriteValue(1080, "dc_0_mode_0_height");
    nnt::settings::WriteValue(60000, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(nn::vi::StereoMode_None, "dc_0_mode_0_stereo_mode");
    nnt::settings::WriteValue(-1, "dc_0_mode_0_aspect_ratio");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }

    nnt::settings::WriteValue(std::numeric_limits<std::underlying_type<nn::vi::AspectRatio>::type>::max(), "dc_0_mode_0_stereo_mode");
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_DefaultWidth)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(1080, "dc_0_mode_0_height");
    nnt::settings::WriteValue(60000, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(nn::vi::StereoMode_None, "dc_0_mode_0_stereo_mode");
    nnt::settings::WriteValue(nn::vi::AspectRatio_16_9, "dc_0_mode_0_aspect_ratio");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_DefaultHeight)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(1920, "dc_0_mode_0_width");
    nnt::settings::WriteValue(60000, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(nn::vi::StereoMode_None, "dc_0_mode_0_stereo_mode");
    nnt::settings::WriteValue(nn::vi::AspectRatio_16_9, "dc_0_mode_0_aspect_ratio");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_DefaultRefreshRate)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(1920, "dc_0_mode_0_width");
    nnt::settings::WriteValue(1080, "dc_0_mode_0_height");
    nnt::settings::WriteValue(nn::vi::StereoMode_None, "dc_0_mode_0_stereo_mode");
    nnt::settings::WriteValue(nn::vi::AspectRatio_16_9, "dc_0_mode_0_aspect_ratio");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(0, dc.modeCount);

    for( int i = 0; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_DefaultStereoMode)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(1920, "dc_0_mode_0_width");
    nnt::settings::WriteValue(1080, "dc_0_mode_0_height");
    nnt::settings::WriteValue(60000, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(nn::vi::AspectRatio_16_9, "dc_0_mode_0_aspect_ratio");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(1, dc.modeCount);
    EXPECT_EQ(1920, dc.modes[0].clientInfo.width);
    EXPECT_EQ(1080, dc.modes[0].clientInfo.height);
    EXPECT_EQ(60.f, dc.modes[0].clientInfo.refreshRate);
    EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[0].clientInfo.mode);
    EXPECT_EQ(nn::vi::AspectRatio_16_9, dc.modes[0].ratio);

    for( int i = 1; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}

TEST_F(DisplayController, Read_DefaultAspectRatio)
{
    nnt::settings::WriteValue(1, "dc_0_mode_count");
    nnt::settings::WriteValue(1920, "dc_0_mode_0_width");
    nnt::settings::WriteValue(1080, "dc_0_mode_0_height");
    nnt::settings::WriteValue(60000, "dc_0_mode_0_refresh_rate");
    nnt::settings::WriteValue(nn::vi::StereoMode_None, "dc_0_mode_0_stereo_mode");

    nn::visrv::settings::DisplayController dc;
    dc.Read(0);

    ASSERT_EQ(1, dc.modeCount);
    EXPECT_EQ(1920, dc.modes[0].clientInfo.width);
    EXPECT_EQ(1080, dc.modes[0].clientInfo.height);
    EXPECT_EQ(60.f, dc.modes[0].clientInfo.refreshRate);
    EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[0].clientInfo.mode);
    EXPECT_EQ(nn::vi::AspectRatio_16_9, dc.modes[0].ratio);

    for( int i = 1; i < sizeof(dc.modes) / sizeof(dc.modes[0]); ++i )
    {
        EXPECT_EQ(0, dc.modes[i].clientInfo.width);
        EXPECT_EQ(0, dc.modes[i].clientInfo.height);
        EXPECT_EQ(0.f, dc.modes[i].clientInfo.refreshRate);
        EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[i].clientInfo.mode);
        EXPECT_EQ(nn::vi::AspectRatio_Unknown, dc.modes[i].ratio);
    }
}
