﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/vi.private.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nv/nv_MemoryManagement.h>
#include <nn/vi/vi_Content.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/vi_LayerConfig.h>
#include "testVi_Graphics.h"

static void* NvAllocate(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return std::aligned_alloc(alignment, size);
}

static void NvFree(void* ptr, void* userPtr)
{
    NN_UNUSED(userPtr);
    std::free(ptr);
}

static void* NvReallocate(void* ptr, size_t size, void* userPtr)
{
    NN_UNUSED(userPtr);
    return std::realloc(ptr, size);
}

extern "C" void nnMain()
{
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::InitializeGraphics( std::malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );

    nn::vi::Initialize();

    nn::vi::Display* pDisplay;

    {
        nn::Result result = nn::vi::OpenDefaultDisplay(&pDisplay);
        NN_ASSERT(result.IsSuccess(), "Failed to open default display.");
    }

    //{
    //    nn::vi::Display* pLcd;
    //    nn::Result result = nn::vi::OpenDisplay(&pLcd, "Internal");
    //    NN_ASSERT(result.IsSuccess(), "Failed to open internal display.");
    //    nn::vi::SetDisplayPowerState(pLcd, nn::vi::PowerState_Off);
    //    nn::vi::CloseDisplay(pLcd);
    //}

    //{
    //    nn::vi::Display* pHdmi;
    //    nn::Result result = nn::vi::OpenDisplay(&pHdmi, "External");
    //    NN_ASSERT(result.IsSuccess(), "Failed to open external display.");
    //    nn::vi::SetDisplayPowerState(pHdmi, nn::vi::PowerState_Off);
    //    nn::vi::CloseDisplay(pHdmi);
    //}

    nn::vi::Layer* pLayer;

    const int LayerWidth = 300;
    const int LayerHeight = 300;

    {
        nn::Result result = nn::vi::CreateLayer(&pLayer, pDisplay);
        NN_ASSERT(result.IsSuccess(), "Failed to create layer.");
    }

    {
        nn::Result result = nn::vi::AddToLayerStack(pLayer, nn::vi::LayerStack_Null);
        NN_ASSERT(result.IsSuccess(), "Failed to add to null layer stack.");
    }

    {
        nn::Result result = nn::vi::RemoveFromLayerStack(pLayer, nn::vi::LayerStack_Default);
        NN_ASSERT(result.IsSuccess(), "Failed to remove from default layer stack.");
    }

    Graphics g;

    g.RegisterLayer(pLayer, LayerWidth, LayerHeight, Color(1.f, 0.f, 0.f));

    for (;;)
    {
        auto start = nn::os::GetSystemTick();
        g.Update();
        auto end = nn::os::GetSystemTick();
        NN_LOG("%f ms\n", (end - start).ToTimeSpan().GetMicroSeconds() / 1000.f);
    }

    nn::vi::Finalize();
}
