﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>
#include "testVi_MemoryFixture.h"

namespace
{
    void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(userPtr);
        void* ptr = static_cast<nn::mem::StandardAllocator*>(userPtr)->Allocate(size, alignment);
        NN_ASSERT_NOT_NULL(ptr);
        return ptr;
    }

    void NvFree(void* ptr, void* userPtr) NN_NOEXCEPT
    {
        if( ptr == nullptr )
        {
            return;
        }

        NN_ASSERT_NOT_NULL(userPtr);
        static_cast<nn::mem::StandardAllocator*>(userPtr)->Free(ptr);
    }

    void* NvReallocate(void* ptr, size_t size, void* userPtr) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(userPtr);
        return static_cast<nn::mem::StandardAllocator*>(userPtr)->Reallocate(ptr, size);
    }
}

class MemoryFixtureImpl
{
public:
    MemoryFixtureImpl()
        : m_GraphicsMem(nullptr)
        , m_AllocatorMem(nullptr)
    {
    }

    ~MemoryFixtureImpl()
    {
    }

    void Initialize() NN_NOEXCEPT
    {
        m_GraphicsMem = std::malloc(GraphicsMemorySize);
        ASSERT_NE(nullptr, m_GraphicsMem);

        m_AllocatorMem = std::malloc(AllocatorMemorySize);
        ASSERT_NE(nullptr, m_AllocatorMem);

        m_Allocator.Initialize(m_AllocatorMem, GraphicsMemorySize);

        nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, &m_Allocator);
        nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, &m_Allocator);
        nv::InitializeGraphics(m_GraphicsMem, GraphicsMemorySize);
    }

    void Finalize() NN_NOEXCEPT
    {
        nv::FinalizeGraphics();
        CheckLeaks();

        m_Allocator.Finalize();
        std::free(m_AllocatorMem);

        std::free(m_GraphicsMem);
    }

    void CheckLeaks() NN_NOEXCEPT
    {
        EXPECT_EQ(0, m_Allocator.Hash().allocCount);
        m_Allocator.Dump();
    }
private:
    static const size_t GraphicsMemorySize = 8 * 1024 * 1024;
    static const size_t AllocatorMemorySize = 8 * 1024 * 1024;

    void* m_GraphicsMem;
    void* m_AllocatorMem;

    nn::mem::StandardAllocator m_Allocator;
};

MemoryFixture::MemoryFixture() NN_NOEXCEPT
    : m_Impl(new MemoryFixtureImpl)
{
}

MemoryFixture::~MemoryFixture() NN_NOEXCEPT
{
}

void MemoryFixture::SetUp() NN_NOEXCEPT
{
    m_Impl->Initialize();
}

void MemoryFixture::TearDown() NN_NOEXCEPT
{
    m_Impl->Finalize();
}

void MemoryFixture::CheckLeaks() NN_NOEXCEPT
{
    m_Impl->CheckLeaks();
}
