﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/vi.private.h>

typedef nnt::vi::util::Fixture SetLayerPosition;

TEST_F(SetLayerPosition, FullscreenEnabled_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer;
    nn::vi::LayerSettings settings;
    settings.Reset();
    settings.Set<nn::vi::LayerFlags::Fullscreen>();
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay, settings));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, 0.f, 0.f));

    // out of bounds
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, -1.f, 0.f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, 0.f, -1.f));

    // fullscreen scaling is set, so anything away from the origin will be out of range
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, 0.f, 1.f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, 1.f, 0.f));

    nn::vi::DestroyLayer(pLayer);
    nn::vi::CloseDisplay(pDisplay);
}

// all applets will use fullscreen...
TEST_F(SetLayerPosition, DISABLED_FullscreenDisabled_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer;
    nn::vi::LayerSettings settings;
    settings.Reset();
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay, settings));

    const int LayerWidth  = 200;
    const int LayerHeight = 200;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, LayerWidth, LayerHeight));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, pDisplay));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, 0.f, 0.f));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, width - LayerWidth, 0.f));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, 0.f, height - LayerHeight));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, width - LayerWidth, height - LayerHeight));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, -1.f, 0.f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, 0.f, -1.f));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, width - LayerWidth + 1, 0.f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, 0.f, height - LayerHeight + 1));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerPosition(pLayer, width - LayerWidth + 1, height - LayerHeight + 1));

    nn::vi::DestroyLayer(pLayer);
    nn::vi::CloseDisplay(pDisplay);
}

TEST(SetLayerPositionPreconditions, ViNotInitialized_Any)
{
    nn::vi::Layer* pLayer = reinterpret_cast<nn::vi::Layer*>(0x1234);
    ASSERT_DEATH_IF_SUPPORTED(nn::vi::SetLayerPosition(pLayer, 0.f, 0.f), "");
}

TEST(SetLayerPositionPreconditions, pLayerNull_Any)
{
    nn::vi::Initialize();

    EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetLayerPosition(nullptr, 0.f, 0.f), "");

    nn::vi::Finalize();
}

TEST(SetLayerPositionPreconditions, pLayerInvalidHandle_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::Layer* pLayer = nullptr;
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay));

        if( pLayer != nullptr )
        {
            nn::vi::DestroyLayer(pLayer);
            EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetLayerPosition(pLayer, 0.f, 0.f), "");
        }

        nn::vi::CloseDisplay(pDisplay);
    }

    nn::vi::Finalize();
}
