﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "testVi_Context.h"
#include "testVi_Macro.h"
#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/vi/fbshare/vi_SharedLayerWindowImpl.h>

#include "testVi_NvnUtility.h"
#include "testVi_SceneUtility.h"

static NN_ALIGNAS(4096) char g_CommandStorage[10 * 4096];
static NN_ALIGNAS(4096) char g_ControlStorage[4096];

static const int DefaultLoopCount  = 3;
static const int DefaultFrameCount = 100;

NNT_VI_TEST_SHIM(SharedLayerWindowImpl_Basic)
{
    static const int LoopCount = DefaultLoopCount;
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;
    static const int FrameCount = DefaultFrameCount;
    NN_LOG("SharedLayerWindow>Basic-------------------------------\n");
    static const int SubFrameCount = 10;

    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);

        ContextExt context0({"server"});
        ContextExt context1({"client"});

        context0.ConnectService();
        context1.ConnectService();

        // setup SharedLowLevelLayer
        auto layerId = context0.GetDefaultLayerId();
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

        // setup SharedBuffer
        auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

        // setup SharedLayer
        nn::vi::fbshare::SharedLayerHandle hLayer = {};
        nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

        // attach
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1));

        int attachedIndex0 = 0;
        int attachedIndex1 = 0;

        // export SharedBuffer to me
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.RegisterSharedBufferImporterAruid(hBuffer, nn::applet::GetAppletResourceUserId()));

        {
            // init nvn and device
            NNT_VI_SCOPED_DEVICE           (device);
            NNT_VI_SCOPED_COMMANDMEMORYPOOL(commandPool, device, g_CommandStorage, sizeof(g_CommandStorage));
            NNT_VI_SCOPED_QUEUE            (queue, device);
            NNT_VI_SCOPED_COMMANDBUFFER    (commandBuffer, device);
            NNT_VI_SCOPED_FRAMETEXTUREVIEW (texView);

            nn::vi::fbshare::SharedLayerWindowImpl sharedLayerWindow;
            sharedLayerWindow.Initialize(context1.GetSystemService(), &device, hBuffer, hLayer, 2, NVN_FORMAT_RGBA8_SRGB);

            for(int frame = 0; frame < FrameCount; frame++)
            {
                if(frame % SubFrameCount == 0)
                {
                    context0.SafeDetachSharedLayerFromLowLevelLayer(hLayer);
                    int iBuf0 = (frame / SubFrameCount) % FrameBufferCount;
                    int iBuf1 = (iBuf0 + 1) % FrameBufferCount;
                    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, iBuf0, iBuf1));
                    //nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                    attachedIndex0 = iBuf0;
                    attachedIndex1 = iBuf1;
                }

                NNT_VI_SCOPED_SYNC(acqSync, device);
                int index = -1;
                NVNtexture* pTargetTexture = nullptr;

                while(sharedLayerWindow.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(16)).IsFailure())
                {
                    NN_LOG("waiting for pre-acquire texture\n");
                }
                sharedLayerWindow.AcquirePreAcquiredTexture(&acqSync, &index, &pTargetTexture);
                int internalTexIdx = -1;
                NNT_VI_EXPECT_SUCCESS(sharedLayerWindow.GetInternalTextureIndex(&internalTexIdx, pTargetTexture));
                EXPECT_TRUE(internalTexIdx == attachedIndex0 || internalTexIdx == attachedIndex1);
                EXPECT_EQ(nvnTextureGetWidth(pTargetTexture), sharedLayerWindow.GetTextureWidth(pTargetTexture));
                EXPECT_EQ(nvnTextureGetHeight(pTargetTexture), sharedLayerWindow.GetTextureHeight(pTargetTexture));
                NN_LOG("Frame %d / index = %d / internal=%d\n", frame, index, internalTexIdx);

                nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, sizeof(g_CommandStorage));
                nvnCommandBufferAddControlMemory(&commandBuffer, g_ControlStorage, sizeof(g_ControlStorage));
                nvnCommandBufferBeginRecording(&commandBuffer);

                // wait display
                nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

                nnt::vi::scene::DrawRotationBoxScene(&commandBuffer, pTargetTexture, &texView, frame / 60.f);

                auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);

                NN_LOG("submitting\n");
                nvnQueueSubmitCommands(&queue, 1, &hCommand);
                nvnQueueFinish(&queue);

                NN_LOG("presenting\n");
                sharedLayerWindow.PresentTexture(&queue, index);
            }

            context0.SafeDetachSharedLayerFromLowLevelLayer(hLayer);
            sharedLayerWindow.Finalize();
        }

        context0.DisconnectService();
        context1.DisconnectService();
        context0.CleanSharedBuffer(SlotIndex);
    }
}

NNT_VI_TEST_SHIM(SharedLayerWindowImpl_Canceling)
{
    static const int LoopCount = DefaultLoopCount;
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;
    static const int FrameCount = DefaultFrameCount;
    NN_LOG("SharedLayerWindow>Canceling-------------------------------\n");
    static const int SubFrameCount = 10;

    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);

        ContextExt context0({"server"});
        ContextExt context1({"client"});

        context0.ConnectService();
        context1.ConnectService();

        // setup SharedLowLevelLayer
        auto layerId = context0.GetDefaultLayerId();
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

        // setup SharedBuffer
        auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

        // setup SharedLayer
        nn::vi::fbshare::SharedLayerHandle hLayer = {};
        nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

        nn::os::SystemEventType detachReadyEvent = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetSharedLayerDetachReadyEvent(&detachReadyEvent, hLayer));
        NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&detachReadyEvent); };

        // attach
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));

        // export SharedBuffer to me
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.RegisterSharedBufferImporterAruid(hBuffer, nn::applet::GetAppletResourceUserId()));

        {
            // init nvn and device
            NNT_VI_SCOPED_DEVICE           (device);
            NNT_VI_SCOPED_COMMANDMEMORYPOOL(commandPool, device, g_CommandStorage, sizeof(g_CommandStorage));
            NNT_VI_SCOPED_QUEUE            (queue, device);
            NNT_VI_SCOPED_COMMANDBUFFER    (commandBuffer, device);
            NNT_VI_SCOPED_FRAMETEXTUREVIEW (texView);

            nn::vi::fbshare::SharedLayerWindowImpl sharedLayerWindow;
            sharedLayerWindow.Initialize(context1.GetSystemService(), &device, hBuffer, hLayer, 2, NVN_FORMAT_RGBA8_SRGB);

            for(int frame = 0; frame < FrameCount; frame++)
            {
                NNT_VI_SCOPED_SYNC(acqSync, device);
                int index = -1;
                NVNtexture* pTargetTexture = nullptr;

                EXPECT_FALSE(sharedLayerWindow.IsTextureAcquired());
                EXPECT_FALSE(sharedLayerWindow.IsTexturePreAcquired());

                // PreAcquireTexture する前に呼びだす
                NN_LOG("AcquirePreAcquiredTexture\n");
                sharedLayerWindow.AcquirePreAcquiredTexture(&acqSync, &index, &pTargetTexture);
                EXPECT_TRUE(sharedLayerWindow.IsTextureAcquired());
                EXPECT_FALSE(sharedLayerWindow.IsTexturePreAcquired());
                EXPECT_EQ(pTargetTexture, nullptr); // nullptr が取れるべき
                EXPECT_EQ(nvnSyncWait(&acqSync, 0), NVN_SYNC_WAIT_RESULT_ALREADY_SIGNALED); // 空の sync オブジェクトになるべき
                sharedLayerWindow.CancelTexture(index); // 戻しておく

                NN_LOG("PreAcquireTexture\n");
                while(sharedLayerWindow.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(16)).IsFailure())
                {
                    NN_LOG("  waiting for pre-acquire texture\n");
                }
                EXPECT_FALSE(sharedLayerWindow.IsTextureAcquired());
                EXPECT_TRUE(sharedLayerWindow.IsTexturePreAcquired());
                NNT_EXPECT_RESULT_SUCCESS(sharedLayerWindow.PreAcquireTexture(nn::TimeSpan())); // 一度成功したら繰り返し呼んでも成功を返す

                NN_LOG("AcquirePreAcquiredTexture\n");
                sharedLayerWindow.AcquirePreAcquiredTexture(&acqSync, &index, &pTargetTexture);
                EXPECT_TRUE(sharedLayerWindow.IsTextureAcquired());
                EXPECT_FALSE(sharedLayerWindow.IsTexturePreAcquired());

                NN_LOG("Frame %d / index = %d\n", frame, index);

                if(frame % SubFrameCount == 0)
                {
                    // Acquire してないテクスチャに対してキャンセルを呼んでも大丈夫
                    {
                        sharedLayerWindow.CancelPreAcquiredTexture();
                        for(int k = 0; k < FrameBufferCount; k++)
                        {
                            if(k == index)
                            {
                                continue;
                            }
                            sharedLayerWindow.CancelTexture(k);
                        }
                    }

                    NN_LOG("Detaching\n");
                    NN_LOG("  starting detach texture\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.StartDetachSharedLayerFromLowLevelLayer(hLayer));
                    NN_LOG("  canceling texture\n");

                    // Acquire してないテクスチャに対してキャンセルを呼んでも大丈夫
                    {
                        sharedLayerWindow.CancelPreAcquiredTexture();
                        for(int k = 0; k < FrameBufferCount; k++)
                        {
                            if(k == index)
                            {
                                continue;
                            }
                            sharedLayerWindow.CancelTexture(k);
                        }
                    }

                    sharedLayerWindow.CancelTexture(index);
                    EXPECT_FALSE(sharedLayerWindow.IsTextureAcquired());
                    EXPECT_FALSE(sharedLayerWindow.IsTexturePreAcquired());
                    sharedLayerWindow.CancelTexture(index); // 重複 ok

                    NN_LOG("  waiting for detach ready\n");
                    nn::os::WaitSystemEvent(&detachReadyEvent);

                    NN_LOG("  finishing detach texture\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.FinishDetachSharedLayerFromLowLevelLayer(hLayer));
                    int iBuf0 = (frame / SubFrameCount) % FrameBufferCount;
                    int iBuf1 = (iBuf0 + 1) % FrameBufferCount;

                    // Detach 済でキャンセルを呼んでも大丈夫
                    {
                        sharedLayerWindow.CancelPreAcquiredTexture();
                        for(int k = 0; k < FrameBufferCount; k++)
                        {
                            sharedLayerWindow.CancelTexture(k);
                        }
                    }

                    NN_LOG("Attaching (iBuf=%d,%d)\n", iBuf0, iBuf1);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, iBuf0, iBuf1));

                    // Acquire 前にキャンセルを呼んでも大丈夫
                    {
                        sharedLayerWindow.CancelPreAcquiredTexture();
                        for(int k = 0; k < FrameBufferCount; k++)
                        {
                            sharedLayerWindow.CancelTexture(k);
                        }
                    }

                    NN_LOG("Acquiring\n");
                    while(sharedLayerWindow.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(16)).IsFailure())
                    {
                        NN_LOG("waiting for pre-acquire texture\n");
                    }
                    EXPECT_FALSE(sharedLayerWindow.IsTextureAcquired());
                    EXPECT_TRUE(sharedLayerWindow.IsTexturePreAcquired());
                    // 一度成功したら繰り返し呼んでも成功を返す
                    NNT_VI_EXPECT_SUCCESS(sharedLayerWindow.PreAcquireTexture(nn::TimeSpan()));
                    sharedLayerWindow.AcquirePreAcquiredTexture(&acqSync, &index, &pTargetTexture);
                    EXPECT_TRUE(sharedLayerWindow.IsTextureAcquired());
                    EXPECT_FALSE(sharedLayerWindow.IsTexturePreAcquired());
                    NN_LOG("  Frame %d / index = %d\n", frame, index);
                }


                nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, sizeof(g_CommandStorage));
                nvnCommandBufferAddControlMemory(&commandBuffer, g_ControlStorage, sizeof(g_ControlStorage));
                nvnCommandBufferBeginRecording(&commandBuffer);

                // wait display
                nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

                nnt::vi::scene::DrawRotationBoxScene(&commandBuffer, pTargetTexture, &texView, frame / 60.f);

                auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);

                NN_LOG("submitting\n");
                nvnQueueSubmitCommands(&queue, 1, &hCommand);
                nvnQueueFinish(&queue);

                NN_LOG("presenting\n");
                EXPECT_TRUE(sharedLayerWindow.IsTextureAcquired());
                EXPECT_FALSE(sharedLayerWindow.IsTexturePreAcquired());
                sharedLayerWindow.PresentTexture(&queue, index);
                EXPECT_FALSE(sharedLayerWindow.IsTextureAcquired());
                EXPECT_FALSE(sharedLayerWindow.IsTexturePreAcquired());
            }

            context0.SafeDetachSharedLayerFromLowLevelLayer(hLayer);
            sharedLayerWindow.Finalize();
        }

        context0.DisconnectService();
        context1.DisconnectService();
        context0.CleanSharedBuffer(SlotIndex);
    }
}// NOLINT(impl/function_size)

NNT_VI_TEST_SHIM(SharedLayerWindowImpl_PreCanceling)
{
    static const int LoopCount = DefaultLoopCount;
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;
    static const int FrameCount = DefaultFrameCount;
    NN_LOG("SharedLayerWindow>PreCanceling-------------------------------\n");
    static const int SubFrameCount = 10;

    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);

        ContextExt context0({"server"});
        ContextExt context1({"client"});

        context0.ConnectService();
        context1.ConnectService();

        // setup SharedLowLevelLayer
        auto layerId = context0.GetDefaultLayerId();
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

        // setup SharedBuffer
        auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

        // setup SharedLayer
        nn::vi::fbshare::SharedLayerHandle hLayer = {};
        nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

        nn::os::SystemEventType detachReadyEvent = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetSharedLayerDetachReadyEvent(&detachReadyEvent, hLayer));
        NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&detachReadyEvent); };

        // attach
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));

        // export SharedBuffer to me
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.RegisterSharedBufferImporterAruid(hBuffer, nn::applet::GetAppletResourceUserId()));

        {
            // init nvn and device
            NNT_VI_SCOPED_DEVICE           (device);
            NNT_VI_SCOPED_COMMANDMEMORYPOOL(commandPool, device, g_CommandStorage, sizeof(g_CommandStorage));
            NNT_VI_SCOPED_QUEUE            (queue, device);
            NNT_VI_SCOPED_COMMANDBUFFER    (commandBuffer, device);
            NNT_VI_SCOPED_FRAMETEXTUREVIEW (texView);

            nn::vi::fbshare::SharedLayerWindowImpl sharedLayerWindow;
            sharedLayerWindow.Initialize(context1.GetSystemService(), &device, hBuffer, hLayer, 2, NVN_FORMAT_RGBA8_SRGB);

            for(int frame = 0; frame < FrameCount; frame++)
            {
                NNT_VI_SCOPED_SYNC(acqSync, device);
                int index = -1;
                NVNtexture* pTargetTexture = nullptr;

                NN_LOG("PreAcquiring\n");
                while(sharedLayerWindow.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(1)).IsFailure())
                {
                    NN_LOG("  retrying...\n");
                }

                if(frame % SubFrameCount == 0)
                {
                    // Acquire してないテクスチャに対してキャンセルを呼んでも大丈夫
                    {
                        for(int k = 0; k < FrameBufferCount; k++)
                        {
                            sharedLayerWindow.CancelTexture(k);
                        }
                    }

                    NN_LOG("Detaching\n");
                    NN_LOG("  starting detach texture\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.StartDetachSharedLayerFromLowLevelLayer(hLayer));
                    EXPECT_FALSE(nn::os::TryWaitSystemEvent(&detachReadyEvent));

                    // Acquire してないテクスチャに対してキャンセルを呼んでも大丈夫
                    {
                        for(int k = 0; k < FrameBufferCount; k++)
                        {
                            sharedLayerWindow.CancelTexture(k);
                        }
                    }

                    NN_LOG("  canceling pre-acquired texture\n");
                    sharedLayerWindow.CancelPreAcquiredTexture();
                    // 重複 ok
                    sharedLayerWindow.CancelPreAcquiredTexture();
                    NN_LOG("  waiting for detach ready\n");
                    nn::os::WaitSystemEvent(&detachReadyEvent);
                    NN_LOG("  finishing detach texture\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.FinishDetachSharedLayerFromLowLevelLayer(hLayer));
                    int iBuf0 = (frame / SubFrameCount) % FrameBufferCount;
                    int iBuf1 = (iBuf0 + 1) % FrameBufferCount;

                    // Detach 済でキャンセルを呼んでも大丈夫
                    {
                        sharedLayerWindow.CancelPreAcquiredTexture();
                        for(int k = 0; k < FrameBufferCount; k++)
                        {
                            sharedLayerWindow.CancelTexture(k);
                        }
                    }

                    NN_LOG("Attaching (iBuf=%d,%d)\n", iBuf0, iBuf1);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, iBuf0, iBuf1));

                    // Acquire 前にキャンセルを呼んでも大丈夫
                    {
                        sharedLayerWindow.CancelPreAcquiredTexture();
                        for(int k = 0; k < FrameBufferCount; k++)
                        {
                            sharedLayerWindow.CancelTexture(k);
                        }
                    }

                    NN_LOG("Acquiring\n");
                    while(sharedLayerWindow.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(1)).IsFailure())
                    {
                        // 2 枚しかアタッチしていないのでディスプレイ待ちが発生する
                        // （といっても vi の処理が進むのを待つだけで display release fence は待たない）
                        NN_LOG("  retrying...\n");
                    }
                }
                while(sharedLayerWindow.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(16)).IsFailure())
                {
                    NN_LOG("waiting for pre-acquire texture\n");
                }
                sharedLayerWindow.AcquirePreAcquiredTexture(&acqSync, &index, &pTargetTexture);
                NN_LOG("  Frame %d / index = %d\n", frame, index);


                nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, sizeof(g_CommandStorage));
                nvnCommandBufferAddControlMemory(&commandBuffer, g_ControlStorage, sizeof(g_ControlStorage));
                nvnCommandBufferBeginRecording(&commandBuffer);

                // wait display
                nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

                nnt::vi::scene::DrawRotationBoxScene(&commandBuffer, pTargetTexture, &texView, frame / 60.f);

                auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);

                NN_LOG("submitting\n");
                nvnQueueSubmitCommands(&queue, 1, &hCommand);
                nvnQueueFinish(&queue);

                NN_LOG("presenting\n");
                sharedLayerWindow.PresentTexture(&queue, index);
            }

            context0.SafeDetachSharedLayerFromLowLevelLayer(hLayer);
            sharedLayerWindow.Finalize();
        }

        context0.DisconnectService();
        context1.DisconnectService();
        context0.CleanSharedBuffer(SlotIndex);
    }
}// NOLINT(impl/function_size)
