﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <memory>
#include <random>
#include "testVi_Context.h"
#include "testVi_Macro.h"
#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/vi/fbshare/vi_SharedLayerWindowImpl.h>

#include "testVi_NvnUtility.h"
#include "testVi_SceneUtility.h"

class TestIndirectLayerState
{
public:
    static const int ConsumerThreadCoreNumber = 1;
    static const int ProducerThreadCoreNumber = 2;
    static const int SharedBufferSlotIndex = 0;
    static const int SharedBufferFrameBufferCount = 4;
public:
    NN_ALIGNAS(4096) char m_ConsumerThreadStack[16 * 4096];
    NN_ALIGNAS(4096) char m_ProducerThreadStack[16 * 4096];

    NN_ALIGNAS(4096) char m_ConsumerCommandStorage[100 * 4096];
    NN_ALIGNAS(4096) char m_ConsumerControlStorage[4096];
    NN_ALIGNAS(4096) char m_ProducerCommandStorage[100 * 4096];
    NN_ALIGNAS(4096) char m_ProducerControlStorage[4096];

    nn::os::EventType m_ExitEvent;
    nn::os::MutexType m_Mutex;

    nn::os::ThreadType m_ConsumerThread;
    nn::os::ThreadType m_ProducerThread;

    ContextExt m_Context;
    ContextExt m_ContextConsumer;
    ContextExt m_ContextProducer;

    nn::vi::IndirectLayerHandleType m_hIndirectLayer;
    nn::vi::IndirectConsumerHandleType m_hIndirectConsumer;
    nn::vi::IndirectProducerHandleType m_hIndirectProducer;

    struct {
        NVNdevice* m_pDevice;
        nn::vi::LayerId m_LowLayerId;
        nn::vi::fbshare::SharedBufferHandle m_hBuffer;
        nn::vi::fbshare::SharedLayerHandle  m_hLayer;
        int m_BufferIndex0;
        int m_BufferIndex1;
        void* m_pCommandMemory;
        size_t m_CommandSize;
        void* m_pControlMemory;
        size_t m_ControlSize;
        nn::vi::IndirectConsumerHandleType m_hConsumer;
    } m_ConsumerParam;

    struct {
        NVNdevice* m_pDevice;
        nn::vi::LayerId m_LowLayerId;
        nn::vi::fbshare::SharedBufferHandle m_hBuffer;
        nn::vi::fbshare::SharedLayerHandle  m_hLayer;
        int m_BufferIndex720_0;
        int m_BufferIndex720_1;
        int m_BufferIndex1080_0;
        int m_BufferIndex1080_1;
        void* m_pCommandMemory;
        size_t m_CommandSize;
        void* m_pControlMemory;
        size_t m_ControlSize;
    } m_ProducerParam;

public:
    TestIndirectLayerState() NN_NOEXCEPT
        : m_Context({"manager"})
        , m_ContextConsumer({"cnsmr"})
        , m_ContextProducer({"prdsr"})
    {
        m_Context.ConnectService();
        m_ContextConsumer.ConnectService();
        m_ContextProducer.ConnectService();
    }

    ~TestIndirectLayerState() NN_NOEXCEPT
    {
        m_Context.DisconnectService();
        m_ContextConsumer.DisconnectService();
        m_ContextProducer.DisconnectService();
    }


    void Run() NN_NOEXCEPT
    {
        nn::os::InitializeEvent(&m_ExitEvent, false, nn::os::EventClearMode_ManualClear);
        NN_UTIL_SCOPE_EXIT{ nn::os::FinalizeEvent(&m_ExitEvent); };

        nn::os::InitializeMutex(&m_Mutex, false, 0);
        NN_UTIL_SCOPE_EXIT{ nn::os::FinalizeMutex(&m_Mutex); };

        auto& context = m_Context;

        // setup SharedBuffer
        auto hBuffer = context.CreateSharedBufferEx720p1080p(SharedBufferSlotIndex);
        // export SharedBuffer to me
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.RegisterSharedBufferImporterAruid(hBuffer, nn::applet::GetAppletResourceUserId()));

        // setup IndirectLayer
        {
            nn::vi::IndirectLayerHandleType hIndirect = {};
            nn::vi::IndirectConsumerHandleType hConsumer = {};
            nn::vi::IndirectProducerHandleType hProducer = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetManagerService()->CreateIndirectLayer(&hIndirect));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetManagerService()->CreateIndirectConsumerEndPoint(&hConsumer, hIndirect, nn::applet::GetAppletResourceUserId()));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetManagerService()->CreateIndirectProducerEndPoint(&hProducer, hIndirect, nn::applet::GetAppletResourceUserId()));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetManagerService()->SetIndirectProducerFlipOffset(hIndirect, hProducer, nn::TimeSpan::FromMicroSeconds(5000)));
            m_hIndirectLayer = hIndirect;
            m_hIndirectConsumer = hConsumer;
            m_hIndirectProducer = hProducer;
        }

        NNT_VI_SCOPED_DEVICE(device);

        // setup ConsumerParameter
        {
            m_ConsumerParam.m_pDevice = &device;
            // setup SharedLowLevelLayer
            auto layerId = context.GetDefaultLayerId();
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.BindLowLevelLayerToManagedLayer(layerId));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));
            // setup SharedLayer
            nn::vi::fbshare::SharedLayerHandle hLayer = {};
            nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.SetSharedLayerLayerStack(hLayer, stacks));

            m_ConsumerParam.m_LowLayerId = layerId;
            m_ConsumerParam.m_hBuffer = hBuffer;
            m_ConsumerParam.m_hLayer  = hLayer;
            m_ConsumerParam.m_pCommandMemory = m_ConsumerCommandStorage;
            m_ConsumerParam.m_CommandSize    = sizeof(m_ConsumerCommandStorage);
            m_ConsumerParam.m_pControlMemory = m_ConsumerControlStorage;
            m_ConsumerParam.m_ControlSize    = sizeof(m_ConsumerControlStorage);
            m_ConsumerParam.m_hConsumer      = m_hIndirectConsumer;
            // 720p
            m_ConsumerParam.m_BufferIndex0 = 0;
            m_ConsumerParam.m_BufferIndex1 = 1;
        }

        // setup ProducerParameter
        {
            m_ProducerParam.m_pDevice = &device;
            // setup SharedLowLevelLayer
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetManagerService()->BindSharedLowLevelLayerToIndirectLayer(m_hIndirectProducer, nn::applet::GetAppletResourceUserId()));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.ConnectLowLevelLayerToSharedBuffer(m_hIndirectProducer, hBuffer));
            // setup SharedLayer
            nn::vi::fbshare::SharedLayerHandle hLayer = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
            m_ProducerParam.m_LowLayerId = m_hIndirectProducer;
            m_ProducerParam.m_hBuffer = hBuffer;
            m_ProducerParam.m_hLayer  = hLayer;
            m_ProducerParam.m_pCommandMemory = m_ProducerCommandStorage;
            m_ProducerParam.m_CommandSize    = sizeof(m_ProducerCommandStorage);
            m_ProducerParam.m_pControlMemory = m_ProducerControlStorage;
            m_ProducerParam.m_ControlSize    = sizeof(m_ProducerControlStorage);
            {
                // 720p
                m_ProducerParam.m_BufferIndex720_0 = 2;
                m_ProducerParam.m_BufferIndex720_1 = 3;
            }
            {
                // 1080p
                m_ProducerParam.m_BufferIndex1080_0 = 4;
                m_ProducerParam.m_BufferIndex1080_1 = 5;
            }

        }

        NN_LOG("[m]start threads\n");
        // create threads
        nn::os::CreateThread(&m_ProducerThread, ProducerThreadFunction, this, m_ProducerThreadStack, sizeof(m_ProducerThreadStack), nn::os::DefaultThreadPriority, ProducerThreadCoreNumber);
        nn::os::CreateThread(&m_ConsumerThread, ConsumerThreadFunction, this, m_ConsumerThreadStack, sizeof(m_ConsumerThreadStack), nn::os::DefaultThreadPriority, ConsumerThreadCoreNumber);
        nn::os::StartThread(&m_ProducerThread);
        nn::os::StartThread(&m_ConsumerThread);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        // ぱたぱたアタッチ・デタッチする
        if(false)
        {
            NN_LOG("[m]random attach/detach consumer&producer\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.AttachSharedLayerToLowLevelLayer(m_ConsumerParam.m_hLayer, m_ConsumerParam.m_LowLayerId, m_ConsumerParam.m_BufferIndex0, m_ConsumerParam.m_BufferIndex1));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.AttachSharedLayerToLowLevelLayer(m_ProducerParam.m_hLayer, m_ProducerParam.m_LowLayerId, m_ProducerParam.m_BufferIndex720_0, m_ProducerParam.m_BufferIndex720_1));
            std::mt19937 rand(0x9992222);
            std::uniform_int_distribution<int> dist;
            int isAttached[2] = {1, 1}; // 0: consumer, 1: producer
            nn::vi::LayerId lowId[2] = { m_ConsumerParam.m_LowLayerId, m_ProducerParam.m_LowLayerId };
            nn::vi::fbshare::SharedLayerHandle hLayer[2] = { m_ConsumerParam.m_hLayer, m_ProducerParam.m_hLayer };
            int bufferIndex[][2] = {
                { m_ConsumerParam.m_BufferIndex0, m_ConsumerParam.m_BufferIndex1 },
                { m_ProducerParam.m_BufferIndex720_0, m_ProducerParam.m_BufferIndex720_1 },
            };
            for(int i = 0; i < 100; i++)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(dist(rand) % 1000000));
                NN_LOG("[m]change attach state %d\n", i);
                int target = dist(rand) % 2;
                if(isAttached[target])
                {
                    context.SafeDetachSharedLayerFromLowLevelLayer(hLayer[target]);
                    isAttached[target] = 0;
                }
                else
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(context.AttachSharedLayerToLowLevelLayer(hLayer[target], lowId[target], bufferIndex[target][0], bufferIndex[target][1]));
                    isAttached[target] = 1;
                }
            }
        }
        if(true)
        {
            NN_LOG("[m]random attach/detach producer 720p/1080p\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.AttachSharedLayerToLowLevelLayer(m_ConsumerParam.m_hLayer, m_ConsumerParam.m_LowLayerId, m_ConsumerParam.m_BufferIndex0, m_ConsumerParam.m_BufferIndex1));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.AttachSharedLayerToLowLevelLayer(m_ProducerParam.m_hLayer, m_ProducerParam.m_LowLayerId, m_ProducerParam.m_BufferIndex720_0, m_ProducerParam.m_BufferIndex720_1));
            auto lowId = m_ProducerParam.m_LowLayerId;
            auto hLayer = m_ProducerParam.m_hLayer;
            int bufferIndex[][2] = {
                { m_ProducerParam.m_BufferIndex720_0, m_ProducerParam.m_BufferIndex720_1 },
                { m_ProducerParam.m_BufferIndex1080_0, m_ProducerParam.m_BufferIndex1080_1 },
            };
            int target = 0;
            for(int i = 0; i < 10; i++)
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                NN_LOG("[m]change attach resolution %d\n", i);
                context.SafeDetachSharedLayerFromLowLevelLayer(hLayer);
                // blank
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.DisconnectLowLevelLayerFromSharedBuffer(lowId));
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.ConnectLowLevelLayerToSharedBuffer(lowId, m_ProducerParam.m_hBuffer));
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                target += 1;
                target %= 2;
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.AttachSharedLayerToLowLevelLayer(hLayer, lowId, bufferIndex[target][0], bufferIndex[target][1]));
            }
        }
        if(false)
        {
            NN_LOG("[m]free run\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.AttachSharedLayerToLowLevelLayer(m_ConsumerParam.m_hLayer, m_ConsumerParam.m_LowLayerId, m_ConsumerParam.m_BufferIndex0, m_ConsumerParam.m_BufferIndex1));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.AttachSharedLayerToLowLevelLayer(m_ProducerParam.m_hLayer, m_ProducerParam.m_LowLayerId, m_ProducerParam.m_BufferIndex720_0, m_ProducerParam.m_BufferIndex720_1));
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
        }

        NN_LOG("[m]stop threads\n");
        nn::os::SignalEvent(&m_ExitEvent);

        nn::os::WaitThread(&m_ProducerThread);
        nn::os::WaitThread(&m_ConsumerThread);
        nn::os::DestroyThread(&m_ProducerThread);
        nn::os::DestroyThread(&m_ConsumerThread);

        context.DestroySharedBuffer(SharedBufferSlotIndex, hBuffer);
    }// NOLINT(impl/function_size)

    static void ProducerThreadFunction(void* arg) NN_NOEXCEPT
    {
        auto pSelf = reinterpret_cast<TestIndirectLayerState*>(arg);
        auto& param = pSelf->m_ProducerParam;
        NN_LOG("[p]start\n");

        auto hBuffer = param.m_hBuffer;
        auto hLayer  = param.m_hLayer;

        auto& context = pSelf->m_ContextProducer;
        auto& device = *param.m_pDevice;
        NNT_VI_SCOPED_COMMANDMEMORYPOOL(commandPool, device, param.m_pCommandMemory, param.m_CommandSize);
        NNT_VI_SCOPED_QUEUE            (queue, device);
        NNT_VI_SCOPED_COMMANDBUFFER    (commandBuffer, device);
        NNT_VI_SCOPED_FRAMETEXTUREVIEW (texView);

        nn::vi::fbshare::SharedLayerWindowImpl window;
        NN_ABORT_UNLESS_RESULT_SUCCESS(window.Initialize(context.GetSystemService(), &device, hBuffer, hLayer, 2, NVN_FORMAT_RGBA8_SRGB));
        NN_UTIL_SCOPE_EXIT{ window.Finalize(); };

        for(int frame = 0;; frame++)
        {
            if(nn::os::TryWaitEvent(&pSelf->m_ExitEvent))
            {
                break;
            }

            NNT_VI_SCOPED_SYNC(acqSync, device);
            int index = -1;
            NVNtexture* pTargetTexture = nullptr;
            if(window.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(16)).IsFailure())
            {
                NN_LOG("[p] pre-acquire texture failed\n");
            }
            window.AcquirePreAcquiredTexture(&acqSync, &index, &pTargetTexture);
            NN_LOG("[p] frame %d\n", frame);

            nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, param.m_CommandSize);
            nvnCommandBufferAddControlMemory(&commandBuffer, param.m_pControlMemory, param.m_ControlSize);
            nvnCommandBufferBeginRecording(&commandBuffer);

            nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

            if(pTargetTexture)
            {
                nnt::vi::scene::DrawRotationBoxScene(&commandBuffer, pTargetTexture, &texView, frame / 60.f);

                // フレーム数
                {
                    nnt::vi::scene::DrawStringParameter strParam;
                    strParam.posX  = 0;
                    strParam.posY  = 0;
                    strParam.size  = 16;
                    strParam.color = nn::util::Color4f(1, 1, 1, 1);
                    char str[256] = {};
                    nn::util::SNPrintf(str, sizeof(str), "%d", frame);
                    nnt::vi::scene::DrawString(&commandBuffer, pTargetTexture, &texView, str, strParam);
                }

                // サイズ
                {
                    const char str[] = "1280x720";
                    nnt::vi::scene::DrawStringParameter strParam;
                    strParam.posX  = 1280 - 16 * std::strlen(str);
                    strParam.posY  = 720 - 16;
                    strParam.size  = 16;
                    strParam.color = nn::util::Color4f(1, 1, 1, 1);
                    nnt::vi::scene::DrawString(&commandBuffer, pTargetTexture, &texView, str, strParam);
                }

                // サイズ
                {
                    const char str[] = "1920x1080";
                    nnt::vi::scene::DrawStringParameter strParam;
                    strParam.posX  = 1920 - 16 * std::strlen(str);
                    strParam.posY  = 1080 - 16;
                    strParam.size  = 16;
                    strParam.color = nn::util::Color4f(1, 1, 1, 1);
                    nnt::vi::scene::DrawString(&commandBuffer, pTargetTexture, &texView, str, strParam);
                }
            }

            NN_LOG("command free = %llu\n", nvnCommandBufferGetCommandMemoryFree(&commandBuffer));
            auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);

            nvnQueueSubmitCommands(&queue, 1, &hCommand);
            window.PresentTexture(&queue, index);
            nvnQueueFinish(&queue);
        }
        NN_LOG("[p]exit\n");
    }

    static void ConsumerThreadFunction(void* arg) NN_NOEXCEPT
    {
        const size_t ImagePoolSize = 4 * 1024 * 1024;

        auto pSelf = reinterpret_cast<TestIndirectLayerState*>(arg);
        auto& param = pSelf->m_ConsumerParam;
        NN_LOG("[c]start\n");
        auto hBuffer = param.m_hBuffer;
        auto hLayer  = param.m_hLayer;

        auto& context = pSelf->m_ContextConsumer;
        auto& device = *param.m_pDevice;
        NNT_VI_SCOPED_COMMANDMEMORYPOOL(commandPool, device, param.m_pCommandMemory, param.m_CommandSize);
        NNT_VI_SCOPED_QUEUE            (queue, device);
        NNT_VI_SCOPED_COMMANDBUFFER    (commandBuffer, device);
        NNT_VI_SCOPED_FRAMETEXTUREVIEW (texView);

        // indirect consumer 用バッファ
        void* pImagePoolMemory = nullptr;
        NVNmemoryPool imagePool;
        {
            pImagePoolMemory = aligned_alloc(4096, ImagePoolSize);
            NN_ABORT_UNLESS_NOT_NULL(pImagePoolMemory);
            int flag = NVN_MEMORY_POOL_FLAGS_CPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT;
            NVNmemoryPoolBuilder builder;
            nvnMemoryPoolBuilderSetDevice(&builder, &device);
            nvnMemoryPoolBuilderSetDefaults(&builder);
            nvnMemoryPoolBuilderSetStorage(&builder, pImagePoolMemory, ImagePoolSize);
            nvnMemoryPoolBuilderSetFlags(&builder, flag);
            NN_ABORT_UNLESS(nvnMemoryPoolInitialize(&imagePool, &builder));
        }
        NN_UTIL_SCOPE_EXIT{
            nvnMemoryPoolFinalize(&imagePool);
            free(pImagePoolMemory);
        };

        NVNbuffer imageBuffer;
        {
            NVNbufferBuilder builder;
            nvnBufferBuilderSetDevice(&builder, &device);
            nvnBufferBuilderSetDefaults(&builder);
            nvnBufferBuilderSetStorage(&builder, &imagePool, 0, ImagePoolSize);
            NN_ABORT_UNLESS(nvnBufferInitialize(&imageBuffer, &builder));
        }
        NN_UTIL_SCOPE_EXIT{ nvnBufferFinalize(&imageBuffer); };

        nn::vi::fbshare::SharedLayerWindowImpl window;
        NN_ABORT_UNLESS_RESULT_SUCCESS(window.Initialize(context.GetSystemService(), &device, hBuffer, hLayer, 2, NVN_FORMAT_RGBA8_SRGB));
        NN_UTIL_SCOPE_EXIT{ window.Finalize(); };

        for(int frame = 0;; frame++)
        {
            if(nn::os::TryWaitEvent(&pSelf->m_ExitEvent))
            {
                break;
            }

            NNT_VI_SCOPED_SYNC(acqSync, device);
            int index = -1;
            NVNtexture* pTargetTexture = nullptr;
            if(window.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(16)).IsFailure())
            {
                NN_LOG("[c] pre-acquire texutre failed\n");
            }
            window.AcquirePreAcquiredTexture(&acqSync, &index, &pTargetTexture);
            NN_LOG("[c] frame %d\n", frame);

            // 画像を読み込み
            int64_t imageDataSize = 0;
            int64_t imageDataStride = 0;
            {
                void* pData = nvnBufferMap(&imageBuffer);
                auto tick0 = nn::os::GetSystemTick();
                auto result = context.GetApplicationService()->GetIndirectLayerImageCropMap(
                    &imageDataSize,
                    &imageDataStride,
                    nn::sf::OutBuffer(reinterpret_cast<char*>(pData), ImagePoolSize),
                    1280,
                    720,
                    0, 0, 1, 1,
                    param.m_hConsumer,
                    nn::applet::GetAppletResourceUserId()
                );
                auto tick1 = nn::os::GetSystemTick();

                if(result.IsFailure())
                {
                    if(nn::vi::ResultNotReady::Includes(result))
                    {
                        NN_LOG("[c]   indirect image not ready (%d-%d)\n", result.GetModule(), result.GetDescription());
                    }
                    else
                    {
                        NN_LOG("[c]   indirect image error (%d-%d)\n", result.GetModule(), result.GetDescription());
                    }
                    imageDataSize = 0;
                    imageDataStride = 0;
                }

                {
                    auto tick = tick1 - tick0;
                    NN_LOG("[c]   indirect image -> %lluus\n", tick.ToTimeSpan().GetMicroSeconds());
                }

                nvnBufferFlushMappedRange(&imageBuffer, 0, imageDataSize);
            }

            nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, param.m_CommandSize);
            nvnCommandBufferAddControlMemory(&commandBuffer, param.m_pControlMemory, param.m_ControlSize);
            nvnCommandBufferBeginRecording(&commandBuffer);

            nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

            if(pTargetTexture)
            {
                if(imageDataSize > 0)
                {
                    nnt::vi::scene::CopyBufferToTexture(&commandBuffer, pTargetTexture, &imageBuffer, 0, imageDataSize, nnt::vi::scene::CopyBufferToTextureParameter::GetPreset());
                }
                else
                {
                    nnt::vi::scene::ClearTexture(&commandBuffer, pTargetTexture, nn::util::Color4f(.5f, .5f, .5f, 1));
                }

                {
                    nnt::vi::scene::DrawStringParameter strParam;
                    strParam.posX  = 0;
                    strParam.posY  = 32;
                    strParam.size  = 16;
                    strParam.color = nn::util::Color4f(1, 1, 1, 1);
                    char str[256] = {};
                    nn::util::SNPrintf(str, sizeof(str), "%d", frame);
                    nnt::vi::scene::DrawString(&commandBuffer, pTargetTexture, &texView, str, strParam);
                }
            }

            auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);

            nvnQueueSubmitCommands(&queue, 1, &hCommand);
            window.PresentTexture(&queue, index);
            nvnQueueFinish(&queue);
        }
        NN_LOG("[c]exit\n");

    }// NOLINT(impl/function_size)
};

NNT_VI_TEST_INTEGRATED(IndirectLayer_Basic)
{
    auto p = std::make_shared<TestIndirectLayerState>();
    p->Run();
}
