﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include "testVi_Window.h"

// Please refer to the NvnTutorials for more details about the NVN functions.
Window::Window(NVNdevice* pDevice, nn::vi::Layer* pLayer, int width, int height, Color backgroundColor) NN_NOEXCEPT
    : m_Window(new NVNwindow)
    , m_Textures{nullptr, nullptr}
    , m_TextureWidth(width)
    , m_TextureHeight(height)
    , m_BackgroundColor(backgroundColor)
{
    InitializeTextureBuilder(pDevice);
    // Note:  This is excessive padding, but it will cover most dynamic sizes
    m_Pool.Initialize(pDevice, nvnTextureBuilderGetStorageSize(&m_TextureBuilder) * (TextureCount + 1));
    InitializeTextures();

    NVNwindowBuilder builder;
    nvnWindowBuilderSetDefaults(&builder);

    nvnWindowBuilderSetDevice(&builder, pDevice);
    nvnWindowBuilderSetTextures(&builder, TextureCount, m_Textures);

    nn::vi::NativeWindowHandle handle;
    nn::Result result = nn::vi::GetNativeWindow(&handle, pLayer);
    NN_ASSERT(result.IsSuccess());
    nvnWindowBuilderSetNativeWindow(&builder, handle);

    nvnWindowInitialize(m_Window, &builder);
}

Window::~Window() NN_NOEXCEPT
{
    for( int i = 0; i < TextureCount; ++i )
    {
        if( m_Textures[i] != nullptr )
        {
            nvnTextureFinalize(m_Textures[i]);
            delete m_Textures[i];
        }
    }

    if( m_Window != nullptr )
    {
        nvnWindowFinalize(m_Window);
        delete m_Window;
    }
}

void Window::InitializeTextureBuilder(NVNdevice* pDevice) NN_NOEXCEPT
{
    nvnTextureBuilderSetDefaults(&m_TextureBuilder);
    nvnTextureBuilderSetDevice(&m_TextureBuilder, pDevice);
    nvnTextureBuilderSetSize2D(&m_TextureBuilder, m_TextureWidth, m_TextureHeight);
    nvnTextureBuilderSetTarget(&m_TextureBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_TextureBuilder, NVN_FORMAT_RGBA8);
}

void Window::InitializeTextures() NN_NOEXCEPT
{
    size_t size = nvnTextureBuilderGetStorageSize(&m_TextureBuilder);
    size_t alignment = nvnTextureBuilderGetStorageAlignment(&m_TextureBuilder);

    size_t offset = m_Pool.GetAlignedOffset(size, alignment, 0);
    for( int i = 0; i < TextureCount; ++i )
    {
        m_Textures[i] = new NVNtexture;
        nvnTextureBuilderSetStorage(&m_TextureBuilder, m_Pool.GetPool(), static_cast<ptrdiff_t>(offset));
        nvnTextureInitialize(m_Textures[i], &m_TextureBuilder);
        offset = m_Pool.GetAlignedOffset(size, alignment, offset + size);
    }
}

NVNwindow* Window::GetWindow() const NN_NOEXCEPT
{
    return m_Window;
}

NVNtexture* Window::GetTexture(int i) const NN_NOEXCEPT
{
    return m_Textures[i];
}

const float* Window::GetBackgroundColorChannels() const NN_NOEXCEPT
{
    return m_BackgroundColor.GetChannels();
}

int Window::GetTextureWidth() const NN_NOEXCEPT
{
    return m_TextureWidth;
}

int Window::GetTextureHeight() const NN_NOEXCEPT
{
    return m_TextureHeight;
}
