﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/edidUtil/testEdid_FullModeInfo.h>
#include <nn/edid/edid_DisplayModeInfo.h>
#include <nn/edid/edid_DisplayTimingInfo.h>
#include <nn/edid/edid_ImageSizeInfo.h>
#include <detail/edid_DisplayModeInfo.h>

bool CopyDisplayMode(const nn::edid::DisplayModeInfo* pMode,
                     const nn::edid::DisplayTimingInfo* pTimingInfo,
                     const nn::edid::ImageSizeInfo* pSizeInfo,
                     void* pUserData) NN_NOEXCEPT
{
    nnt::edid::FullModeInfo* pFull = static_cast<nnt::edid::FullModeInfo*>(pUserData);

    pFull->pMode = new nn::edid::DisplayModeInfo(*pMode);

    if( pTimingInfo != nullptr )
    {
        pFull->pTimingInfo = new nn::edid::DisplayTimingInfo(*pTimingInfo);
    }

    if( pSizeInfo != nullptr )
    {
        pFull->pSizeInfo = new nn::edid::ImageSizeInfo(*pSizeInfo);
    }

    return false;
}

TEST(DisplayModeInfo, VisitEstablishedTiming)
{
    nnt::edid::FullModeInfo mode;

    const int Width = 1920;
    const int Height = 1080;
    const float RefreshRate = 60.00f;
    const bool IsInterlaced = true;

    nn::edid::detail::VisitEstablishedTiming(CopyDisplayMode, &mode, Width, Height, RefreshRate, IsInterlaced);

    EXPECT_EQ(Width, mode.pMode->width);
    EXPECT_EQ(Height, mode.pMode->height);
    EXPECT_EQ(RefreshRate, mode.pMode->refreshRate);
    EXPECT_EQ(static_cast<float>(Width) / Height, mode.pMode->imageAspectRatio);
    EXPECT_EQ(1.f, mode.pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, mode.pMode->stereoMode);
    EXPECT_EQ(IsInterlaced, mode.pMode->isInterlaced);
    EXPECT_EQ(nullptr, mode.pTimingInfo);
    EXPECT_EQ(nullptr, mode.pSizeInfo);
}

void TestEstablishedTiming(std::uint8_t timings,
                           void (*parser)(nn::edid::DisplayModeVisitor,
                                          void* pUserData,
                                          std::uint8_t timings),
                           int width,
                           int height,
                           float refreshRate,
                           bool isInterlaced)
{
    nnt::edid::FullModeInfo mode;
    parser(CopyDisplayMode, &mode, timings);

    EXPECT_EQ(width, mode.pMode->width);
    EXPECT_EQ(height, mode.pMode->height);
    EXPECT_EQ(refreshRate, mode.pMode->refreshRate);
    EXPECT_EQ(static_cast<float>(width) / height, mode.pMode->imageAspectRatio);
    EXPECT_EQ(1.f, mode.pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, mode.pMode->stereoMode);
    EXPECT_EQ(isInterlaced, mode.pMode->isInterlaced);
    EXPECT_EQ(nullptr, mode.pTimingInfo);
    EXPECT_EQ(nullptr, mode.pSizeInfo);
}

TEST(DisplayModeInfo, GetEstablishedTimingI)
{
    // Check Table 3.18 in EDID spec
    TestEstablishedTiming(0x01, nn::edid::detail::GetEstablishedTimingI, 800, 600, 60.00f, false);
    TestEstablishedTiming(0x02, nn::edid::detail::GetEstablishedTimingI, 800, 600, 56.00f, false);
    TestEstablishedTiming(0x04, nn::edid::detail::GetEstablishedTimingI, 640, 480, 75.00f, false);
    TestEstablishedTiming(0x08, nn::edid::detail::GetEstablishedTimingI, 640, 480, 72.00f, false);
    TestEstablishedTiming(0x10, nn::edid::detail::GetEstablishedTimingI, 640, 480, 67.00f, false);
    TestEstablishedTiming(0x20, nn::edid::detail::GetEstablishedTimingI, 640, 480, 60.00f, false);
    TestEstablishedTiming(0x40, nn::edid::detail::GetEstablishedTimingI, 720, 400, 88.00f, false);
    TestEstablishedTiming(0x80, nn::edid::detail::GetEstablishedTimingI, 720, 400, 70.00f, false);
}

TEST(DisplayModeInfo, GetEstablishedTimingII)
{
    // Check Table 3.18 in EDID spec
    TestEstablishedTiming(0x01, nn::edid::detail::GetEstablishedTimingII, 1280, 1024, 75.00f, false);
    TestEstablishedTiming(0x02, nn::edid::detail::GetEstablishedTimingII, 1024, 768, 75.00f, false);
    TestEstablishedTiming(0x04, nn::edid::detail::GetEstablishedTimingII, 1024, 768, 70.00f, false);
    TestEstablishedTiming(0x08, nn::edid::detail::GetEstablishedTimingII, 1024, 768, 60.00f, false);
    TestEstablishedTiming(0x10, nn::edid::detail::GetEstablishedTimingII, 1024, 768, 87.00f, true);
    TestEstablishedTiming(0x20, nn::edid::detail::GetEstablishedTimingII, 832, 624, 75.00f, false);
    TestEstablishedTiming(0x40, nn::edid::detail::GetEstablishedTimingII, 800, 600, 75.00f, false);
    TestEstablishedTiming(0x80, nn::edid::detail::GetEstablishedTimingII, 800, 600, 72.00f, false);
}

TEST(DisplayModeInfo, GetEstablishedTimingIII)
{
    // Check Table 3.18 in EDID spec
    TestEstablishedTiming(0x80, nn::edid::detail::GetEstablishedTimingIII, 1152, 870, 75.00f, false);
}

TEST(DisplayModeInfo, GetStandardTimingAspectRatio)
{
    // Check Table 3.19 in EDID spec for bit definitions

    // bits 0-5 can be any value
    for( std::uint8_t i = 0; i < 64; ++i )
    {
        int width;
        int height;

        nn::edid::detail::GetStandardTimingAspectRatio(&width, &height, i);
        EXPECT_EQ(16, width);
        EXPECT_EQ(10, height);

        nn::edid::detail::GetStandardTimingAspectRatio(&width, &height, (1 << 6) | i);
        EXPECT_EQ(4, width);
        EXPECT_EQ(3, height);

        nn::edid::detail::GetStandardTimingAspectRatio(&width, &height, (1 << 7) | i);
        EXPECT_EQ(5, width);
        EXPECT_EQ(4, height);

        nn::edid::detail::GetStandardTimingAspectRatio(&width, &height, 0xC0 | i);
        EXPECT_EQ(16, width);
        EXPECT_EQ(9, height);
    }
}

TEST(DisplayModeInfo, GetStandardTiming)
{
    // Check Table 3.19 in EDID spec for bit definitions

    // if first byte is 0, it's reserved
    // expecting it to wrap around for termination
    for( std::uint8_t pixels = 1; pixels != 0; ++pixels )
    {
        // field refresh rate range (first 5 bits)
        for( std::uint8_t refreshRate = 0; refreshRate <= 0x1F; ++refreshRate )
        {
            // aspect ratio has 4 unique values
            for( std::uint8_t k = 0; k < 4; ++k )
            {
                std::uint8_t timing = (k << 6) | refreshRate;

                int aspectRatioWidth;
                int aspectRatioHeight;
                nn::edid::detail::GetStandardTimingAspectRatio(&aspectRatioWidth, &aspectRatioHeight, timing);

                nn::edid::DisplayModeInfo mode;
                nn::edid::detail::GetStandardTiming(&mode, pixels, timing);

                int width = (pixels + 31) * 8;
                EXPECT_EQ(width, mode.width);
                EXPECT_EQ((width * aspectRatioWidth) / aspectRatioHeight, mode.height);
                EXPECT_EQ(refreshRate + 60, mode.refreshRate);
                EXPECT_EQ(static_cast<float>(aspectRatioWidth) / aspectRatioHeight, mode.imageAspectRatio);
                EXPECT_EQ(1.f, mode.pixelAspectRatio);
                EXPECT_EQ(nn::edid::StereoMode_None, mode.stereoMode);
                EXPECT_FALSE(mode.isInterlaced);
            }
        }
    }
}

TEST(DisplayModeInfo, GetDetailedTimingVsyncPolarity)
{
    // Check Table 3.22 in EDID spec for bit definitions

    // bits 5-7 can be any value
    for( std::uint8_t i = 0; i < 8; ++i )
    {
        // bits 0, 1 can be any value
        for( std::uint8_t j = 0; j < 4; ++j )
        {
            std::uint8_t unusedBits = j | (i << 5);

            // must be considered digital separate sync
            EXPECT_EQ(nn::edid::Polarity_Negative, nn::edid::detail::GetDetailedTimingVsyncPolarity(0x18 | unusedBits));
            EXPECT_EQ(nn::edid::Polarity_Positive, nn::edid::detail::GetDetailedTimingVsyncPolarity(0x1C | unusedBits));

            // digital composite sync
            EXPECT_EQ(nn::edid::Polarity_Undefined, nn::edid::detail::GetDetailedTimingVsyncPolarity(0x10 | unusedBits));
        }
    }

    // clear bit 4 indicates analog
    // bits 0-3 can be any value
    for( std::uint8_t i = 0; i <= 0x0F; ++i )
    {
        // bits 5-7 can be any value
        for( std::uint8_t j = 0; j < 8; ++j )
        {
            EXPECT_EQ(nn::edid::Polarity_Undefined, nn::edid::detail::GetDetailedTimingVsyncPolarity((j << 5) | i));
        }
    }
}

TEST(DisplayModeInfo, GetDetailedTimingHsyncPolarity)
{
    // Check Table 3.22 in EDID spec for bit definitions

    // bits 5-7 can be any value
    for( std::uint8_t i = 0; i < 8; ++i )
    {
        // bits 2, 3 can be any value
        for( std::uint8_t j = 0; j < 4; ++j )
        {
            std::uint8_t unusedBits = (j << 2) | (i << 5);
            // must be considered digital sync
            EXPECT_EQ(nn::edid::Polarity_Negative, nn::edid::detail::GetDetailedTimingHsyncPolarity(0x10 | unusedBits));
            EXPECT_EQ(nn::edid::Polarity_Positive, nn::edid::detail::GetDetailedTimingHsyncPolarity(0x12 | unusedBits));

            // bit 0 could be any value
            unusedBits |= 1;
            EXPECT_EQ(nn::edid::Polarity_Negative, nn::edid::detail::GetDetailedTimingHsyncPolarity(0x10 | unusedBits));
            EXPECT_EQ(nn::edid::Polarity_Positive, nn::edid::detail::GetDetailedTimingHsyncPolarity(0x12 | unusedBits));
        }
    }

    // clear bit 4 indicates analog
    // bits 0-3 can be any value
    for( std::uint8_t i = 0; i <= 0x0F; ++i )
    {
        // bits 5-7 can be any value
        for( std::uint8_t j = 0; j < 8; ++j )
        {
            EXPECT_EQ(nn::edid::Polarity_Undefined, nn::edid::detail::GetDetailedTimingHsyncPolarity((j << 5) | i));
        }
    }
}

TEST(DisplayModeInfo, GetDetailedTimingStereoMode)
{
    // Check Table 3.22 in EDID spec for bit definitions

    // bit 7 can be any value
    for( std::uint8_t i = 0; i < 2; ++i )
    {
        // bits 1-4 can be any value
        for( std::uint8_t j = 0; j <= 0x0F; ++j )
        {
            std::uint8_t unusedBits = (i << 7) | (j << 1);
            EXPECT_EQ(nn::edid::StereoMode_None, nn::edid::detail::GetDetailedTimingStereoMode(unusedBits));
            // bit 0 can be any value only for StereoMode_None
            EXPECT_EQ(nn::edid::StereoMode_None, nn::edid::detail::GetDetailedTimingStereoMode(unusedBits | 1));
            EXPECT_EQ(nn::edid::StereoMode_FieldSequentialRight, nn::edid::detail::GetDetailedTimingStereoMode(0x20 | unusedBits));
            EXPECT_EQ(nn::edid::StereoMode_FieldSequentialLeft, nn::edid::detail::GetDetailedTimingStereoMode(0x40 | unusedBits));
            EXPECT_EQ(nn::edid::StereoMode_InterleavedRight, nn::edid::detail::GetDetailedTimingStereoMode(0x21 | unusedBits));
            EXPECT_EQ(nn::edid::StereoMode_InterleavedLeft, nn::edid::detail::GetDetailedTimingStereoMode(0x41 | unusedBits));
            EXPECT_EQ(nn::edid::StereoMode_Interleaved, nn::edid::detail::GetDetailedTimingStereoMode(0x60 | unusedBits));
            EXPECT_EQ(nn::edid::StereoMode_SideBySide, nn::edid::detail::GetDetailedTimingStereoMode(0x61 | unusedBits));
        }
    }
}

TEST(DisplayModeInfo, GetDetailedTimingSyncTypes)
{
    // Check Table 3.22 in EDID spec for bit definitions

    // analog cases
    // bit 0 can be any value
    for( std::uint8_t i = 0; i < 2; ++i )
    {
        // bits 5-7 can be any value
        for( std::uint8_t j = 0; j < 8; ++j )
        {
            // bit 2 can be any value
            for( std::uint8_t k = 0; k < 2; ++k )
            {
                std::uint8_t unusedBits = i | (j << 5) | (k << 2);

                nn::edid::SyncTypeSet sync = nn::edid::detail::GetDetailedTimingSyncTypes(unusedBits);
                EXPECT_FALSE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
                EXPECT_TRUE(sync.Test<nn::edid::SyncType::CompositeSync>());
                EXPECT_FALSE(sync.Test<nn::edid::SyncType::SeparateSync>());
                EXPECT_TRUE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
                EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnRgb>());

                sync = nn::edid::detail::GetDetailedTimingSyncTypes(0x08 | unusedBits);
                EXPECT_TRUE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
                EXPECT_FALSE(sync.Test<nn::edid::SyncType::CompositeSync>());
                EXPECT_FALSE(sync.Test<nn::edid::SyncType::SeparateSync>());
                EXPECT_TRUE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
                EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnRgb>());

                sync = nn::edid::detail::GetDetailedTimingSyncTypes(0x02 | unusedBits);
                EXPECT_FALSE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
                EXPECT_TRUE(sync.Test<nn::edid::SyncType::CompositeSync>());
                EXPECT_FALSE(sync.Test<nn::edid::SyncType::SeparateSync>());
                EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
                EXPECT_TRUE(sync.Test<nn::edid::SyncType::SyncOnRgb>());
            }
        }
    }

    // digital cases
    // bit 0-2 can be any value
    for( std::uint8_t i = 0; i < 8; ++i )
    {
        // bits 5-7 can be any value
        for( std::uint8_t j = 0; j < 8; ++j )
        {
            std::uint8_t unusedBits = i | (j << 5);

            nn::edid::SyncTypeSet sync = nn::edid::detail::GetDetailedTimingSyncTypes(0x10 | unusedBits);
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
            EXPECT_TRUE(sync.Test<nn::edid::SyncType::CompositeSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SeparateSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnRgb>());

            sync = nn::edid::detail::GetDetailedTimingSyncTypes(0x18 | unusedBits);
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::CompositeSync>());
            EXPECT_TRUE(sync.Test<nn::edid::SyncType::SeparateSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnRgb>());
        }
    }
}

TEST(DisplayModeInfo, IsDetailedTimingSerrated)
{
    // Check Table 3.22 in EDID spec for bit definitions

    // analog cases
    // bits 0, 1 can be any value
    for( std::uint8_t i = 0; i < 4; ++i )
    {
        // bits 5-7 can be any value
        for( std::uint8_t j = 0; j < 8; ++j )
        {
            std::uint8_t unusedBits = i | (j << 5);

            EXPECT_FALSE(nn::edid::detail::IsDetailedTimingSerrated(unusedBits));
            EXPECT_TRUE(nn::edid::detail::IsDetailedTimingSerrated(0x04 | unusedBits));

            // note bit 3 can also be any value
            unusedBits |= 0x08;
            EXPECT_FALSE(nn::edid::detail::IsDetailedTimingSerrated(unusedBits));
            EXPECT_TRUE(nn::edid::detail::IsDetailedTimingSerrated(0x04 | unusedBits));
        }
    }

    // digital cases
    // bits 0, 1 can be any value
    for( std::uint8_t i = 0; i < 4; ++i )
    {
        // bits 5-7 can be any value
        for( std::uint8_t j = 0; j < 8; ++j )
        {
            std::uint8_t unusedBits = i | (j << 5);

            EXPECT_FALSE(nn::edid::detail::IsDetailedTimingSerrated(0x10 | unusedBits));
            EXPECT_TRUE(nn::edid::detail::IsDetailedTimingSerrated(0x14 | unusedBits));
        }
    }
}

TEST(DisplayModeInfo, IsDetailedTimingInterlaced)
{
    // Check Table 3.22 in EDID spec for bit definitions

    // only care about bit 7
    for( std::uint8_t i = 0; i < 128; ++i )
    {
        EXPECT_FALSE(nn::edid::detail::IsDetailedTimingInterlaced(i));
        EXPECT_TRUE(nn::edid::detail::IsDetailedTimingInterlaced(0x80 | i));
    }
}

TEST(DisplayModeInfo, GetDetailedTiming)
{
    nn::edid::DisplayModeInfo mode;
    nn::edid::DisplayTimingInfo timing;
    nn::edid::ImageSizeInfo size;

    // first DTD from an ASUS monitor
    std::uint8_t block[] =
    {
        0x02, // pixel clock
        0x3A, // pixel clock
        0x80, // lower 8 bits of horizontal pixels
        0x18, // lower 8 bits for HBLANK
        0x71, // Upper: horizontal pixels, Lower: HBLANK
        0x38, // lower 8 bits of vertical lines
        0x2D, // lower 8 bits of VBLANK
        0x40, // Upper: vertical lines, Lower: VBLANK
        0x58, // lower 8 bits of horizontal front porch
        0x2C, // lower 8 bits of horizontal sync pulse
        0x45, // Upper: vertical front porch, Lower: vertical sync pulse
        0x00, // 2 bits each for front porch and sync pulse (horizontal and vertical)
        0x09, // lower 8 bits of image width
        0x25, // lower 8 bits of image height
        0x21, // Upper: image width, Lower: image height
        0x00, // right border
        0x00, // top border
        0x1E, // misc. features
    };

    nn::edid::detail::GetDetailedTiming(&mode, &timing, &size, block, sizeof(block));

    EXPECT_EQ(1920, mode.width);
    EXPECT_EQ(1080, mode.height);
    EXPECT_EQ(60.00f, mode.refreshRate);
    EXPECT_EQ(16.f / 9.f, mode.imageAspectRatio);
    EXPECT_NEAR(1.f, mode.pixelAspectRatio, .0005f);
    EXPECT_EQ(nn::edid::StereoMode_None, mode.stereoMode);
    EXPECT_FALSE(mode.isInterlaced);

    EXPECT_EQ(148500, timing.pixelClock);
    EXPECT_EQ(1920, timing.hactive);
    EXPECT_EQ(280, timing.hblank);
    EXPECT_EQ(88, timing.horizontalFrontPorch);
    EXPECT_EQ(44, timing.horizontalSyncPulse);
    EXPECT_EQ(1080, timing.vactive);
    EXPECT_EQ(45, timing.vblank);
    EXPECT_EQ(4, timing.verticalFrontPorch);
    EXPECT_EQ(5, timing.verticalSyncPulse);
    EXPECT_FALSE(timing.analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_FALSE(timing.analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_TRUE(timing.analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_FALSE(timing.analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(timing.analogSync.Test<nn::edid::SyncType::SyncOnRgb>());
    EXPECT_EQ(nn::edid::Polarity_Positive, timing.vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Positive, timing.hsyncPolarity);
}
