﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <algorithm>
#include <nnt.h>
#include <nnt/edidUtil/testEdid_FullModeInfo.h>
#include <nn/nn_Assert.h>
#include <nn/edid/edid_DisplayTimingInfo.h>
#include <nn/edid/edid_DisplayModeInfo.h>
#include <cea861/edid_Cea861.h>

struct BlockData
{
    nn::edid::cea861::BlockTag tag;
    size_t size;
};

typedef std::vector<BlockData> BlockDataList;
typedef std::vector<nnt::edid::FullModeInfo> DetailedTimingList;

TEST(Cea861, GetBlockTag)
{
    // Check Table 41 in CEA-861 spec for basic tags
    // only care about 3 MSb
    for( std::uint16_t i = 0; i <= 0x1FFF; ++i )
    {
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0x0000 | i));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Audio, nn::edid::cea861::GetBlockTag(0x2000 | i));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Video, nn::edid::cea861::GetBlockTag(0x4000 | i));
        EXPECT_EQ(nn::edid::cea861::BlockTag::VendorSpecific, nn::edid::cea861::GetBlockTag(0x6000 | i));
        EXPECT_EQ(nn::edid::cea861::BlockTag::SpeakerAllocation, nn::edid::cea861::GetBlockTag(0x8000 | i));
        EXPECT_EQ(nn::edid::cea861::BlockTag::VesaDtc, nn::edid::cea861::GetBlockTag(0xA000 | i));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xC000 | i));
    }

    // Check Table 43 in CEA-861 spec for extended tags
    // bits 8-12 can be any value (indicate size of block)
    for( std::uint16_t i = 0; i <= 0x001F; ++i )
    {
        std::uint16_t unusedBits = i << 8;

        EXPECT_EQ(nn::edid::cea861::BlockTag::VideoCapability, nn::edid::cea861::GetBlockTag(0xE000 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::VendorSpecificVideo, nn::edid::cea861::GetBlockTag(0xE001 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::VesaDisplayInfo, nn::edid::cea861::GetBlockTag(0xE002 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::VesaVideo, nn::edid::cea861::GetBlockTag(0xE003 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::HdmiVideo, nn::edid::cea861::GetBlockTag(0xE004 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Colorimetry, nn::edid::cea861::GetBlockTag(0xE005 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE006 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE007 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE008 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE009 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE00A | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE00B | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE00C | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE00D | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE00E | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag(0xE00F | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::AudioMisc, nn::edid::cea861::GetBlockTag(0xE010 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::VendorSpecificAudio, nn::edid::cea861::GetBlockTag(0xE011 | unusedBits));
        EXPECT_EQ(nn::edid::cea861::BlockTag::HdmiAudio, nn::edid::cea861::GetBlockTag(0xE012 | unusedBits));

        // the rest of the possible values for LSB are reserved
        for( std::uint16_t j = 19; j <= 255; ++j )
        {
            EXPECT_EQ(nn::edid::cea861::BlockTag::Undefined, nn::edid::cea861::GetBlockTag((0xE0 << 8) | j | unusedBits));
        }
    }
}

bool WriteBlockTags(const std::uint8_t* pBlock, size_t size, nn::edid::cea861::BlockTag tag, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pBlock);

    BlockData data = { tag, size };
    static_cast<BlockDataList*>(pUserData)->push_back(data);

    return true;
}

void TestBlockData(const BlockDataList& list, size_t size, nn::edid::cea861::BlockTag tag) NN_NOEXCEPT
{
    BlockDataList::const_iterator item = std::find_if(list.begin(),
                                                      list.end(),
                                                      [tag] (const BlockData& data) { return data.tag == tag; });
    ASSERT_NE(list.end(), item);
    EXPECT_EQ(size, item->size);
}

TEST(Cea861, VisitDataBlocks)
{
    // Full CEA-861 block
    // Check Table 38 in CEA-861 spec for more details
    std::uint8_t block[] =
    {
        0x02, // Tag
        0x03, // Revision Number
        0x1C, // DTD start/padding start
        0x00, // some flags, but lower nybble is number of DTDs
        0x20, // Audio data block
        0x40, // Video data block
        0x60, // Vendor specific data block
        0x80, // Speaker allocation data block
        0xA0, // VESA DTC data block
        0xE1, // Extended tag
        0x00, // Video capability data block
        0xE1, // Extended tag
        0x01, // Vendor specific video data block
        0xE1, // Extended tag
        0x02, // VESA video display device information data block
        0xE1, // Extended tag
        0x03, // VESA video data block
        0xE1, // Extended tag
        0x04, // HDMI video data block
        0xE1, // Extended tag
        0x05, // Colorimetry data block
        0xE1, // Extended tag
        0x10, // CEA miscellaneous audio fields
        0xE1, // Extended tag
        0x11, // Vendor specific audio data block
        0xE1, // Extended tag
        0x12, // HDMI audio data block
        0x00, // padding start
        0x00,
        0x00,
        0x00,
        0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        // don't really care about checksum
    };

    NN_STATIC_ASSERT(sizeof(block) == 128);

    BlockDataList list;
    nn::edid::cea861::VisitDataBlocks(block, sizeof(block), WriteBlockTags, &list);
    ASSERT_EQ(14, list.size());

    // check for duplicates
    for( BlockDataList::size_type i = 0; i < list.size(); ++i )
    {
        for( BlockDataList::size_type j = i + 1; j < list.size(); ++j )
        {
            // weird overflow case
            if( j < i )
            {
                break;
            }

            if( list[i].tag == list[j].tag )
            {
                // found a duplicate
                FAIL();
            }
        }
    }

    TestBlockData(list, 1, nn::edid::cea861::BlockTag::Audio);
    TestBlockData(list, 1, nn::edid::cea861::BlockTag::Video);
    TestBlockData(list, 1, nn::edid::cea861::BlockTag::VendorSpecific);
    TestBlockData(list, 1, nn::edid::cea861::BlockTag::SpeakerAllocation);
    TestBlockData(list, 1, nn::edid::cea861::BlockTag::VesaDtc);

    // Extended tags must have at least 2 bytes
    TestBlockData(list, 2, nn::edid::cea861::BlockTag::VideoCapability);
    TestBlockData(list, 2, nn::edid::cea861::BlockTag::VendorSpecificVideo);
    TestBlockData(list, 2, nn::edid::cea861::BlockTag::VesaDisplayInfo);
    TestBlockData(list, 2, nn::edid::cea861::BlockTag::VesaVideo);
    TestBlockData(list, 2, nn::edid::cea861::BlockTag::HdmiVideo);
    TestBlockData(list, 2, nn::edid::cea861::BlockTag::Colorimetry);
    TestBlockData(list, 2, nn::edid::cea861::BlockTag::AudioMisc);
    TestBlockData(list, 2, nn::edid::cea861::BlockTag::VendorSpecificAudio);
    TestBlockData(list, 2, nn::edid::cea861::BlockTag::HdmiAudio);
}

bool WriteDetailedTimings(const nn::edid::DisplayModeInfo* pMode, const nn::edid::DisplayTimingInfo* pTiming, const nn::edid::ImageSizeInfo* pImage, void* pUserData) NN_NOEXCEPT
{
    static_cast<DetailedTimingList*>(pUserData)->push_back(nnt::edid::FullModeInfo(pMode, pTiming, pImage));

    return true;
}

TEST(Cea861, VisitDetailedTimings)
{
    std::uint8_t block[] =
    {
        0x02, // Tag
        0x03, // Revision Number
        0x04, // DTD start/padding start
        0x00, // flags
        0x01, // pixel clock for DTD
        0x00,
        0x00,
        0x00,

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        // don't really care about checksum
    };

    DetailedTimingList list;
    nn::edid::cea861::VisitDetailedTimings(block, sizeof(block), WriteDetailedTimings, &list);
    ASSERT_EQ(2, list.size());
    EXPECT_EQ(10, list[0].pTimingInfo->pixelClock);
    EXPECT_EQ(20, list[1].pTimingInfo->pixelClock);
}

void TestVic(std::uint8_t vic, int width, int height, float refreshRate, float imageAspectRatio, float pixelAspectRatio, bool isInterlaced)
{
    nn::edid::DisplayModeInfo mode;
    ASSERT_TRUE(nn::edid::cea861::ConvertVic(&mode, vic));

    EXPECT_EQ(width, mode.width);
    EXPECT_EQ(height, mode.height);
    EXPECT_EQ(refreshRate, mode.refreshRate);
    EXPECT_EQ(imageAspectRatio, mode.imageAspectRatio);
    EXPECT_EQ(pixelAspectRatio, mode.pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, mode.stereoMode);
    EXPECT_EQ(isInterlaced, mode.isInterlaced);
}

TEST(Cea861, ConvertVic)
{
    // unused codes
    // expecting overflow to happen in this loop
    for( std::uint8_t i = 108; i != 1; ++i )
    {
        nn::edid::DisplayModeInfo mode;
        EXPECT_FALSE(nn::edid::cea861::ConvertVic(&mode, i));
    }

    // Check Table 4 in CEA-861 spec for more details
    TestVic(1, 640, 480, 60.00f, 4.f / 3.f, 1.f, false);
    TestVic(2, 720, 480, 60.00f, 4.f / 3.f, 8.f / 9.f, false);
    TestVic(3, 720, 480, 60.00f, 16.f / 9.f, 32.f / 27.f, false);
    TestVic(4, 1280, 720, 60.00f, 16.f / 9.f, 1.f, false);
    TestVic(5, 1920, 1080, 60.00f, 16.f / 9.f, 1.f, true);
    TestVic(6, 720, 480, 60.00f, 4.f / 3.f, 8.f / 9.f, true);
    TestVic(7, 720, 480, 60.00f, 16.f / 9.f, 32.f / 27.f, true);
    TestVic(8, 720, 240, 60.00f, 4.f / 3.f, 4.f / 9.f, false);
    TestVic(9, 720, 240, 60.00f, 16.f / 9.f, 16.f / 27.f, false);
    TestVic(10, 2880, 480, 60.00f, 4.f / 3.f, 2.f / 9.f, true);
    TestVic(11, 2880, 480, 60.00f, 16.f / 9.f, 8.f / 27.f, true);
    TestVic(12, 2880, 240, 60.00f, 4.f / 3.f, 1.f / 9.f, false);
    TestVic(13, 2880, 240, 60.00f, 16.f / 9.f, 4.f / 27.f, false);
    TestVic(14, 1440, 480, 60.00f, 4.f / 3.f, 4.f / 9.f, false);
    TestVic(15, 1440, 480, 60.00f, 16.f / 9.f, 16.f / 27.f, false);
    TestVic(16, 1920, 1080, 60.00f, 16.f / 9.f, 1.f, false);
    TestVic(17, 720, 576, 50.00f, 4.f / 3.f, 16.f / 15.f, false);
    TestVic(18, 720, 576, 50.00f, 16.f / 9.f, 64.f / 45.f, false);
    TestVic(19, 1280, 720, 50.00f, 16.f / 9.f, 1.f, false);
    TestVic(20, 1920, 1080, 50.00f, 16.f / 9.f, 1.f, true);
    TestVic(21, 720, 576, 50.00f, 4.f / 3.f, 16.f / 15.f, true);
    TestVic(22, 720, 576, 50.00f, 16.f / 9.f, 64.f / 45.f, true);
    TestVic(23, 720, 288, 50.00f, 4.f / 3.f, 8.f / 15.f, false);
    TestVic(24, 720, 288, 50.00f, 16.f / 9.f, 32.f / 45.f, false);
    TestVic(25, 2880, 576, 50.00f, 4.f / 3.f, 2.f / 15.f, true);
    TestVic(26, 2880, 576, 50.00f, 16.f / 9.f, 16.f / 45.f, true);
    TestVic(27, 2880, 288, 50.00f, 4.f / 3.f, 1.f / 15.f, false);
    TestVic(28, 2880, 288, 50.00f, 16.f / 9.f, 8.f / 45.f, false);
    TestVic(29, 1440, 576, 50.00f, 4.f / 3.f, 8.f / 15.f, false);
    TestVic(30, 1440, 576, 50.00f, 16.f / 9.f, 32.f / 45.f, false);
    TestVic(31, 1920, 1080, 50.00f, 16.f / 9.f, 1.f, false);
    TestVic(32, 1920, 1080, 24.00f, 16.f / 9.f, 1.f, false);
    TestVic(33, 1920, 1080, 25.00f, 16.f / 9.f, 1.f, false);
    TestVic(34, 1920, 1080, 30.00f, 16.f / 9.f, 1.f, false);
    TestVic(35, 2880, 480, 60.00f, 4.f / 3.f, 2.f / 9.f, false);
    TestVic(36, 2880, 480, 60.00f, 16.f / 9.f, 8.f / 27.f, false);
    TestVic(37, 2880, 576, 50.00f, 4.f / 3.f, 4.f / 15.f, false);
    TestVic(38, 2880, 576, 50.00f, 16.f / 9.f, 16.f / 45.f, false);
    TestVic(39, 1920, 1080, 50.00f, 16.f / 9.f, 1.f, true);
    TestVic(40, 1920, 1080, 100.00f, 16.f / 9.f, 1.f, true);
    TestVic(41, 1280, 720, 100.00f, 16.f / 9.f, 1.f, false);
    TestVic(42, 720, 576, 100.00f, 4.f / 3.f, 16.f / 15.f, false);
    TestVic(43, 720, 576, 100.00f, 16.f / 9.f, 64.f / 45.f, false);
    TestVic(44, 720, 576, 100.00f, 4.f / 3.f, 16.f / 15.f, true);
    TestVic(45, 720, 576, 100.00f, 16.f / 9.f, 64.f / 45.f, true);
    TestVic(46, 1920, 1080, 120.00f, 16.f / 9.f, 1.f, true);
    TestVic(47, 1280, 720, 120.00f, 16.f / 9.f, 1.f, false);
    TestVic(48, 720, 480, 120.00f, 4.f / 3.f, 8.f / 9.f, false);
    TestVic(49, 720, 480, 120.00f, 16.f / 9.f, 32.f / 27.f, false);
    TestVic(50, 720, 480, 120.00f, 4.f / 3.f, 8.f / 9.f, true);
    TestVic(51, 720, 480, 120.00f, 16.f / 9.f, 32.f / 27.f, true);
    TestVic(52, 720, 576, 200.00f, 4.f / 3.f, 16.f / 15.f, false);
    TestVic(53, 720, 576, 200.00f, 16.f / 9.f, 64.f / 45.f, false);
    TestVic(54, 720, 576, 200.00f, 4.f / 3.f, 16.f / 15.f, true);
    TestVic(55, 720, 576, 200.00f, 16.f / 9.f, 64.f / 45.f, true);
    TestVic(56, 720, 480, 240.00f, 4.f / 3.f, 8.f / 9.f, false);
    TestVic(57, 720, 480, 240.00f, 16.f / 9.f, 32.f / 27.f, false);
    TestVic(58, 720, 480, 240.00f, 4.f / 3.f, 8.f / 9.f, true);
    TestVic(59, 720, 480, 240.00f, 16.f / 9.f, 32.f / 27.f, true);
    TestVic(60, 1280, 720, 24.00f, 16.f / 9.f, 1.f, false);
    TestVic(61, 1280, 720, 25.00f, 16.f / 9.f, 1.f, false);
    TestVic(62, 1280, 720, 30.00f, 16.f / 9.f, 1.f, false);
    TestVic(63, 1920, 1080, 120.00f, 16.f / 9.f, 1.f, false);
    TestVic(64, 1920, 1080, 100.00f, 16.f / 9.f, 1.f, false);
    TestVic(65, 1280, 720, 24.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(66, 1280, 720, 25.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(67, 1280, 720, 30.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(68, 1280, 720, 50.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(69, 1280, 720, 60.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(70, 1280, 720, 100.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(71, 1280, 720, 120.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(72, 1920, 1080, 24.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(73, 1920, 1080, 25.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(74, 1920, 1080, 30.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(75, 1920, 1080, 50.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(76, 1920, 1080, 60.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(77, 1920, 1080, 100.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(78, 1920, 1080, 120.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(79, 1680, 720, 24.00f, 64.f / 27.f, 64.f / 63.f, false);
    TestVic(80, 1680, 720, 25.00f, 64.f / 27.f, 64.f / 63.f, false);
    TestVic(81, 1680, 720, 30.00f, 64.f / 27.f, 64.f / 63.f, false);
    TestVic(82, 1680, 720, 50.00f, 64.f / 27.f, 64.f / 63.f, false);
    TestVic(83, 1680, 720, 60.00f, 64.f / 27.f, 64.f / 63.f, false);
    TestVic(84, 1680, 720, 100.00f, 64.f / 27.f, 64.f / 63.f, false);
    TestVic(85, 1680, 720, 120.00f, 64.f / 27.f, 64.f / 63.f, false);
    TestVic(86, 2560, 1080, 24.00f, 64.f / 27.f, 1.f, false);
    TestVic(87, 2560, 1080, 25.00f, 64.f / 27.f, 1.f, false);
    TestVic(88, 2560, 1080, 30.00f, 64.f / 27.f, 1.f, false);
    TestVic(89, 2560, 1080, 50.00f, 64.f / 27.f, 1.f, false);
    TestVic(90, 2560, 1080, 60.00f, 64.f / 27.f, 1.f, false);
    TestVic(91, 2560, 1080, 100.00f, 64.f / 27.f, 1.f, false);
    TestVic(92, 2560, 1080, 120.00f, 64.f / 27.f, 1.f, false);
    TestVic(93, 3840, 2160, 24.00f, 16.f / 9.f, 1.f, false);
    TestVic(94, 3840, 2160, 25.00f, 16.f / 9.f, 1.f, false);
    TestVic(95, 3840, 2160, 30.00f, 16.f / 9.f, 1.f, false);
    TestVic(96, 3840, 2160, 50.00f, 16.f / 9.f, 1.f, false);
    TestVic(97, 3840, 2160, 60.00f, 16.f / 9.f, 1.f, false);
    TestVic(98, 4096, 2160, 24.00f, 256.f / 135.f, 1.f, false);
    TestVic(99, 4096, 2160, 25.00f, 256.f / 135.f, 1.f, false);
    TestVic(100, 4096, 2160, 30.00f, 256.f / 135.f, 1.f, false);
    TestVic(101, 4096, 2160, 50.00f, 256.f / 135.f, 1.f, false);
    TestVic(102, 4096, 2160, 60.00f, 256.f / 135.f, 1.f, false);
    TestVic(103, 3840, 2160, 24.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(104, 3840, 2160, 25.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(105, 3840, 2160, 30.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(106, 3840, 2160, 50.00f, 64.f / 27.f, 4.f / 3.f, false);
    TestVic(107, 3840, 2160, 60.00f, 64.f / 27.f, 4.f / 3.f, false);
} //NOLINT(impl/function_size)
