﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nn/ssl.h>

#include "testFixtures.h"

bool isRunningInCI = false;

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
}

// Setup/teardown performed for each test
void CurlTest::SetUp()
{
    cHandle = curl_easy_init();
    ASSERT_NE(nullptr, cHandle);

    hosts = nullptr;

    if (isRunningInCI)
    {
        hosts = curl_slist_append(hosts, "ntdlibcurltesting01.ntd.nintendo.com:80:192.168.50.19");
        hosts = curl_slist_append(hosts, "ntdlibcurltesting01.ntd.nintendo.com:443:192.168.50.19");

        CURLcode cResult = curl_easy_setopt(cHandle, CURLOPT_RESOLVE, hosts);
        ASSERT_EQ(CURLE_OK, cResult);
    }
}

void CurlTest::TearDown()
{
    curl_easy_cleanup(cHandle);

    if (isRunningInCI)
    {
        curl_slist_free_all(hosts);
    }
}

// Global setup/teardown functions called by GTest
class CurlEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp()
    {
        nn::Result result;

        NN_LOG("==== Mounting SD card\n");
        result = nn::fs::MountSdCardForDebug("sd");
        ASSERT_TRUE(result.IsSuccess());
        isSdCardMounted = true;

        NN_LOG("==== Mounting ROM area\n");
        size_t romCacheSize;
        result = nn::fs::QueryMountRomCacheSize(&romCacheSize);
        ASSERT_TRUE(result.IsSuccess());
        romCache = new char[romCacheSize];
        result = nn::fs::MountRom("rom", romCache, romCacheSize);
        ASSERT_TRUE(result.IsSuccess());
        isRomMounted = true;

        NN_LOG("==== Initializing NIFM library\n");
        result = nn::nifm::Initialize();
        ASSERT_TRUE(result.IsSuccess());

        nn::nifm::SetExclusiveClient(nn::nifm::GetClientId());

        NN_LOG("==== Requesting network access\n");
        m_pNifmConnection = new nn::nifm::NetworkConnection();
        nn::nifm::RequestHandle reqHandle = m_pNifmConnection->GetRequestHandle();
        nn::nifm::SetRequestConnectionConfirmationOption(reqHandle, nn::nifm::ConnectionConfirmationOption_Prohibited);
        nn::nifm::SetRequestPersistent(reqHandle, true);

        m_pNifmConnection->SubmitRequestAndWait();
        ASSERT_TRUE(m_pNifmConnection->IsAvailable());
        isNifmInitialized = true;

        NN_LOG("==== Initializing SSL library\n");
        result = nn::ssl::Initialize();
        ASSERT_TRUE(result.IsSuccess());
        isSslInitialized = true;

        NN_LOG("==== Initializing socket library\n");
        result = nn::socket::Initialize(g_SocketConfigWithMemory);
        ASSERT_TRUE(result.IsSuccess());
        isSocketInitialized = true;

        NN_LOG("==== Initializing libcurl\n");
        CURLcode cResult = curl_global_init(CURL_GLOBAL_ALL);
        ASSERT_EQ(CURLE_OK, cResult);
        isLibcurlInitialized = true;
    }

    virtual void TearDown()
    {
        nn::Result result;

        NN_LOG("==== Cleaning up libcurl\n");
        if (isLibcurlInitialized)
        {
            curl_global_cleanup();
        }

        NN_LOG("==== Cleaning up socket library\n");
        if (isSocketInitialized)
        {
            result = nn::socket::Finalize();
            ASSERT_TRUE(result.IsSuccess());
        }

        NN_LOG("==== Cleaning up SSL library\n");
        if (isSslInitialized)
        {
            result = nn::ssl::Finalize();
            ASSERT_TRUE(result.IsSuccess());
        }

        NN_LOG("==== Cleaning up NIFM library\n");
        if (isNifmInitialized)
        {
            m_pNifmConnection->CancelRequest();
        }

        NN_LOG("==== Unmounting ROM area\n");
        if (isRomMounted)
        {
            nn::fs::Unmount("rom");
            delete[] romCache;
        }

        NN_LOG("==== Unmounting SD card\n");
        if (isSdCardMounted)
        {
            nn::fs::Unmount("sd");
        }
    }

protected:
    bool isLibcurlInitialized = false;
    bool isNifmInitialized = false;
    bool isRomMounted = false;
    bool isSdCardMounted = false;
    bool isSocketInitialized = false;
    bool isSslInitialized = false;

    char *romCache;
    nn::nifm::NetworkConnection *m_pNifmConnection;
};

extern "C"
void nnMain()
{
    // Get parameters from OS
    int    argc = nnt::GetHostArgc();
    char **argv = nnt::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);

    // Check if we're running in CI
    for (int i = 0; i < argc; ++i)
    {
        if (strcmp(argv[i], "-ci") == 0)
        {
            isRunningInCI = true;
        }
    }

    // Register global environment
    ::testing::AddGlobalTestEnvironment(new CurlEnvironment);

    // Begin testing
    int testResult = RUN_ALL_TESTS();

    NN_LOG("==== Done with curl testing\n");

    nnt::Exit(testResult);
}
