﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <map>
#include <string>

#include "testFixtures.h"
#include "utils.h"

TEST_F(CurlTest, BasicHttpTest)
{
    CURLcode cResult;
    nn::Result nResult;

    const char *testUrl = "http://natf.com/headers.php";

    // Strings used for header testing
    const char *testUserAgent = "libcurl/7.42.1 (Nintendo)";
    const char *testReferer = "http://example.com/aboutme.html";
    const char *testCookie = "PRODUCT=NX; COMPANY=Nintendo";

    // List of non-standard headers
    curl_slist *customHeaders = nullptr;
    customHeaders = curl_slist_append(customHeaders, "X-NTD-Testing: libcurl 7.42.1");
    customHeaders = curl_slist_append(customHeaders, "X-Dummy-Header: swordfish");

    nn::fs::FileHandle outputFile;
    const char *outputFileName = "sd:/response.xml";
    nResult = CreateFileForWriting(&outputFile, outputFileName);
    ASSERT_TRUE(nResult.IsSuccess());

    // Set up an info structure for the writer function
    FileWriterInfo fwInfo;
    fwInfo.handle = outputFile;
    fwInfo.bytesWritten = 0;

    // Set up libcurl options
    cResult = curl_easy_setopt(cHandle, CURLOPT_HTTPGET, 1);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEDATA, &fwInfo);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEFUNCTION, FileWriterCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_0);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_HTTPHEADER, customHeaders);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_USERAGENT, testUserAgent);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_REFERER, testReferer);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_COOKIE, testCookie);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_LOCALPORT, 10000);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_LOCALPORTRANGE, 10);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    // Make the request
    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_OK, cResult);

    // Ensure that everything worked
    long responseCode;
    cResult = curl_easy_getinfo(cHandle, CURLINFO_RESPONSE_CODE, &responseCode);
    ASSERT_EQ(CURLE_OK, cResult);
    ASSERT_EQ(200, responseCode);

    curl_slist_free_all(customHeaders);

    // Close the response file
    nn::fs::CloseFile(outputFile);

    // Parse the XML file into a document
    pugi::xml_document xmlResponse;
    ASSERT_EQ(ReadFileResult_Success, ReadXmlDocFromFile(&xmlResponse, outputFileName));
    pugi::xml_node headers = xmlResponse.child("RequestHeaders");

    // Loop through the "Header" elements of the document
    std::map<std::string, std::string> headerMap;
    for (pugi::xml_node header = headers.child("Header"); header; header = header.next_sibling("Header"))
    {
        const char *headerName = header.child("Name").child_value();
        const char *headerValue = header.child("Value").child_value();

        headerMap.insert(std::make_pair(headerName, headerValue));
    }

    // Check the headers we set
    ASSERT_STREQ(testUserAgent, headerMap["User-Agent"].c_str());
    ASSERT_STREQ(testReferer, headerMap["Referer"].c_str());
    ASSERT_STREQ("libcurl 7.42.1", headerMap["X-NTD-Testing"].c_str());
    ASSERT_STREQ("swordfish", headerMap["X-Dummy-Header"].c_str());
    ASSERT_STREQ(testCookie, headerMap["Cookie"].c_str());

    // Verify other request parameters
    ASSERT_STREQ("GET", headers.child("Method").child_value());
    ASSERT_STREQ("HTTP/1.0", headers.child("Protocol").child_value());

    int localPort = headers.child("RemoteHost").child("Port").text().as_int();
    ASSERT_GE(localPort, 10000);
    ASSERT_LT(localPort, 10010);
}
