﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <ImageTransferModeTestTask.h>
#include <nn/xcd/xcd.h>
#include "LensDistortionTable.h"

const ImageTransferModeTestTask::ImageSize ImageTransferModeTestTask::FORMAT_SIZE[SIZE_FORMAT_MAX_NUM] =
{
    {320, 240},
    {160, 120},
    {80, 60},
};

const ImageTransferModeTestTask::ImageSize ImageTransferModeTestTask::FORMAT_SIZE_UNDISTORTED[SIZE_FORMAT_MAX_NUM] =
{
    {400, 300},
    {200, 150},
    {100, 75},
};

const char* ImageTransferModeTestTask::MENU_ITEM_IMAGE_TRANSFER_NAMES[ImageTransferModeTestTask::MENU_ITEM_IMAGE_TRANSFER_NUM] =
{
    "SizeFormat",
    "DistortionCompensation"
};

namespace
{
    const char* WINDOW_NAME_DISTORTION_COMPENSATION   = "ProcessorTestTask DistortionCompensated";

    const char* OUTPUT_IMAGE_FILE_NAME = "%s\\%s_%04d%02d%02d_%02d%02d%02d_%04d.png";
    const char* OUTPUT_IMAGE_TRANSFER_FILE_NAME = "%s\\%s_%s.csv";
    FILE* fp;
    const size_t FILE_NAME_SIZE = 256;
    const size_t LINE_SIZE = 4096;

    char s_PrevBuffer[nn::xcd::IrImageSizeMax];
}
//============================================================
void ImageTransferModeTestTask::DoInitializeCore()
{
    m_State.pImage = new uint8_t[nn::xcd::IrImageSizeMax];
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::ImageTransfer;
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::ImageTransfer))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }

    // 内部パラメータの初期化
    SetDefaultImageTransferConfig(m_Config);
    SetDefaultImageTransferConfig(m_CurrentConfig);

} //ImageTransferModeTestTask::DoInitialize()

//============================================================
void ImageTransferModeTestTask::DoFinalizeCore()
{
    m_pDriver->SetSixAxisSampling(false);
    m_pDriver->SetLraSending(false);
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::Ready;
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::Ready))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }
    // ディストーション補正表示中の場合はリソースを解放
    if (m_CurrentDistortionCompensationConfig != DISTORTION_COMPENSATION_CONFIG_NONE)
    {
        cvReleaseImage(&m_pDistortionCompensatedImage);
        m_pDistortionCompensatedImage = nullptr;
        cvDestroyWindow(WINDOW_NAME_DISTORTION_COMPENSATION);
        m_CurrentDistortionCompensationConfig = DISTORTION_COMPENSATION_CONFIG_NONE;
        m_DistortionCompensationConfig = DISTORTION_COMPENSATION_CONFIG_NONE;
    }

    delete [] m_State.pImage;
    m_State.pImage = nullptr;
} //ImageTransferModeTestTask::DoFinalize()

//============================================================
void ImageTransferModeTestTask::DoCalcCore(IplImage* pImage, IplImage* pDstImage)
{
    // データ取得
    m_SamplingResult = m_pDriver->GetImageTransferData(&m_ImageHeader, &m_State);

    // データが少しでも変化していたらコピーする
    if (pImage->imageData == nullptr ||
        memcmp(s_PrevBuffer, reinterpret_cast<char*>(m_State.pImage), FORMAT_SIZE[static_cast<int>(m_CurrentConfig)].width * FORMAT_SIZE[static_cast<int>(m_CurrentConfig)].height) != 0)
    {
        m_IsDataUpdated = true;
        pImage->imageData = reinterpret_cast<char*>(m_State.pImage);
        pImage->width = FORMAT_SIZE[static_cast<int>(m_CurrentConfig)].width;
        pImage->height = FORMAT_SIZE[static_cast<int>(m_CurrentConfig)].height;

        if (m_CurrentDistortionCompensationConfig != DISTORTION_COMPENSATION_CONFIG_NONE)
        {
            CompensateDistortion(pImage, m_pDistortionCompensatedImage);
        }

        cvCvtColor(pImage, pDstImage, CV_GRAY2BGR);
        memcpy(s_PrevBuffer, pImage->imageData, FORMAT_SIZE[static_cast<int>(m_CurrentConfig)].width * FORMAT_SIZE[static_cast<int>(m_CurrentConfig)].height);
    }
    else
    {
        m_IsDataUpdated = false;
    }
} //ImageTransferModeTestTask::DoCalc()

//============================================================
void ImageTransferModeTestTask::DoDrawCore(IplImage* pImage)
{
    NN_UNUSED(pImage);
    if (m_CurrentDistortionCompensationConfig != DISTORTION_COMPENSATION_CONFIG_NONE)
    {
        if (m_pDistortionCompensatedImage != NULL)
        {
            cvShowImage(WINDOW_NAME_DISTORTION_COMPENSATION, m_pDistortionCompensatedImage);
        }
    }

    if (m_IsLogEnabled)
    {
        SaveImage(pImage);
    }
} //ImageTransferModeTestTask::DoDraw()

//============================================================
void ImageTransferModeTestTask::DoDrawMenuCore(IplImage* pImage)
{
    const CvSize FONT_SIZE = cvSize(16, 16);
    const char* SIZE_FORMAT_NAMES[] = {"QVGA", "QQVGA", "QQQVGA"};
    const char* DISOTRTION_COMPENSATION_NAMES[] = {"NONE", "IDEAL", "OPTIMIZE"};

    for (s32 i = MENU_ITEM_NUM; i < (MENU_ITEM_NUM + MENU_ITEM_IMAGE_TRANSFER_NUM); i++)
    {
        s32 tx = 2 * FONT_SIZE.width;
        s32 ty = i * FONT_SIZE.height;

        m_TextWriter.SetFontColor((m_Cursor == i) ? test::CONST_COLOR_RED : test::CONST_COLOR_WHITE);
        m_TextWriter.PutText(pImage, 0, ty, "%s", (m_Cursor == i) ? ">" : " ");

        if (i == (MENU_ITEM_IMAGE_TRANSFER_SIZE_FORMAT + MENU_ITEM_NUM))
        {
            m_TextWriter.PutText(pImage, tx, ty, "%s : %s", MENU_ITEM_IMAGE_TRANSFER_NAMES[MENU_ITEM_IMAGE_TRANSFER_SIZE_FORMAT], SIZE_FORMAT_NAMES[static_cast<int>(m_Config)]);
        }
        if (i == (MENU_ITEM_IMAGE_TRANSFER_DISTORTION_COMPENSATION + MENU_ITEM_NUM))
        {
            m_TextWriter.PutText(pImage, tx, ty, "%s : %s", MENU_ITEM_IMAGE_TRANSFER_NAMES[MENU_ITEM_IMAGE_TRANSFER_DISTORTION_COMPENSATION], DISOTRTION_COMPENSATION_NAMES[static_cast<int>(m_DistortionCompensationConfig)]);
        }
    }
} //ImageTransferModeTestTask::DoDraw()


//============================================================
void ImageTransferModeTestTask::DoCalcMenuCore(int* pMenuCnt)
{
    s32 keycode = GetKeycode();
    if (m_Cursor == (MENU_ITEM_IMAGE_TRANSFER_SIZE_FORMAT + MENU_ITEM_NUM))
    {
        if (keycode == KEY_CODE_LEFT)
        {
            m_Config = static_cast<SizeFormat>(m_Config - 1);
            if (m_Config < 0)
            {
                m_Config = static_cast<SizeFormat>(SIZE_FORMAT_MAX_NUM - 1);
            }
        }
        if (keycode == KEY_CODE_RIGHT)
        {
            m_Config= static_cast<SizeFormat>(m_Config + 1);
            if (m_Config >= SIZE_FORMAT_MAX_NUM)
            {
                m_Config = static_cast<SizeFormat>(0);
            }
        }
    }
    if (m_Cursor == (MENU_ITEM_IMAGE_TRANSFER_DISTORTION_COMPENSATION + MENU_ITEM_NUM))
    {
        if (keycode == KEY_CODE_LEFT)
        {
            m_DistortionCompensationConfig = static_cast<DistortionCompensationConfig>(m_DistortionCompensationConfig - 1);
            if (m_DistortionCompensationConfig < 0)
            {
                m_DistortionCompensationConfig = static_cast<DistortionCompensationConfig>(DISTORTION_COMPENSATION_CONFIG_MAX_NUM - 1);
            }
        }
        if (keycode == KEY_CODE_RIGHT)
        {
            m_DistortionCompensationConfig = static_cast<DistortionCompensationConfig>(m_DistortionCompensationConfig + 1);
            if (m_DistortionCompensationConfig >= DISTORTION_COMPENSATION_CONFIG_MAX_NUM)
            {
                m_DistortionCompensationConfig = static_cast<DistortionCompensationConfig>(0);
            }
        }
    }
    *pMenuCnt = MENU_ITEM_IMAGE_TRANSFER_NUM;
}

void ImageTransferModeTestTask::DoWriteRegSettingCore(nn::xcd::IrWriteRegisterSetting& setting, int& index)
{
    NN_UNUSED(setting);
    NN_UNUSED(index);
    if (m_Config != m_CurrentConfig)
    {
        // format 変更
        uint8_t data = 0;
        if (m_Config == SIZE_FORMAT_320x240)
        {
            data = 0x00;
        }
        else if (m_Config == SIZE_FORMAT_160x120)
        {
            data = 0x50;
        }
        else if (m_Config == SIZE_FORMAT_80x60)
        {
            data = 0x64;
        }
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x2E;
        setting.registerBlock[index].data = data;
        index++;
        setting.registerCount = index;
        //m_IsImageSizeChanged = true;

        // 画像サイズ変更時は強制OFF にする
        m_DistortionCompensationConfig = DISTORTION_COMPENSATION_CONFIG_NONE;
    }

    if (m_DistortionCompensationConfig != m_CurrentDistortionCompensationConfig)
    {
        if (m_DistortionCompensationConfig != DISTORTION_COMPENSATION_CONFIG_NONE)
        {
            cvNamedWindow(WINDOW_NAME_DISTORTION_COMPENSATION);
            int undistortedWidth = FORMAT_SIZE_UNDISTORTED[m_CurrentConfig].width;
            int undistortedHeight = FORMAT_SIZE_UNDISTORTED[m_CurrentConfig].height;
            m_pDistortionCompensatedImage = cvCreateImage(cvSize(undistortedWidth, undistortedHeight), IPL_DEPTH_8U, 1);
            TEST_ASSERT(m_pDistortionCompensatedImage != NULL);
        }
        else
        {
            cvReleaseImage(&m_pDistortionCompensatedImage);
            m_pDistortionCompensatedImage = nullptr;
            cvDestroyWindow(WINDOW_NAME_DISTORTION_COMPENSATION);
        }
    }
    m_CurrentDistortionCompensationConfig = m_DistortionCompensationConfig;

    m_CurrentConfig = m_Config;
    m_ImageSizeIndex = static_cast<int>(m_CurrentConfig);
}

void ImageTransferModeTestTask::DoSetDefaultConfigCore(ProcessorConfig& config)
{
    config.digitalGain = GAIN_X8;
    config.exposureTimeUsec = 300;
    config.irLedMode = IR_LED_BOTH;
    config.irLedIntensity = 16;
    config.postProcess = POSTPROCESS_NONE;
    config.isOnOffSubtractionEnabled = true;
    config.isHdrEnabled = false;
    config.isLensShadingEnabled = true;
    config.fps = 60;
}

void ImageTransferModeTestTask::SetDefaultImageTransferConfig(SizeFormat& config)
{
    config = SIZE_FORMAT_320x240;
    m_ImageSizeIndex = 0;
}

void ImageTransferModeTestTask::SaveImage(IplImage* pImage)
{
    char path[1024];

    SYSTEMTIME st;
    GetLocalTime(&st);

    if (m_IsDataUpdated)
    {
        m_TotalFrameCount++;
        bool isCaptureEnabled = false;
        if (m_TotalFrameCount % (m_SkipFrame + 1) == 0)
        {
            m_CaptureFrameCount++;
            isCaptureEnabled = true;
        }
        if (isCaptureEnabled)
        {
            if (m_IsVisualLogEnabled && m_SamplingResult.IsSuccess())
            {
                sprintf_s(path, OUTPUT_IMAGE_FILE_NAME, m_DirectoryPath, GetName(), st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
                s32 success = cvSaveImage(path, pImage);
                if (success == 0)
                {
                    TEST_PRINTF("Failed to save an image: %s\n", path);
                    return;
                }
                TEST_PRINTF("Save an image: %s\n", path);
            }

            // データログ
            if (fp != NULL)
            {
                char data[LINE_SIZE];
                sprintf_s(data, "%04d%02d%02d_%02d%02d%02d_%04d", st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
                if (m_SamplingResult.IsSuccess())
                {
                    sprintf_s(data, "%s\n", data);
                }
                else
                {
                    sprintf_s(data, "%s,SamplingFailed,%0X\n", data, m_SamplingResult.GetInnerValueForDebug());
                }
                fputs(data, fp);
            }
        }
    }
}

void ImageTransferModeTestTask::DoCreateLogFileCore()
{
    char path[FILE_NAME_SIZE];
    sprintf_s(path, OUTPUT_IMAGE_TRANSFER_FILE_NAME, m_DirectoryPath, GetName(), m_DirectoryPath);
    if ((fp = fopen(path, "w")) == NULL)
    {
        TEST_PRINTF("Failed to create file\n");
    }
    char data[LINE_SIZE];
    sprintf_s(data, "time,,\n");
    fputs(data, fp);
}

void ImageTransferModeTestTask::DoCloseLogFileCore()
{
    fclose(fp);
}

inline float ApproximationConversion(float value)
{
    // 6次近似
    const float a6 = 234.726f;
    const float a5 = -532.274f;
    const float a4 = 479.245f;
    const float a3 = -280.277f;
    const float a2 = 96.4481f;
    const float a1 = 0.902929f;
    const float b  = 0.019044f;

    return a6 * powf(value, 6.0f) + a5 * powf(value, 5.0f) + a4 * powf(value, 4.0f) + a3 * powf(value, 3.0f) + a2 * powf(value, 2.0f) + a1 * value * b;
}

void ImageTransferModeTestTask::CompensateDistortion(IplImage* pImage, IplImage* pDstImage)
{
    int width = FORMAT_SIZE[m_CurrentConfig].width;
    int height = FORMAT_SIZE[m_CurrentConfig].height;
    float maxDistortedDistance = sqrtf(powf(width / 2.0f - 0.5f, 2.0f) + powf(height / 2.0f - 0.5f, 2.0f));
    for (auto i = 0; i < height; i++)
    {
        for (auto j = 0; j < width; j++)
        {
            // 元座標のデータ
            int origData = pImage->imageData[i * width + j];
            float distance = sqrtf(powf(i - height / 2.0f + 0.5f, 2.0f) + powf(j - width / 2.0f + 0.5f, 2.0f));
            float imageHeightOnSensor = distance / maxDistortedDistance;
            //TEST_PRINTF("dist:%f hos:%f distortion:%f\n", distance, imageHeightOnSensor, DistortionTable[static_cast<int>(imageHeightOnSensor * 1000 - 0.5f)] * 0.01);
            // 変換先の座標（中心を(0,0)とした相対座標）
            int undistortedX = 0;
            int undistortedY = 0;
            if (m_CurrentDistortionCompensationConfig == DISTORTION_COMPENSATION_CONFIG_IDEAL)
            {
                undistortedX = static_cast<int>((j - width / 2.0f + 0.5f) / (1 + IdealDistortionTable[static_cast<int>(imageHeightOnSensor * 1000 - 0.5f)] * 0.01) + 0.5f);
                undistortedY = static_cast<int>((i - height / 2.0f + 0.5f) / (1 + IdealDistortionTable[static_cast<int>(imageHeightOnSensor * 1000 - 0.5f)] * 0.01) + 0.5f);
            }
            else if (m_CurrentDistortionCompensationConfig == DISTORTION_COMPENSATION_CONFIG_OPTIMIZED)
            {
                //undistortedX = static_cast<int>((j - width / 2.0f + 0.5f) / (1 + OptimizedDistortionTable[static_cast<int>(imageHeightOnSensor * 1000 - 0.5f)] * 0.01) + 0.5f);
                //undistortedY = static_cast<int>((i - height / 2.0f + 0.5f) / (1 + OptimizedDistortionTable[static_cast<int>(imageHeightOnSensor * 1000 - 0.5f)] * 0.01) + 0.5f);
                undistortedX = static_cast<int>((j - width / 2.0f + 0.5f) / (1 + ApproximationConversion(imageHeightOnSensor) * 0.01) + 0.5f);
                undistortedY = static_cast<int>((i - height / 2.0f + 0.5f) / (1 + ApproximationConversion(imageHeightOnSensor) * 0.01) + 0.5f);
            }

            int undistortedWidth = FORMAT_SIZE_UNDISTORTED[m_CurrentConfig].width;
            int undistortedHeight = FORMAT_SIZE_UNDISTORTED[m_CurrentConfig].height;
            //TEST_PRINTF("(j, i) -> (X, Y) = (%d, %d) -> (%d, %d)\n", j, i, undistortedX, undistortedY);
            if (m_CurrentConfig == SIZE_FORMAT_320x240)
            {
                pDstImage->imageData[static_cast<int>((undistortedHeight / 2.0f - 0.5f + undistortedY) * undistortedWidth + (undistortedWidth / 2.0f - 0.5f + undistortedX) + 0.5f)] = origData;
            }
            else
            {
                // QVGA, QQVGA ではなぜか中心がズレるため、ワークアラウンド
                pDstImage->imageData[static_cast<int>((undistortedHeight / 2.0f - 0.5f + undistortedY) * undistortedWidth + (undistortedWidth - 0.5f + undistortedX) + 0.5f)] = origData;
            }
        }
    }
}


//============================================================
