﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "test_Common.h"

namespace test {

//==========================================================
//! @brief  時間クラスです。
class TimeSpan
{
public:
    //! @brief  時間を 0 に設定するコンストラクタです。
    TimeSpan();
    virtual ~TimeSpan();

public:
    //! @brief  nsec 単位での時間を取得します。
    s64 GetNanoSeconds() const;

    //! @brief  μsec 単位での時間を取得します。
    inline s64 GetMicroSeconds() const;

    //! @brief  msec 単位での時間を取得します。
    inline s64 GetMilliSeconds() const;

    //! @brief  sec 単位での時間を取得します。
    inline s64 GetSeconds() const;

public:
    //! @brief  nsec 単位で時間を設定します。
    void SetNanoSeconds(s64 nsec);

    //! @brief  μsec 単位で時間を設定します。
    inline void SetMicroSeconds(s64 usec);

    //! @brief  Tick 単位の時間を設定します。
    inline void SetTicks(s64 tick);

    //! @brief  時間和を計算します。
    inline TimeSpan operator+(const TimeSpan& rhs) const;

    //! @brief  時間差を計算します。
    inline TimeSpan operator-(const TimeSpan& rhs) const;

    //! @brief  時間に引数を乗じます。
    inline TimeSpan operator*(const s64 n) const;

    //! @brief  時間を引数で除算します。
    inline TimeSpan operator/(const s64 n) const;

    //! @brief  時間和を計算します。
    inline TimeSpan& operator+=(const TimeSpan& rhs);

    //! @brief  時間差を計算します。
    inline TimeSpan& operator-=(const TimeSpan& rhs);

    //! @brief  引数を乗じます。
    inline TimeSpan& operator*=(const s64 n);

    //! @brief  引数で除算します。
    inline TimeSpan& operator/=(const s64 n);

private:
    s64 m_Span;   //!< Tick 数。

private:
    static const s64 sc_Frequency;  //!< クロック周波数。すなわち 1 秒あたりの Tick 数。
}; //class TimeSpan

//==========================================================
//! @brief  時刻クラスです。
class TickTime
{
public:
    //! @brief  時刻を 0 に設定するコンストラクタです。
    inline TickTime();
    inline virtual ~TickTime();

public:
    //! @brief  現在の CPU 時刻を設定します。
    void SetCurrentCpuTime();

public:
    //! @brief  現在の時刻を camera::TimeSpan の値に変換します。
    inline TimeSpan ToTimeSpan() const;

public:
    //! @brief  時刻に時間を加えます。
    inline TickTime operator+(const TimeSpan& rhs) const;

    //! @brief  時刻を減算します。
    inline TickTime operator-(const TickTime& rhs) const;

private:
    s64 m_Tick;   //!< Tick 値。
};//class TickTime

} //namespace test

//==========================================================
#include "test_Time.hpp"
//==========================================================
