﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "test_Common.h"

namespace test {

class FrameworkBase;

//============================================================
/*
 *  @brief タスクの基底クラスです。タスクはこのクラスを継承して実装してください。
 */
class TaskBase
{
public:
    enum
    {
        TASK_NAME_MAX_LEN = 255
    };

public:
    TaskBase();
    TaskBase(const char* name, uptr arg);  // TaskBase では arg は使用されません。サブクラスのためのものです。サブクラスから呼び出すときには NULL でかまいません。
    virtual ~TaskBase();

private:
    TaskBase(const TaskBase& rhs);
    TaskBase& operator=(const TaskBase& rhs);

public:  // フレームワークから呼ばれる実行関数
    //! @brief タスクを初期化します。内部で DoInitialize を呼びます。
    void Initialize(FrameworkBase* pFramework);     //@todo ヒープを実装したらヒープを渡すようにする

    //! @brief タスクを終了します。内部で DoFinalize を呼びます。
    void Finalize();

    //! @brief Calc を実行します。内部で DoCalc を呼びます。
    void Calc();

    //! @brief Draw を実行します。内部で DoDraw を呼びます。
    void Draw();

protected:  // 実行関数の本体の実装。サブクラスで実装してください。実装しない場合、空となります。
    virtual void DoInitialize(){}
    virtual void DoFinalize(){}
    virtual void DoCalc(){}
    virtual void DoDraw(){}

public:
    //! @brief 名前を返します。
    const char* GetName() const
    {
        return m_Name;
    } //GetName()

    //! @brief 初期化したかどうかを返します。
    bool IsInitialized() const
    {
        return m_IsInitialized;
    } //IsInitialized()

    //! @brief 現在のフレームカウント数を返します。
    s32 GetFrameCount() const
    {
        return m_FrameCount;
    } //GetFrameCount()

protected: // サブクラスから呼ばれる想定の関数
    //! @brief タスクが動作しているフレームワークを取得します。
    FrameworkBase* GetFramework()
    {
        return m_pFramework;
    } //GetFramework()

    const FrameworkBase* GetFrameworkConst() const
    {
        return m_pFramework;
    }

private:
    bool m_IsInitialized; //!< 初期化されたかどうか。
    s32 m_FrameCount; //!< フレームカウント数。
    FrameworkBase* m_pFramework; //!< タスクが動作するフレームワーク。
    char m_Name[TASK_NAME_MAX_LEN + 1]; //!< タスク名
}; //class TaskBase

} //namespace test
