﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "test_Common.h"

namespace test {

class TaskBase;

//============================================================
/**
 *  @brief フレームワークの基底クラスです。プラットフォームに合わせてサブクラスを実装してください。
 */
class FrameworkBase
{
public:
    FrameworkBase();
    virtual ~FrameworkBase();

private:
    FrameworkBase(const FrameworkBase& rhs);
    FrameworkBase& operator=(const FrameworkBase& rhs);

public:
    //! @brief 初期化します。
    void Initialize();

    //! @brief 終了します。
    void Finalize();

    //! @brief メインループを実行します。
    void Run(TaskBase* pTask);

protected:  // 実行処理の本体です。サブクラスで実装してください。
    virtual void DoInitialize() = 0;
    virtual void DoFinalize() = 0;
    virtual void DoMainLoop() = 0;  // メインループを無限ループで回せないプラットフォーム(iOSなど)があるため、メインループもサブクラスで実装させることにします。

protected:
    //! @brief 実行状態です。
    enum State
    {
        STATE_NOT_INITIALIZED,    //!< 初期化されていない。
        STATE_INITIALIZED,        //!< 初期化されたが Run はされていない。
        STATE_PREPARED,           //!< メインループの実行準備が完了。
        STATE_RUNNING,            //!< メインループの実行中。
        STATE_NUM
    }; //enum State

    //! @brief 現在のステートを取得します。
    State GetState() const
    {
        return m_State;
    } //GetState()

    //! @brief タスクを取得します。サブクラスから呼び出されることを想定しています。
    TaskBase* GetTask() const
    {
        TEST_ASSERT(m_State >= STATE_PREPARED);  // 準備ができるまで呼び出してはいけない
        return m_pTask;
    } //GetTask()

private:
    State m_State; //!< 実行状態。
    TaskBase* m_pTask; //!< 実行するタスクです。
}; //class FrameworkBase

} //namespace test

