﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace {

    struct TestBssParams
    {
        uint32_t testNo;
        nn::wlan::MasterBssParameters bssParam;
    };

    const char WlanTesStaticAesSecuritySSID[]   = "WlanTest";

    const size_t WlanTestBufferSize             = 512;
    const size_t WlanTestMsgBufferSize          = 128;

    const uint32_t HeaderSize = nn::wlan::MacAddress::MacAddressSize * 2;

    const uint8_t TestEtherType[] = {
        0x80, 0xB7
    };

    const TestBssParams CreateBssTestParams[] = {
        // ステルス有
        // 正常値テストNo11 Open
        { 11, {testMasterBss.channel, testMasterBss.hiddenSsid, testMasterBss.inactivePeriod, testMasterBss.autoKeepAlive, testMasterBss.supportedRates,
            testMasterBss.basicRates, testMasterBss.security, testMasterBss.ssid, testMasterBss.beaconInterval} },
        // 正常値テストNo19 StaticAes
        { 19, { -1, true, 30, false, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, "" }, nn::wlan::Ssid(WlanTesStaticAesSecuritySSID), 100} },
        // ステルス無
        // 正常値テストNo20 Open
        { 20, {-1, false, 30, false, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(WlanTesStaticAesSecuritySSID), 100} },
        // 正常値テストNo28 StaticAes
        { 28, {-1, false, 30, false, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, "" }, nn::wlan::Ssid(WlanTesStaticAesSecuritySSID), 100} }
    };

    const TestBssParams ConnectBssTestParams[] = {
        // ステルス無
        { 35, {testMasterBss.channel, testMasterBss.hiddenSsid, testMasterBss.inactivePeriod, testMasterBss.autoKeepAlive, testMasterBss.supportedRates,
            testMasterBss.basicRates, testMasterBss.security, testMasterBss.ssid, testMasterBss.beaconInterval} },
        { 54, {-1, false, 30, false, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, "" }, nn::wlan::Ssid(WlanTesStaticAesSecuritySSID), 100} }
    };
}

namespace WlanTest {

    class LocalMasterNormalInitUnitTest : public ::testing::Test
    {
    protected:
        // LocalMasterの通信開始
        LocalApiClass localMaster;

        nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
        nn::wlan::ConnectionStatus connectStatus;
        nn::wlan::WlanState wlanState = nn::wlan::WlanState_Stop;
        nn::Bit32 clientStatusBitMap;
        nn::os::SystemEventType connectEvent;
        nn::wlan::MacAddress macAddress;
        nn::wlan::MacAddress putMacAddress;
        uint32_t rxId;
        uint32_t actionRxId;

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
        }
    };

    class LocalMasterNormalUnitTest : public LocalMasterNormalInitUnitTest
    {
    protected:

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
        }
    };

    class LocalMasterBssParamsTest : public LocalMasterNormalUnitTest,
        public ::testing::WithParamInterface<TestBssParams>
    {
    protected:
        std::unique_ptr<char[]> failMessage;
        nn::os::SystemEventType connectEvent;
        LocalApiClass localMaster;

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            std::unique_ptr<char[]> message(new char[WlanTestMsgBufferSize]);
            WLANTEST_ASSERT_TRUE(message.get() != nullptr);
            failMessage = std::move(message);

            LocalMasterNormalUnitTest::SetUp();
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            LocalMasterNormalUnitTest::TearDown();
        }
    };

    typedef LocalMasterBssParamsTest LocalMasterNormalBssUnitTest;
    typedef LocalMasterBssParamsTest LocalMasterNormalConnectUnitTest;

    TEST_F(LocalMasterNormalInitUnitTest, LocalNanager_No1_No2_No3)
    {
        // 正常値テストNo1 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
        // 正常値テストNo2 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
        // 正常値テストNo3 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
    }

    TEST_F(LocalMasterNormalInitUnitTest, LocalNanager_No4)
    {
        // 正常値テストNo4 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
    }

    TEST_F(LocalMasterNormalInitUnitTest, MasterMode_No5_No6)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());

        // 正常値テストNo5 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());

        // 正常値テストNo6 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
    }

    //No11~No29 NTが含まれるため、実質No11, No19, No20, No28, No29
    TEST_P(LocalMasterNormalBssUnitTest, CreateDestroyBss_No11_No29)
    {
        TestBssParams testBssParam = GetParam();

        NN_LOG("             LocalClient LocalMasterNormalBssUnit Test No.%d START\n", testBssParam.testNo);
        nn::util::SNPrintf(failMessage.get(), WlanTestMsgBufferSize, "CreateDestroyBss Test No.%d ssid(%s) hidden(%d)",
            testBssParam.testNo, testBssParam.bssParam.ssid.GetSsidData(), testBssParam.bssParam.hiddenSsid);
        WLANTEST_SCOPED_TRACE(failMessage.get());

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(testBssParam.bssParam));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
        NN_LOG("             LocalClient LocalMasterNormalBssUnit Test No.%d END\n", testBssParam.testNo);
    }

    TEST_F(LocalMasterNormalUnitTest, GetConnectionStatus_No77_No80_No83)
    {
        nn::os::SystemEvent conEvent;
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1));

        // Connect Wait
        // 正常値テストNo80
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(conEvent.GetBase()));
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(conEvent.GetBase(), nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectStatus));
        WLANTEST_ASSERT_TRUE(connectStatus.state == nn::wlan::ConnectionState_Connected);

        // 正常値テストNo77, No83
        nn::wlan::DisconnectClient dis;
        dis.clientMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        dis.reasonCode = nn::wlan::Dot11ReasonCode_UnspecifiedFailure;
        nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &dis);

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, GetConnectionStatus_No86)
    {
        nn::os::SystemEvent conEvent;
        nn::wlan::MacAddress fromMac;

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&fromMac));

        // Connect Wait
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanMaxClientConnectTimeOut, 1) > 0);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectStatus));
        WLANTEST_ASSERT_TRUE(connectStatus.channel == testMasterBss.channel);
        WLANTEST_ASSERT_TRUE(connectStatus.ssid == testMasterBss.ssid);
        WLANTEST_ASSERT_TRUE(connectStatus.bssid == fromMac);

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, GetClientStatus_No89)
    {
        nn::os::SystemEvent conEvent;
        localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1);

        // Connect Wait
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(conEvent.GetBase()));
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(conEvent.GetBase(), nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap));
        WLANTEST_ASSERT_TRUE(clientStatusBitMap > 0);

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, GetState_No92)
    {
        // 正常値テストNo92 MASTER

        localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetState(&wlanState));
        WLANTEST_ASSERT_TRUE(wlanState >= nn::wlan::WlanState_LocalMasterIdle && wlanState <= nn::wlan::WlanState_LocalMasterBssScan);
        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, AddDeleteIe_No94_No97)
    {
        // 正常値テストNo94 MASTER
        static const uint8_t LocalVendorIe[] = {
            0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x78
        };
        uint32_t ieId;
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddIe(&ieId, nn::wlan::ManagementFrameType_Beacon, LocalVendorIe, sizeof(LocalVendorIe)));

        // 正常値テストNo97 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteIe(ieId));
    }

    TEST_F(LocalMasterNormalUnitTest, GetMacAddress_No100)
    {
        // 正常値テストNo100 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));
    }

    TEST_F(LocalMasterNormalUnitTest, ConvertRssiToLinkLevel_No105)
    {
        // 正常値テストNo105 MASTER
        nn::wlan::LinkLevel link;
        int32_t rssi = -30;

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1));

        // Connect Wait
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));

        link = nn::wlan::Local::ConvertRssiToLinkLevel(rssi);
        WLANTEST_ASSERT_TRUE(nn::wlan::LinkLevel_0 <= link && nn::wlan::LinkLevel_3 >= link);

        nn::wlan::DisconnectClient dis;
        dis.clientMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        dis.reasonCode = nn::wlan::Dot11ReasonCode_UnspecifiedFailure;
        nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &dis);

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, PutFrameRaw_No106)
    {
        // 正常値テストNo106 MASTER(Target CLIENT)
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestBufferSize]);
        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        // LocalMasterの通信開始
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));

        // Connect Wailt
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap));

        bool isClientConnected = false;
        for (int8_t i = nn::wlan::ConnectableClientsCountMax; i >= 0; i--)
        {
            if ((clientStatusBitMap >> i) & 0x00000001)
            {
                putMacAddress = clientStatus[i].clientMacAddress;
                isClientConnected = true;
                break;
            }
        }
        WLANTEST_ASSERT_TRUE(isClientConnected);

        std::memcpy(&putBuffer.get()[0], putMacAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
        std::memcpy(&putBuffer.get()[nn::wlan::MacAddress::MacAddressSize], macAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

        // Payload埋め
        // EtherType
        putBuffer.get()[HeaderSize]     = 0x08;
        putBuffer.get()[HeaderSize + 1] = 0x00;

        uint16_t ethertypes[] = { 0x80b7 };
        LocalApiClass::LocalMakeFrame(putBuffer.get(), WlanTestBufferSize, putMacAddress, macAddress, ethertypes[0], TestMatchInfo[0]);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), WlanTestBufferSize));

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, GetFrameRaw_No108)
    {
        // 正常値テストNo108 MASTER(Target CLIENT)
        size_t outSize = 0;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize * 3]);
        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));

        // Connect Wailt
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap));

        bool isClientConnected = false;
        for (int8_t i = nn::wlan::ConnectableClientsCountMax; i >= 0; i--)
        {
            if ((clientStatusBitMap >> i) & 0x00000001)
            {
                putMacAddress = clientStatus[i].clientMacAddress;
                isClientConnected = true;
                break;
            }
        }
        WLANTEST_ASSERT_TRUE(isClientConnected);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(getBuffer.get(), WlanTestBufferSize * 3, &outSize, localMaster.m_entRxId));

        WLANTEST_EXPECT_TRUE(outSize >= TestMatchInfo[0].matchLength);
        if (outSize >= TestMatchInfo[0].matchLength)
        {
            LocalFrameHeader* pFrameHeader = reinterpret_cast<LocalFrameHeader*>(getBuffer.get());
            LocalPayloadData* pPayload = reinterpret_cast<LocalPayloadData*>(pFrameHeader->data);
            WLANTEST_EXPECT_TRUE(std::memcmp(pPayload->matchData, TestMatchInfo[0].matchData, TestMatchInfo[0].matchLength) == 0);
        }

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, CancelFrameRaw_No111)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestBufferSize]);
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);
        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));

        WLANTEST_ASSERT_TRUE(localMaster.LocalGetFrame(WlanTestBufferSize));

        // 正常値テストNo111 MASTER(Target CLIENT)
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CancelGetFrame(localMaster.m_entRxId));

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, CreateDeleteRxEntry_No114_No117)
    {
        // 正常値テストNo114 MASTER
        uint16_t ethertypes[] = { 0x88b7 };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, 1, 30));

        // 正常値テストNo117 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalMasterNormalUnitTest, AddDeleteEthertype_No120_No123)
    {
        uint32_t rxId/*, outRxId*/;
        uint16_t ethertypes[] = { 0x88b7 };
        uint16_t addethertypes[] = { 0x88b8 };

        // 正常値テストNo120 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, 1, 30));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(rxId, addethertypes[0]));

        // 正常値テストNo123 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&rxId, ethertypes[0]));
    }

    TEST_F(LocalMasterNormalUnitTest, AddRemoveMatchData_No126_No129)
    {
        uint16_t ethertypes[] = { 0x88b7 };

        // 正常値テストNo126 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, 1, 30));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, TestMatchInfo[0]));

        // 正常値テストNo129 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::RemoveMatchingDataFromRxEntry(rxId, TestMatchInfo[0]));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&rxId, ethertypes[0]));
    }

    TEST_F(LocalMasterNormalUnitTest, SetActionFrameWithBeacon_No135)
    {
        size_t putSize = 1024;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        const uint8_t testMacAddress[nn::wlan::MacAddress::MacAddressSize] = {
            0x7F, 0x11, 0x11, 0x11, 0x11, 0x11
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1));

        WLANTEST_ASSERT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, nn::wlan::MacAddress(testMacAddress),
                                macAddress, TestEtherType[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), putSize));

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, CancelActionFrameWithBeacon_No136)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestBufferSize]);
        const uint8_t testMacAddress[nn::wlan::MacAddress::MacAddressSize] = {
            0x7F, 0x11, 0x11, 0x11, 0x11, 0x11
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1));

        WLANTEST_ASSERT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), WlanTestBufferSize, nn::wlan::MacAddress(testMacAddress),
            macAddress, TestEtherType[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));

        localMaster.LocalSetActionFrameWithBeacon(putBuffer.get(), WlanTestBufferSize);

        // 正常値テストNo136 MASTER(Target CLIENT)
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CancelActionFrameWithBeacon());

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, GetActionFrame_No137)
    {
        uint32_t actionRxId;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
        nn::wlan::MacAddress outMacAddress;
        size_t outSize;

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1, false));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectStatus));

        // Connect Wailt
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanMaxClientConnectTimeOut, 1) > 0);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap));

        // 正常値テストNo137 MASTER(Target CLIENT)
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), WlanTestBufferSize, &outSize, actionRxId));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, CreateDeleteRxEntryForActionFrame_No143_No146)
    {
        // 正常値テストNo143 MASTER
        uint32_t actionRxId;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        uint32_t frameTypeCount = sizeof(FrameTypes) / sizeof(FrameTypes[0]);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1, false));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, frameTypeCount, 10));

        // 正常値テストNo146 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterNormalUnitTest, AddDeleteSubtypeToRxEntryForActionFrame_No149_No152)
    {
        uint32_t actionRxId, outActionRxId;
        const uint16_t FrameTypes[] = {
            nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Beacon,
        };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, 1, 10));

        // 正常値テストNo149 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(actionRxId, nn::wlan::ActionFrameType_Local));

        // 正常値テストNo152 MASTER
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&outActionRxId, nn::wlan::ActionFrameType_Local));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
    }

    TEST_F(LocalMasterNormalUnitTest, SetMaxAssociationNumber_No155)
    {
        const uint32_t ClientMaxConnect = 1;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::SetMaxAssociationNumber(ClientMaxConnect));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1, false));

        // Connect Wailt
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanMaxClientConnectTimeOut, 1) > 0);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap));

        localMaster.LocalRelease();
    }

    INSTANTIATE_TEST_CASE_P(WithBssPattern,
        LocalMasterNormalBssUnitTest,
        ::testing::ValuesIn(CreateBssTestParams));

    INSTANTIATE_TEST_CASE_P(WithBssPattern,
        LocalMasterNormalConnectUnitTest,
        ::testing::ValuesIn(ConnectBssTestParams));

    extern "C" void nnMain()
    {
        int     argc = nn::os::GetHostArgc();
        char**  argv = nn::os::GetHostArgv();
        NN_LOG("argc=%d\n", argc);
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("argv[%d]=%s\n", argc, argv[i]);
        }

        const bool isEnabled = true;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        ::testing::InitGoogleTest(&argc, argv);

        int result = RUN_ALL_TESTS();
        nnt::Exit(result);
    }

} // namespace WlanTest
