﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace WlanTest{

    class LocalMasterBoundaryCreateBssUnitTest : public ::testing::Test
    {
    protected:
        nn::Result Result;
        nn::wlan::Ssid ssid;
        nn::Bit8 ssdid_test[Wlantest_ssidlengthmax];

        nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "012345678",  // Key
        };
        nn::wlan::MasterBssParameters bssParameters = {
            1,
            false,
            30,
            false,
            nn::wlan::RateSetLegacy_11gMask,
            nn::wlan::RateSetLegacy_11gMask,
            security,
            nn::wlan::Ssid(LocalMasterTestSSID),
            0
        };

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            memset(ssdid_test, 0, sizeof(ssdid_test));
            strcpy((char *)ssdid_test, "a");
            ssid.Set(ssdid_test, Ssidsizemin);

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::wlan::WlanState state;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetState(&state));
            if (state >= nn::wlan::WlanState_LocalMasterBss &&
                state <= nn::wlan::WlanState_LocalMasterBssScan)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
            }

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
        }
    };

    class LocalMasterBoundaryAddIeUnitTest : public LocalMasterBoundaryCreateBssUnitTest
    {
    protected:
        LocalApiClass localMaster;
        size_t     size = BufferSize100k;

        uint8_t ieBody[7] = { 0x00, 0x22, 0xAA, 0x02, 0xBB, 0x55, 0x16 };
        uint8_t ie_sizemax[256] = { 0x00 };

        size_t ie_size = 7;

        std::unique_ptr<uint32_t[]>  pTestMemUnit32;
        std::unique_ptr<uint8_t[]>   pTestMemUnit8;

        nn::wlan::ConnectionStatus connectionStatus;
        nn::Bit32 clientStatusBitMap;
        nn::wlan::MacAddress macAddress;
        nn::wlan::MacAddress putMacAddress;

        const uint8_t TestEtherType[2] = {
            0x80, 0xB7
        };
        const uint32_t HeaderSize = nn::wlan::MacAddress::MacAddressSize * 2;

        LocalMasterBoundaryAddIeUnitTest()  NN_NOEXCEPT
        {
            std::unique_ptr<uint32_t[]>pTestMemUnit32_temp(new uint32_t[size]);
            pTestMemUnit32 = std::move(pTestMemUnit32_temp);

            std::unique_ptr<uint8_t[]>pTestMemUnit8_temp(new uint8_t[size]);
            pTestMemUnit8 = std::move(pTestMemUnit8_temp);

        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_TRUE(pTestMemUnit32.get() != nullptr);
            WLANTEST_ASSERT_TRUE(pTestMemUnit8.get() != nullptr);

            LocalMasterBoundaryCreateBssUnitTest::SetUp();

            localMaster.SetupLocalMaster(testMasterBss, TestMatchInfo, 1);
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            localMaster.LocalRelease();

            LocalMasterBoundaryCreateBssUnitTest::TearDown();
        }
    };

    typedef LocalMasterBoundaryAddIeUnitTest        LocalMasterBoundaryDeleteIeUnitTest;
    typedef LocalMasterBoundaryAddIeUnitTest        LocalMasterBoundaryPutFrameRawUnitTest;
    typedef LocalMasterBoundaryAddIeUnitTest        LocalMasterBoundaryDisconnectUnitTest;
    typedef LocalMasterBoundaryCreateBssUnitTest    LocalMasterBoundarySetMaxAssociationNumber;

    class LocalMasterBoundarySetActionFrameWithBeaconUnitTest : public LocalMasterBoundaryAddIeUnitTest
    {
    protected:

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            LocalMasterBoundaryAddIeUnitTest::SetUp();
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            LocalMasterBoundaryAddIeUnitTest::TearDown();
        }

    };

    /*************************/
    // CreateBss Test
    /*************************/
    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No1)
    {
        bssParameters.channel = BssChannelmin - 1;
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No2)
    {
        bssParameters.channel = BssChannelmin;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No3)
    {
        bssParameters.channel = BssChannelmax + 1;
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No4)
    {
        bssParameters.channel = BssChannelmax;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No5)
    {
        bssParameters.inactivePeriod = InactivePeriodmin;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No6)
    {
        bssParameters.inactivePeriod = InactivePeriodmax + 1;
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No7)
    {
        bssParameters.inactivePeriod = InactivePeriodmax;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No8)
    {
        bssParameters.inactivePeriod = 30;
        bssParameters.supportedRates = static_cast<nn::wlan::RateSetLegacy>(nn::wlan::RateSetLegacy_Default - 1);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No9)
    {
        bssParameters.supportedRates = nn::wlan::RateSetLegacy_Default;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No10)
    {
        bssParameters.supportedRates = static_cast<nn::wlan::RateSetLegacy>(nn::wlan::RateSetLegacy_11gMask + 1);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No11)
    {
        bssParameters.supportedRates = nn::wlan::RateSetLegacy_11gMask;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No12)
    {
        bssParameters.basicRates = static_cast<nn::wlan::RateSetLegacy>(nn::wlan::RateSetLegacy_Default - 1);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No13)
    {
        bssParameters.basicRates = nn::wlan::RateSetLegacy_Default;
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No14)
    {
        bssParameters.basicRates = static_cast<nn::wlan::RateSetLegacy>(nn::wlan::RateSetLegacy_11gMask + 1);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No15)
    {
        bssParameters.basicRates = static_cast<nn::wlan::RateSetLegacy>(nn::wlan::RateSetLegacy_11gMask);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No16)
    {
        bssParameters.security.privacyMode = static_cast<nn::wlan::SecurityMode>(nn::wlan::SecurityMode_Open - 1);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No17)
    {
        bssParameters.security.privacyMode = nn::wlan::SecurityMode_Open;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No18)
    {
        bssParameters.security.privacyMode = static_cast<nn::wlan::SecurityMode>(nn::wlan::SecurityMode_StaticAes + 1);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No19)
    {
        bssParameters.security.privacyMode = nn::wlan::SecurityMode_StaticAes;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No20)
    {
        bssParameters.security.groupPrivacyMode = static_cast<nn::wlan::SecurityMode>(nn::wlan::SecurityMode_Open - 1);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No21)
    {
        bssParameters.security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No22)
    {
        bssParameters.security.groupPrivacyMode = static_cast<nn::wlan::SecurityMode>(nn::wlan::SecurityMode_StaticAes + 1);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No23)
    {
        bssParameters.security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No24)
    {
        bssParameters.security.keyIdx = Keyinxmax + 1;
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No25)
    {
        bssParameters.security.keyIdx = Keyinxmax;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No26)
    {
        memset(ssdid_test, 0, sizeof(ssdid_test));
        bssParameters.ssid.Set(ssdid_test, Ssidsizemin - 1);
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No27)
    {
        memset(ssdid_test, 0, sizeof(ssdid_test));
        ssdid_test[0] = 1;
        bssParameters.ssid.Set(ssdid_test, Ssidsizemin);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No28)
    {
        memset(ssdid_test, 1, Ssidsizemax);
        // SSIDクラスのSet関数がMAXの長さ以上の文字列の場合にMAXサイズまでの文字列にリサイズされ、戻り値はFALSEで返される
        WLANTEST_EXPECT_FALSE(bssParameters.ssid.Set(ssdid_test, Ssidsizemax + 1));
        // SSIDはMAXサイズの文字列に丸め込まれるためPASSで返ることを期待結果とする
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    TEST_F(LocalMasterBoundaryCreateBssUnitTest, CreateBss_No29)
    {
        memset(ssdid_test, 1, Ssidsizemax);
        bssParameters.ssid.Set(ssdid_test, Ssidsizemax);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(bssParameters));
    }

    /*************************/
    // Disconnect Test
    /*************************/
    TEST_F(LocalMasterBoundaryDisconnectUnitTest, DisconnectTest_No109)
    {
        // Disconnect
        nn::wlan::DisconnectClient disconnect = {
            nn::wlan::MacAddress::CreateBroadcastMacAddress(),
            nn::wlan::Dot11ReasonCode_Reserved
        };
        nn::Result result;

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanMaxClientWaitMsecTimeOut, 1) > 0);

        disconnect.reasonCode = static_cast<nn::wlan::Dot11ReasonCode>(nn::wlan::Dot11ReasonCode_Reserved - 1);
        result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &disconnect);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalMasterBoundaryDisconnectUnitTest, DisconnectTest_No110)
    {
        // Disconnect
        nn::wlan::DisconnectClient disconnect = {
            nn::wlan::MacAddress::CreateBroadcastMacAddress(),
            nn::wlan::Dot11ReasonCode_Reserved
        };
        nn::Result result;

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanMaxClientConnectTimeOut, 1) > 0);

        disconnect.reasonCode = nn::wlan::Dot11ReasonCode_Reserved;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &disconnect));
    }

    TEST_F(LocalMasterBoundaryDisconnectUnitTest, DisconnectTest_No111)
    {
        // Disconnect
        nn::wlan::DisconnectClient disconnect = {
            nn::wlan::MacAddress::CreateBroadcastMacAddress(),
            nn::wlan::Dot11ReasonCode_Reserved
        };
        nn::Result result;

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanMaxClientConnectTimeOut, 1) > 0);

        disconnect.reasonCode = static_cast<nn::wlan::Dot11ReasonCode>(nn::wlan::Dot11ReasonCode_Sized16 + 1);
        result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &disconnect);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(result));
    }

    TEST_F(LocalMasterBoundaryDisconnectUnitTest, DisconnectTest_No112)
    {
        // Disconnect
        nn::wlan::DisconnectClient disconnect = {
            nn::wlan::MacAddress::CreateBroadcastMacAddress(),
            nn::wlan::Dot11ReasonCode_Reserved
        };
        nn::Result result;

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanMaxClientConnectTimeOut, 1) > 0);

        disconnect.reasonCode = nn::wlan::Dot11ReasonCode_Sized16;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &disconnect));
    }

    /*************************/
    // AddIe Test
    /*************************/
    TEST_F(LocalMasterBoundaryAddIeUnitTest, AddIe_No113)
    {
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(nn::wlan::Local::AddIe(pTestMemUnit32.get(), static_cast<nn::wlan::ManagementFrameType>(Iemin - 1), ieBody, ie_size));
    }

    TEST_F(LocalMasterBoundaryAddIeUnitTest, AddIe_No114)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddIe(pTestMemUnit32.get(), Iemin, ieBody, ie_size));
    }

    TEST_F(LocalMasterBoundaryAddIeUnitTest, AddIe_No115)
    {
        nn::wlan::ManagementFrameType testType = static_cast<nn::wlan::ManagementFrameType>(Iemax + 1);
        nn::Result result;
        NN_LOG("         AddIe_No115->Frame Type : 0x%08x\n", testType);
        result = nn::wlan::Local::AddIe(pTestMemUnit32.get(), testType, ieBody, ie_size);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(result);
    }

    TEST_F(LocalMasterBoundaryAddIeUnitTest, AddIe_No116)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddIe(pTestMemUnit32.get(), nn::wlan::ManagementFrameType_AssocReq, ieBody, ie_size));
    }

    //TEST_F(LocalMasterBoundaryAddIeUnitTest, AddIe_No117)
    //{
    //  ie_size = -1;
    //  Result = nn::wlan::Local::AddIe(pTestMemUnit32.get(), nn::wlan::ManagementFrameType_AssocReq, ie_sizemax, ie_size);
    //  WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    //}

    TEST_F(LocalMasterBoundaryAddIeUnitTest, AddIe_No118)
    {
        ie_size = 1;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddIe(pTestMemUnit32.get(), nn::wlan::ManagementFrameType_AssocReq, ie_sizemax, ie_size));
    }

    TEST_F(LocalMasterBoundaryAddIeUnitTest, AddIe_No119)
    {
        ie_size = 256;
        Result = nn::wlan::Local::AddIe(pTestMemUnit32.get(), nn::wlan::ManagementFrameType_AssocReq, ie_sizemax, ie_size);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryAddIeUnitTest, AddIe_No120)
    {
        ie_size = 255;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddIe(pTestMemUnit32.get(), nn::wlan::ManagementFrameType_AssocReq, ie_sizemax, ie_size));
    }

    /*************************/
    // PutFrameRaw Test
    /*************************/
    TEST_F(LocalMasterBoundaryPutFrameRawUnitTest, PutFrameRaw_No127)
    {
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(nn::wlan::Local::PutFrameRaw(pTestMemUnit8.get(), 0));
    }

    TEST_F(LocalMasterBoundaryPutFrameRawUnitTest, PutFrameRaw_No128)
    {
        size_t putSize = 1024 * 1024;
        nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));

        // Connect Wailt
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));

        for (int8_t i = nn::wlan::ConnectableClientsCountMax; i >= 0; i--)
        {
            if ((clientStatusBitMap >> i) & 0x00000001)
            {
                putMacAddress = clientStatus[i].clientMacAddress;
                break;
            }
        }

        std::memcpy(&putBuffer.get()[0], putMacAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
        std::memcpy(&putBuffer.get()[nn::wlan::MacAddress::MacAddressSize], macAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

        uint16_t ethertypes[] = { 0x80b7 };
        LocalApiClass::LocalMakeFrame(putBuffer.get(), putSize, putMacAddress, macAddress, ethertypes[0], TestMatchInfo[0]);

        Result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), FrameSizemax + 1);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterBoundaryPutFrameRawUnitTest, PutFrameRaw_No129)
    {
        size_t putSize = 1024 * 1024;
        nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));

        // Connect Wailt
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));

        for (int8_t i = nn::wlan::ConnectableClientsCountMax; i >= 0; i--)
        {
            if ((clientStatusBitMap >> i) & 0x00000001)
            {
                putMacAddress = clientStatus[i].clientMacAddress;
                break;
            }
        }

        std::memcpy(&putBuffer.get()[0], putMacAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
        std::memcpy(&putBuffer.get()[nn::wlan::MacAddress::MacAddressSize], macAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

        uint16_t ethertypes[] = { 0x80b7 };
        LocalApiClass::LocalMakeFrame(putBuffer.get(), putSize, putMacAddress, macAddress, ethertypes[0], TestMatchInfo[0]);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), FrameSizemax));
    }

    /*************************/
    // SetActionFrameWithBeacon Test
    /*************************/
    TEST_F(LocalMasterBoundarySetActionFrameWithBeaconUnitTest, SetActionFrameWithBeaconTest_No184)
    {
        size_t putSize = 1024 * 1024;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);

        std::memcpy(&putBuffer.get()[0], putMacAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
        std::memcpy(&putBuffer.get()[nn::wlan::MacAddress::MacAddressSize], macAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

        // Payload埋め
        // EtherType
        std::memcpy(&putBuffer.get()[HeaderSize], TestEtherType, sizeof(TestEtherType));

        // APIでチェックされている形式
        putBuffer.get()[0] = 0x7F;
        putBuffer.get()[4] = nn::wlan::ActionFrameType_Beacon;

        Result = nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), 0);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterBoundarySetActionFrameWithBeaconUnitTest, SetActionFrameWithBeaconTest_No185)
    {
        size_t putSize = 1024 * 1024;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);

        std::memcpy(&putBuffer.get()[0], putMacAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
        std::memcpy(&putBuffer.get()[nn::wlan::MacAddress::MacAddressSize], macAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

        // Payload埋め
        // EtherType
        std::memcpy(&putBuffer.get()[HeaderSize], TestEtherType, sizeof(TestEtherType));

        // APIでチェックされている形式
        putBuffer.get()[0] = 0x7F;
        putBuffer.get()[4] = nn::wlan::ActionFrameType_Beacon;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), Setactionsizemin));

        localMaster.LocalRelease();
    }

    /*************************/
    // SetMaxAssociationNumber Test
    /*************************/
    TEST_F(LocalMasterBoundarySetMaxAssociationNumber, SetMaxAssociationNumberTest_No215)
    {
        gResult = nn::wlan::Local::SetMaxAssociationNumber(ConnectClientlimitmin - 1);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalMasterBoundarySetMaxAssociationNumber, SetMaxAssociationNumberTest_No216)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetMaxAssociationNumber(ConnectClientlimitmin));
    }

    TEST_F(LocalMasterBoundarySetMaxAssociationNumber, SetMaxAssociationNumberTest_No217)
    {
        gResult = nn::wlan::Local::SetMaxAssociationNumber(ConnectClientlimitmax + 1);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalMasterBoundarySetMaxAssociationNumber, SetMaxAssociationNumberTest_No218)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetMaxAssociationNumber(ConnectClientlimitmax));
    }

    extern "C" void nnMain()
    {
        int     argc = nn::os::GetHostArgc();
        char**  argv = nn::os::GetHostArgv();
        NN_LOG("argc=%d\n", argc);
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("argv[%d]=%s\n", argc, argv[i]);
        }

        const bool isEnabled = true;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        ::testing::InitGoogleTest(&argc, argv);

        int result = RUN_ALL_TESTS();
        nnt::Exit(result);
    }

} // namespace WlanTest
