﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define WLANTEST_GTESTLIB  // gtest有効/無効切り替え

#include <memory>
#include "../../Common/testWlan_UnitTest.h"
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#define PRINT_TSF_VALUE
#define SCAN_ON_CONNECTION

// ScanBufferサイズ
const size_t TestScanMinSize = 50 * 1024;
const size_t TestScanMaxSize = 100 * 1024;

namespace {

// Connect試験パラメータ定義
const int ssid_num = 9;
//nn::wlan::Security TestSecurityPattern[nn::wlan::ScanningSsidCountMax][testsecurtypattern_num];
nn::wlan::Security TestSecurityPattern[nn::wlan::ScanningSsidCountMax];

std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[TestScanMaxSize]);
nn::wlan::Ssid ssid;
nn::os::ThreadType scanStopThread;
nn::os::EventType scanStopEvent;
nn::wlan::ConnectionStatus connectionStatus;

nn::wlan::MacAddress myMacAddress;
nn::os::ThreadType cancelConnectThread;

int rssiValue = 0;
nn::wlan::LinkLevel linkLevelValue;

// GetConnectionEvent 接続イベント
nn::os::SystemEventType connectEvent;

// Scanタイプ定義
#define WLANTEST_ACTIVESCAN         nn::wlan::ScanType::ScanType_Active
#define WLANTEST_PASSIVESCAN        nn::wlan::ScanType::ScanType_Passive

// ブロードキャストアドレス生成定義
#define WLANTEST_BROADCASTADDRESS   nn::wlan::MacAddress::CreateBroadcastMacAddress()

// StartScanパラメータ初期値定義
#define WLANTEST_CHANNLEL_PARAM     {   0, 0, 0, 0, 0, 0, 0, 0, 0, 0,       \
                                        0, 0, 0, 0, 0, 0, 0, 0, 0, 0,       \
                                        0, 0, 0, 0, 0, 0, 0, 0, 0, 0,       \
                                        0, 0, 0, 0, 0, 0, 0, 0          }   \

// Wait時間
const int wlantest_waittime = 3000;

// TestName
#define NORMAL_TESTNAME   "Normal"
int32_t NormalTestNo = 1;
int32_t GetStateTestNo = 0;


// スレッドタイムアウト
const int32_t ThreadWaitTime = 10;
const int32_t ThreadTimeOut  = 30000 / ThreadWaitTime;

const int32_t ScanWaintTime = 1000;

// スレッドのスタック
const size_t ThreadStackSize = 4096;
const int32_t ThreadCount = 2;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ThreadCount][ ThreadStackSize ];

// 接続待機タイムアウト
const int32_t ConnectTimeOut = 10000;

// 接続キャンセル
const int32_t cancelRetryMaxCount = 20;
const int32_t cancelThreadWailt = 3;

// StartScanテストパラメータ構造体
struct ScanTestPattern
{
    const char* pTestCaseName;
    uint32_t scanSize;
    nn::wlan::ScanParameters scanParam;
};


static const uint8_t InfraMacAddress[ssid_num][nn::wlan::MacAddress::MacAddressSize] =
{
    { 0xa6,0x12,0x42,0x7e,0xf8,0x0e },
    { 0x10,0x6F,0x3F,0x82,0x74,0x48 },
    { 0x00,0x90,0xC7,0x88,0x0F,0xA7 },
    { 0x10,0x6F,0x3F,0x6A,0x37,0x30 },
    { 0x00,0x90,0xC7,0x88,0x0F,0xA7 },
    { 0x00,0x0D,0x02,0x55,0x08,0x85 },
    { 0x10,0x6F,0x3F,0x6A,0x37,0x30 },
    { 0x10,0x6F,0x3F,0x82,0x74,0x48 },
    { 0xa4,0x12,0x42,0x7e,0xf8,0x0f }
};

ScanTestPattern scanTestParam[] = {
    { "nn::wlan::Infra::StartScan() Small buffer and Active Scan NormalTest",
    TestScanMinSize,
    { WLANTEST_ACTIVESCAN,{ 1 }, 1, 120, -1, &ssid, 0, WLANTEST_BROADCASTADDRESS } },
    { "nn::wlan::Infra::StartScan() Big buffer and Active Scan NormalTest",
    TestScanMaxSize,
    { WLANTEST_ACTIVESCAN,{ 1 }, 1, 120, -1, &ssid, 0, WLANTEST_BROADCASTADDRESS } },
    { "nn::wlan::Infra::StartScan() Small buffer and Active Scan NormalTest",
    TestScanMinSize,
    { WLANTEST_ACTIVESCAN,{ 1 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
    { "nn::wlan::Infra::StartScan() Big buffer and Active Scan NormalTest",
    TestScanMaxSize,
    { WLANTEST_ACTIVESCAN,{ 1 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
    { "nn::wlan::Infra::StartScan() Small buffer and Passive Scan NormalTest",
    TestScanMinSize,
    { WLANTEST_PASSIVESCAN,{ 1 }, 1, 120, -1, &ssid, 0, WLANTEST_BROADCASTADDRESS } },
    { "nn::wlan::Infra::StartScan() Big buffer and Passive Scan NormalTest",
    TestScanMaxSize,
    { WLANTEST_PASSIVESCAN,{ 1 }, 1, 120, -1, &ssid, 0, WLANTEST_BROADCASTADDRESS } },
    { "nn::wlan::Infra::StartScan() Small buffer and Passive Scan NormalTest",
    TestScanMinSize,
    { WLANTEST_PASSIVESCAN,{ 1 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
    { "nn::wlan::Infra::StartScan() Big buffer and Passive Scan NormalTest",
    TestScanMaxSize,
    { WLANTEST_PASSIVESCAN,{ 1 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } }
};

// CancelConnect試験スレッド引き渡しパラメータ構造体
struct cancelEvent
{
    int32_t successCount;
    int32_t retryCount;
    nn::os::EventType connectingEvent;
    nn::os::EventType StopConnectingEvent;
};

/******************************************************/
//  機能試験で使用している定義群
/******************************************************/
#define SDEVEDEV // SDEV EDEVのみ有効とする 製品版の場合は本デファインを無効にすること

// Connect試験 SSID、MACAddress有効無効切り替え
#define CONNECT_SSIDTEST
#define CONNECT_MACADDRESSTEST

} // namespace

namespace WlanTest {

    nn::wlan::WlanState wlanState = nn::wlan::WlanState_Stop;
    nn::os::ThreadType wlanStateThread;
    const size_t threadStackSize = 20480;
    NN_OS_ALIGNAS_THREAD_STACK char  ThreadStack[threadStackSize];

// StopScan試験スレッド関数
void StopScanThread(void* arg) NN_NOEXCEPT
{
    nn::os::EventType* pStartEvent = static_cast<nn::os::EventType*>(arg);
    bool isStartEvent;

    for ( int tmVaue = 0; tmVaue < ThreadTimeOut; tmVaue += ThreadWaitTime )
    {
        isStartEvent = nn::os::TimedWaitEvent(pStartEvent, nn::TimeSpan::FromMilliSeconds(ThreadWaitTime));
        if (isStartEvent != true)
        {
            continue;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ScanWaintTime));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::StopScan());

        nn::os::ClearEvent(pStartEvent);
        break;
    }
}

// CancelConnect試験スレッド関数
void CancelConnectThread(void* arg) NN_NOEXCEPT
{
    nn::os::EventType* pStartEvent = static_cast<nn::os::EventType*>(arg);
    bool isStartEvent;
    int32_t ThreadWaitTime = 10;
    int32_t ThreadTimeOut = 30000 / ThreadWaitTime;
    int32_t ConnectWaintTime = 100;
    nn::wlan::ConnectionStatus connectionStatus;

    for (int tmVaue = 0; tmVaue < ThreadTimeOut; tmVaue += ThreadWaitTime)
    {
        isStartEvent = nn::os::TimedWaitEvent(pStartEvent, nn::TimeSpan::FromMilliSeconds(ThreadWaitTime));
        if (isStartEvent != true)
        {
            continue;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ConnectWaintTime));
        NN_LOG("             WIFI Chancel Connect Start\n");
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::CancelConnect());
        NN_LOG("             WIFI Chancel Connect End\n");
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
        {
            NN_LOG("             WlanTest: CHANCEL OK!!!\n");
            break;
        }

        nn::os::ClearEvent(pStartEvent);
        break;
    }
}

void ScanStateThread(void* arg) NN_NOEXCEPT
{
    NN_LOG("             StartStan\n");
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[0].scanSize, scanTestParam[0].scanParam));
}

static void WlanStartScan() NN_NOEXCEPT
{
    // Bufer
    const int  Buffer_size_100k = 100 * 1024;
    std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[Buffer_size_100k]);

    nn::os::CreateThread(&wlanStateThread, ScanStateThread, nullptr, ThreadStack, threadStackSize, nn::os::DefaultThreadPriority);
    nn::os::StartThread(&wlanStateThread);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    nn::wlan::Infra::GetState(&wlanState);
    if (nn::wlan::WlanState_InfraIdleScan == wlanState)
    {
        GetStateTestNo = 44;
        WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &GetStateTestNo);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&wlanState));
        NN_LOG("             GetState(): %d \n", wlanState);
        WLANTEST_ASSERT_TRUE(nn::wlan::WlanState_InfraIdleScan == wlanState);
    }
    else if (nn::wlan::WlanState_InfraStaScan == wlanState)
    {
        GetStateTestNo = 45;
        WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &GetStateTestNo);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&wlanState));
        NN_LOG("             GetState(): %d \n", wlanState);
        WLANTEST_ASSERT_TRUE(nn::wlan::WlanState_InfraStaScan == wlanState);
    }

    nn::os::WaitThread(&wlanStateThread);
    nn::os::DestroyThread(&wlanStateThread);
}

// 正常系試験関数
TEST(InfraNormalUnitTest, NormalWlanTest) NN_NOEXCEPT
{
    cancelEvent cancelInfo;
    //auto securityInfo = TestSecurityPattern[0][0];
    auto securityInfo = TestSecurityPattern[0];
    nn::wlan::Ssid dummySsid;
    nn::wlan::MacAddress dummyBssid;
    nn::wlan::Ssid connectSsid(WtestSsid[0]);
    int16_t channel[nn::wlan::ScanningSsidCountMax] = {3,13,6,48,6,13,48,13,64,0};                     // 接続先のChannel格納先
    uint32_t connect_AfterTest_SSID = 0;                                 // 有効なSSIDのCount値を保持（Connect後の試験のため）
    nn::Bit8 ssdid_test[32] = { 0 };

    // Connectテストパターン設定
    for (int i = 0; i < nn::wlan::ScanningSsidCountMax; i++)
    {
        TestSecurityPattern[i].privacyMode = Default_privacyMode[i];
        TestSecurityPattern[i].groupPrivacyMode = Default_groupPrivacyMode[i];
        TestSecurityPattern[i].keyIdx = Default_keyIdx[i];
        memcpy(TestSecurityPattern[i].key, Default_key[i], Wlantest_keylengthmax);
    }

    WLANTEST_ASSERT_TRUE(pTestScanData.get() != nullptr);

    // InitializeInfraManager試験
    WLANTEST_SET_TESTNAME( NORMAL_TESTNAME, &NormalTestNo );
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());

    // GetState試験 無線デバイスドライバーの初期化済み状態
    GetStateTestNo = 41;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &GetStateTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&wlanState));
    NN_LOG("             GetState(): %d \n", wlanState);
    WLANTEST_ASSERT_TRUE(nn::wlan::WlanState_Ready == wlanState);

    // FinalizeInfraManager試験
    NormalTestNo = 2;
    WLANTEST_SET_TESTNAME( NORMAL_TESTNAME, &NormalTestNo );
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());

    // OpenMode試験
    NormalTestNo = 3;
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());

    // GetState試験 インフラストラクチャモードのアイドル状態
    GetStateTestNo = 42;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &GetStateTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&wlanState));
    NN_LOG("             GetState(): %d \n", wlanState);
    WLANTEST_ASSERT_TRUE(nn::wlan::WlanState_InfraIdle == wlanState);

    // RequestSleep()試験
    NormalTestNo = 71;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::RequestSleep());

    // GetState試験 SLEEP状態
    GetStateTestNo = 58;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &GetStateTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&wlanState));
    NN_LOG("             GetState(): %d \n", wlanState);

    // RequestWakeUp()試験
    NormalTestNo = 68;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::RequestWakeUp());


    // CloseMode試験
    NormalTestNo = 4;
    WLANTEST_SET_TESTNAME( NORMAL_TESTNAME, &NormalTestNo );
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());

    // StartScan試験
    NormalTestNo = 5;
    ScanTestPattern scanNormalParam[] = {
        { "nn::wlan::Infra::StartScan() NormalTest",
            TestScanMaxSize,
            {WLANTEST_PASSIVESCAN, {1, 6, 11}, 0, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} }
    };
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());

    struct TestConnectParams
    {
        char SettingNo[4];
        nn::wlan::Ssid ssid;
        nn::wlan::MacAddress bssid;
        int16_t channel;
        nn::wlan::Security security;
        bool autoKeepAlive;
        int beaconLostTimeout;
    };

    for (auto scanInfo : scanTestParam)
    {
        WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);

        std::unique_ptr<uint8_t[]>pScanBuffer(new uint8_t[scanInfo.scanSize]);
        WLANTEST_ASSERT_TRUE(pScanBuffer.get() != nullptr);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pScanBuffer.get(), scanInfo.scanSize, scanInfo.scanParam));
    }

    nn::os::InitializeEvent(&scanStopEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);

    WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&scanStopThread, StopScanThread, &scanStopEvent, g_ThreadStack[0], ThreadStackSize, nn::os::DefaultThreadPriority));
    nn::os::StartThread(&scanStopThread);

    // ScanStop試験
    ScanTestPattern scanStopTestParam[] = {
        { "nn::wlan::Infra::StopScan() NormalTest",
            100 * 1024,
            {WLANTEST_ACTIVESCAN,{ 1, 6, 11 }, 0, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} }
    };

    for (auto scanInfo : scanStopTestParam)
    {
        WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);

        std::unique_ptr<uint8_t[]>pScanBuffer(new uint8_t[scanInfo.scanSize]);
        WLANTEST_ASSERT_TRUE(pScanBuffer.get() != nullptr);

        // ScanStopスレッドを動作させる
        nn::os::SignalEvent(&scanStopEvent);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pScanBuffer.get(), scanInfo.scanSize, scanInfo.scanParam));
    }

    // ScanStopスレッド後処理
    nn::os::WaitThread(&scanStopThread);
    nn::os::DestroyThread(&scanStopThread);
    nn::os::FinalizeEvent(&scanStopEvent);

    // StopScanの後に少し間をおく
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(wlantest_waittime));

    // WiFi接続準備
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanNormalParam[0].scanSize, scanNormalParam[0].scanParam));

    // Connect試験 SSID
#ifdef CONNECT_SSIDTEST
    NormalTestNo = 14;
    for (uint32_t i = 0; i < ssid_num; i++)
    {
        securityInfo = TestSecurityPattern[i];

        memcpy(ssdid_test, WtestSsid[i], sizeof(WtestSsid[i]));
        connectSsid.Set(ssdid_test, sizeof(ssdid_test));

        if (securityInfo.privacyMode != nn::wlan::SecurityMode::SecurityMode_StaticAes)
        {
            // SSID接続試験
            WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);

            char outputSsidString[nn::wlan::Ssid::SsidHexStringLengthMax];
            connectSsid.GetHexString(&outputSsidString[0]);

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectEvent));
            while (1)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Connect(connectSsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), channel[i], securityInfo, true));
                // GetConnectionEvent待機
                WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectEvent, nn::TimeSpan::FromMilliSeconds(ConnectTimeOut)));;
                nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
                if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
                // 3秒待たせる
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(wlantest_waittime));
            }
            nn::os::ClearSystemEvent(&connectEvent);

            // GetState試験 インフラストラクチャモードの接続確立状態
            GetStateTestNo = 43;
            WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &GetStateTestNo);
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&wlanState));
            NN_LOG("             GetState(): %d \n", wlanState);
            WLANTEST_ASSERT_TRUE(nn::wlan::WlanState_InfraSta == wlanState);

            // 接続切断
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(wlantest_waittime));
        }
    }
#endif
    // Connect試験 MAC Address
#ifdef CONNECT_MACADDRESSTEST
    connectSsid.Set("");
    NormalTestNo = 24;
    for (uint32_t i = 0; i < ssid_num; i++)
    {
        securityInfo = TestSecurityPattern[i];
        nn::wlan::MacAddress bssid(InfraMacAddress[i]);

        if (securityInfo.privacyMode != nn::wlan::SecurityMode::SecurityMode_StaticAes)
        {
            // MACアドレス接続試験
            WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);

            char outputMacString[nn::wlan::MacAddress::MacStringSize];
            char outputSsidString[nn::wlan::Ssid::SsidHexStringLengthMax]; connectSsid.GetHexString(&outputSsidString[0]);
            bssid.GetString(&outputMacString[0]);
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectEvent));
            WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(
                nn::wlan::Infra::Connect(connectSsid, nn::wlan::MacAddress(InfraMacAddress[i]), channel[i], securityInfo, true)));
            // 3秒待たせる
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(wlantest_waittime));
            nn::os::ClearSystemEvent(&connectEvent);
        }
    }
#endif

    // ***************************************************/
    //         ステータス試験（StartScanスレッド）
    // ***************************************************/
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());

    //WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[0].scanSize, scanTestParam[0].scanParam));

    NN_LOG("             *****************************\n");
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %d \n", wlanState);
    WlanStartScan();

    // ***************************************************/
    //         Connect状態
    // ***************************************************/

    nn::wlan::Ssid Ssid(WtestSsid[0]);
    nn::os::SystemEventType connectionEvent;
    nn::wlan::Infra::GetConnectionEvent(&connectionEvent);
    while (1)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Connect(Ssid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), channel[0], TestSecurityPattern[0], true));
        nn::os::WaitSystemEvent(&connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
        {
            break;
        }
    }
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %d \n", wlanState);

    GetStateTestNo = 43;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &GetStateTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&wlanState));
    NN_LOG("             GetState(): %d \n", wlanState);
    WLANTEST_ASSERT_TRUE(nn::wlan::WlanState_InfraSta == wlanState);

    WlanStartScan();

    NN_LOG("             *****************************\n");

    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());

    // Connect後試験準備
    connectSsid.Set(WtestSsid[connect_AfterTest_SSID]);
    securityInfo = TestSecurityPattern[connect_AfterTest_SSID];//???
    NormalTestNo = 36;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectEvent));
    while (1)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(connectSsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), channel[0], securityInfo, true));
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectEvent, nn::TimeSpan::FromMilliSeconds(ConnectTimeOut)));
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
        {
            break;
        }
    }

    // GetConnectionStatus試験 接続状態
    NormalTestNo = 38;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
    WLANTEST_EXPECT_TRUE(nn::wlan::ConnectionState_Connected == connectionStatus.state);

    // GetState試験
    NormalTestNo = 39;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&wlanState));
    WLANTEST_ASSERT_TRUE((0 <= wlanState) || (18 >= wlanState));

#ifdef Wlan_UnSupported
    // GetAllowedChannels試験
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    int16_t channels[nn::wlan::WirelessChannelsCountMax];
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetAllowedChannels(channels));
#endif

    // GetMacAddress試験
    NormalTestNo = 60;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetMacAddress(&myMacAddress));

    // GetRssi試験
    NormalTestNo = 61;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetRssi(&rssiValue));

    // GetLinkLevel試験
    NormalTestNo = 62;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetLinkLevel(&linkLevelValue));
    WLANTEST_ASSERT_TRUE((nn::wlan::LinkLevel::LinkLevel_0 >= linkLevelValue) || (nn::wlan::LinkLevel::LinkLevel_3 <= linkLevelValue));

    // ConvertRssiToLinkLevel試験
    NormalTestNo = 63;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    linkLevelValue = nn::wlan::Infra::ConvertRssiToLinkLevel(rssiValue);

    WLANTEST_ASSERT_TRUE((nn::wlan::LinkLevel::LinkLevel_0 >= linkLevelValue) ||
        (nn::wlan::LinkLevel::LinkLevel_3 <= linkLevelValue));

    // ChangeRxAntenna(RxAntennaPattern_0)試験
    NormalTestNo = 64;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::ChangeRxAntenna(nn::wlan::RxAntennaPattern::RxAntennaPattern_0));

    // ChangeRxAntenna(RxAntennaPattern_1)試験
    NormalTestNo = 65;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::ChangeRxAntenna(nn::wlan::RxAntennaPattern::RxAntennaPattern_1));

    // ChangeRxAntenna(RxAntennaPattern_Both)試験
    NormalTestNo = 66;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::ChangeRxAntenna(nn::wlan::RxAntennaPattern::RxAntennaPattern_Both));

    // GetFwVersion(char* pOutStr)試験
    NormalTestNo = 67;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    char sOutStr[256] = { 0 };
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetFwVersion(&sOutStr[0], sizeof(sOutStr)));

    // 接続切断
    NormalTestNo = 34;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());

    // GetConnectionEvent 切断イベント
    NormalTestNo = 37;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectEvent));

    // CancelConnect準備
    NormalTestNo = 38;
    nn::os::InitializeEvent(&cancelInfo.connectingEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&cancelInfo.StopConnectingEvent, true, nn::os::EventClearMode::EventClearMode_ManualClear);
    cancelInfo.successCount = 0;
    cancelInfo.retryCount = 0;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&cancelConnectThread, CancelConnectThread, &cancelInfo, g_ThreadStack[1], ThreadStackSize, nn::os::DefaultThreadPriority));
    nn::os::StartThread(&cancelConnectThread);

    // 接続するまでの時間が1秒未満と予想されるため、キャンセルが出来るか繰り返し実行確認する
    NormalTestNo = 35;
    WLANTEST_SET_TESTNAME(NORMAL_TESTNAME, &NormalTestNo);
    for (int32_t tryCout = 0; tryCout < cancelRetryMaxCount; tryCout++)
    {
        // CancelConnectスレッド動作させる
        nn::os::SignalEvent(&cancelInfo.connectingEvent);
        nn::Result conectResult = nn::wlan::Infra::Connect(connectSsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), channel[0], securityInfo, false);
        nn::os::TimedWaitSystemEvent(&connectEvent, nn::TimeSpan::FromMilliSeconds(ConnectTimeOut));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(cancelThreadWailt));
        if (conectResult.IsSuccess() == true)
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());
        }
    }

    // CancelConnectスレッド後処理
    nn::os::ClearEvent(&cancelInfo.StopConnectingEvent);
    nn::os::SignalEvent(&cancelInfo.connectingEvent);
    nn::os::WaitThread(&cancelConnectThread);
    nn::os::DestroyThread(&cancelConnectThread);
    nn::os::FinalizeEvent(&cancelInfo.connectingEvent);
    nn::os::FinalizeEvent(&cancelInfo.StopConnectingEvent);

    // 正常系試験後処理
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());

} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();
    NN_LOG("argc=%d\n", argc);
    for (int i = 0; i < argc; i++)
    {
        NN_LOG("argv[%d]=%s\n", argc, argv[i]);
    }

    ::testing::InitGoogleTest(&argc, argv);
    WLANTEST_STATE_SET(WlanTestState_Infra);

    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}

}
