﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "../Include/testWlan_LocalInfraStressTest.h"

#define STATIC_IP

namespace {
    NN_ALIGNAS(4096) uint8_t SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    const uint16_t EtherTypes[1] = { 0x88b7 };
};

namespace WlanTest {

    //class LocalClientStressTest : public ::testing::Test
    class LocalModeChangeStressTest : public ::testing::Test
    {
    protected:
        LocalApiClass localMaster;

        std::unique_ptr<uint8_t[]> getFrameBuffer;

        nn::wlan::MacAddress macAddress;
        nn::wlan::MacAddress connectMacArray[ClientMaxConnect];

        uint32_t rxId;
        uint32_t actionRxId;
        bool isCreBss;

        nn::os::ThreadType actionframeThread;
        nn::os::EventType actionframeEvent;

        nn::os::SystemEventType connectionEvent;

        nn::os::Tick sysTick;
        nn::os::Tick logOutTime;

        LocalConnectsInfo connectsInfo;

        int32_t sockOpen;

        nn::os::ThreadType scanThread;
        nn::wlan::ScanParameters unsyncScanParam;
        uint64_t intervalScanTime;
        bool isStartScan;
        const bool isEnabled = true;

        nn::wlan::MacAddress testPutMac;
        int16_t testChannel;

        enum OpenModeType
        {
            OpenMode_Init = 0,
            OpenMode_Master,
            OpenMode_Client,
            OpenMode_Spectator,
        };

        LocalApiClass localClient;
        nn::wlan::ConnectionStatus connectionStatus;
        nn::wlan::ConnectionStatus clientStatus;

        uint32_t clientCount;
        uint32_t myNumber;
        uint32_t nextNumber;
        uint32_t lastCount;
        uint32_t testPos;

        int64_t agingTime;
        int32_t waitClientCount;

    protected:
        LocalModeChangeStressTest() NN_NOEXCEPT
            : clientCount(0),
            myNumber(0),
            testPos(0)
        {
            nextNumber = 1;

            for (int32_t i = 0; i < (sizeof(TestMasterClient) / sizeof(TestMasterClient[0])); i++)
            {
                TestMasterClient[i].seqNumber = i + 1;
                nn::util::SNPrintf(TestMasterClient[i].ssidArray,
                        sizeof(TestMasterClient[i].ssidArray), "WlanTest_Local_%d", (i + 1));
            }
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_STATE_SET(WlanTestState_Local);

            agingTime       = GetAgintTime();
            waitClientCount = GetClientsCount();

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm",
                    "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(false);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::MinSocketAllocatorSize,
                kLocalConcurrencyLimit));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());

            localMaster.InitializeBtm();

            // デフォルトのジョイコン設定は０台
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNodeNone));
#ifdef WLAN_TEST_STATE_ASSER_STOP
            localMaster.StateThreadStart(Time100msec, TraceState);
#else
            NN_UNUSED(Time100msec);
#endif
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(&connectionEvent));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            localClient.LocalRelease();
            localMaster.LocalRelease();

            if (connectionEvent._state != nn::os::SystemEventType::State::State_NotInitialized)
            {
                nn::os::DestroySystemEvent(&connectionEvent);
            }

            if (rxId != static_cast<uint32_t>(-1))
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
            }

            if (actionRxId != static_cast<uint32_t>(-1))
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
            }

            if (isCreBss == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
                isCreBss = false;
            }

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());

#ifdef WLAN_TEST_STATE_ASSER_STOP
            localMaster.StateThreadStop();
#endif

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::socket::Finalize());
        }

        uint32_t GetSeqNumber() NN_NOEXCEPT
        {
            uint32_t seqNum = 0;
            nn::wlan::MacAddress myMacAddress;
            nn::Result result;

            if (ChangeMode(OpenModeType::OpenMode_Master) == 0)
            {
                NN_LOG("             ChangeMode Error\n");
                return seqNum;
            }

            result = nn::wlan::Local::GetMacAddress(&myMacAddress);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             GetMacAddress Error\n");
                return seqNum;
            }

            for (auto info : TestMasterClient)
            {
                nn::wlan::MacAddress compareMacAddress(info.sdevMacAddress);
                if (myMacAddress == compareMacAddress)
                {
                    seqNum = info.seqNumber;
                    break;
                }
            }

            return seqNum;
        }

        uint32_t GetFirstSeqNumber() NN_NOEXCEPT
        {
            testPos = 0;
            nextNumber = TestMasterClient[testPos].seqNumber;

            return nextNumber;
        }

        uint32_t GetNextSeqNumber()
        {
            ++testPos %= lastCount;
            nextNumber = TestMasterClient[testPos].seqNumber;

            return nextNumber;
        }

        bool isMasterSeqNumber() NN_NOEXCEPT
        {
            uint32_t myNum = GetSeqNumber();

            return (myNum == nextNumber);
        }

        nn::wlan::Ssid GetMasterSsid()
        {
            return nn::wlan::Ssid(TestMasterClient[testPos].ssidArray);
        }

        void StartClientConnect()
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());
        }

        OpenModeType GetMode() NN_NOEXCEPT
        {
            nn::wlan::WlanState state;
            nn::Result result;

            result = localClient.GetState(&state);
            if (result.IsSuccess() != true)
            {
                return OpenModeType::OpenMode_Init;
            }

            if (nn::wlan::WlanState_LocalMasterIdle <= state && state < nn::wlan::WlanState_LocalClientIdle)
            {
                return OpenModeType::OpenMode_Master;
            }

            if (nn::wlan::WlanState_LocalClientIdle <= state && state < nn::wlan::WlanState_LocalSpectatorIdle)
            {
                return OpenModeType::OpenMode_Client;
            }

            if (nn::wlan::WlanState_LocalSpectatorIdle <= state && state < nn::wlan::WlanState_Exit)
            {
                return OpenModeType::OpenMode_Spectator;
            }

            return OpenModeType::OpenMode_Init;
        }

        bool ChangeMode(const OpenModeType mode) NN_NOEXCEPT
        {
            nn::Result result = nn::ResultSuccess();
            bool isModeOpen = false;

            if (CheckOpenMode(mode) != true)
            {
                isModeOpen = CloseMode();
                if (isModeOpen != true)
                {
                    return isModeOpen;
                }

                switch (mode)
                {
                case OpenModeType::OpenMode_Master:
                    result = nn::wlan::Local::OpenMasterMode();
                    NN_LOG("             Open Mode Master : 0x%08x\n", result.GetInnerValueForDebug());
                    break;
                case OpenModeType::OpenMode_Client:
                    result = nn::wlan::Local::OpenClientMode();
                    NN_LOG("             Open Mode Client : 0x%08x\n", result.GetInnerValueForDebug());
                    break;
                case OpenModeType::OpenMode_Spectator:
                    result = nn::wlan::Local::OpenSpectatorMode();
                    NN_LOG("             Open Mode Spectator : 0x%08x\n", result.GetInnerValueForDebug());
                    break;
                default:
                    return false;
                }

                if (result.IsSuccess() != true)
                {
                    return false;
                }
            }

            return true;
        }

        bool CheckOpenMode(const OpenModeType mode) NN_NOEXCEPT
        {
            if (GetMode() != mode)
            {
                return false;
            }

            return true;
        }

        bool CloseMode() NN_NOEXCEPT
        {
            nn::Result result = nn::ResultSuccess();
            switch (GetMode())
            {
            case OpenModeType::OpenMode_Master:
                result = nn::wlan::Local::CloseMasterMode();
                NN_LOG("             Close Master : 0x%08x\n", result.GetInnerValueForDebug());
                break;
            case OpenModeType::OpenMode_Client:
                result = nn::wlan::Local::CloseClientMode();
                NN_LOG("             Close Client : 0x%08x\n", result.GetInnerValueForDebug());
                break;
            case OpenModeType::OpenMode_Spectator:
                result = nn::wlan::Local::CloseSpectatorMode();
                NN_LOG("             Close Spectator : 0x%08x\n", result.GetInnerValueForDebug());
                break;
            default:
                break;
            }

            if (result.IsSuccess() != true)
            {
                return false;
            }

            return true;
        }

        void Connect(TestLocalConnectParams ConnectParams, uint32_t waitTime) NN_NOEXCEPT
        {
            TraceState(&localMaster);

            nn::wlan::ConnectionStatus connectionStatus;

            sysTick = nn::os::GetSystemTick();

            // 指定時間接続リトライ
            while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= waitTime)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ConnectParams.ssid,
                    ConnectParams.bssid,
                    ConnectParams.channel,
                    ConnectParams.security,
                    ConnectParams.autoKeepAlive,
                    ConnectParams.indication,
                    ConnectParams.beaconLostTimeout));

                if (nn::os::TimedWaitSystemEvent(&connectionEvent,
                        nn::TimeSpan::FromMilliSeconds(WlanTestConnectTimeOut)) != true)
                {
                    continue;
                }
                NN_LOG("             WiFi WaitSystemEvent Signal\n");

                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                NN_LOG("             WiFi ConnectionStatus(%d)\n", connectionStatus.state);
                if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             WlanTest: Connected\n");
                    break;
                }

                // 3秒待って再試行
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
            }

            TraceState(&localMaster);
        }

        void Disconnect() NN_NOEXCEPT
        {
            TraceState(&localMaster);

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            NN_LOG("             GetConnectionStatus(%d)\n", connectionStatus.state);
            if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
            {
                NN_LOG("             Disconnect Start\n");
                nn::wlan::LocalCommunicationMode ComMode = nn::wlan::LocalCommunicationMode_ClientSpectator;
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(ComMode, nullptr));
                NN_LOG("             Disconnect End\n");
            }

            TraceState(&localMaster);
            while (NN_STATIC_CONDITION(1))
            {
                nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(PutInterval));
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             DIS CONNECTED!!!\n");
                    break;
                }
            }
            TraceState(&localMaster);
        }

        void MasterSetup(int32_t clientMacListCount) NN_NOEXCEPT
        {
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[LocalApiFramePacketSize]);
            TcpMacAddressPacketFmt* pPacketData = reinterpret_cast<TcpMacAddressPacketFmt*>(pTestBuffer.get());
            int32_t offsetPos;
            WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

            const nn::wlan::MasterBssParameters fncTestMasterBss = {
                11, false, 30, true, nn::wlan::RateSetLegacy_11bMask, nn::wlan::RateSetLegacy_11bMask,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key },
                nn::wlan::Ssid("LocalGetTcp"), 100
            };

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));

            WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo,
                    sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
            WLANTEST_ASSERT_TRUE(localMaster.SetupTcpMaster(tcpMaster, clientMacListCount));

            // MASTER CLIENT接続待機
            WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray,
                    WlanTestConnectTimeOut, waitClientCount) > 0);
            WLANTEST_ASSERT_TRUE(localMaster.TcpMasterAcceptWait(Time1h));

            offsetPos = 0;
            lastCount = clientMacListCount + 1;
            pPacketData->len = 1;
            std::memcpy(pPacketData->macAddress[offsetPos], localMaster.m_macAddress.GetOuiData(),
                    sizeof(pPacketData->macAddress[offsetPos]));
            std::memcpy(TestMasterClient[offsetPos].sdevMacAddress, pPacketData->macAddress[offsetPos],
                    sizeof(TestMasterClient[offsetPos].sdevMacAddress));
            offsetPos++;

            for (int32_t i = 0; i < clientMacListCount; i++)
            {
                char clientMacArray[nn::wlan::MacAddress::MacStringSize];
                pPacketData->len++;
                std::memcpy(pPacketData->macAddress[offsetPos], localMaster.m_clientMacArray[i].GetOuiData(),
                        sizeof(pPacketData->macAddress[offsetPos]));
                std::memcpy(TestMasterClient[offsetPos].sdevMacAddress, pPacketData->macAddress[offsetPos],
                        sizeof(TestMasterClient[offsetPos].sdevMacAddress));
                NN_LOG("             LocalApi MasterSetup No%d.MacAddress : %s\n",
                        TestMasterClient[offsetPos].seqNumber,
                        nn::wlan::MacAddress(TestMasterClient[offsetPos].sdevMacAddress).GetString(clientMacArray));
                offsetPos++;
            }

            localMaster.TcpMasterSend(pTestBuffer.get(), LocalApiFramePacketSize);

            localMaster.LocalRelease();
        }

        void ClientSetup() NN_NOEXCEPT
        {
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[LocalApiFramePacketSize]);
            TcpMacAddressPacketFmt* pPacketData = reinterpret_cast<TcpMacAddressPacketFmt*>(pTestBuffer.get());

            IpV4Address tcpClient = {
                "0.0.0.0", "255.255.255.0", "192.168.11.1",
                "8.8.8.8", "8.8.8.9", 0, false
            };

            nn::util::SNPrintf(tcpClient.strIpAddress, sizeof(tcpClient.strIpAddress), "192.168.11.%d", GetOct4IpAddress());
            WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

            LocalClientConnectParameter ConnectParam = {
                nn::wlan::Ssid("LocalGetTcp"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key },
                true, nn::wlan::BeaconIndication_Enable, 10
            };

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Client));

            WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security,
                    TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
            WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParam, static_cast<int64_t>(Time60s)));

            WLANTEST_ASSERT_TRUE(localClient.SetupTcpClient(tcpMaster, tcpClient));
            WLANTEST_ASSERT_TRUE(localClient.TcpClientConnectAndRecv(pTestBuffer.get(), LocalApiFramePacketSize, Time1h));

            lastCount = pPacketData->len;
            for (int32_t i = 0; i < pPacketData->len; i++)
            {
                char clientMacArray[nn::wlan::MacAddress::MacStringSize];
                std::memcpy(TestMasterClient[i].sdevMacAddress, pPacketData->macAddress[i], sizeof(TestMasterClient[i].sdevMacAddress));
                NN_LOG("             LocalApi ClientSetup No%d.MacAddress : %s\n", TestMasterClient[i].seqNumber,
                        nn::wlan::MacAddress(TestMasterClient[i].sdevMacAddress).GetString(clientMacArray));
            }

            WLANTEST_ASSERT_TRUE(localClient.LocalClientDisconnect(Time60s));

            localClient.LocalRelease();
        }

        void MasterFrameTest(bool& isStart, nn::os::Tick& outTime, size_t bufferSize, int64_t agingTime) NN_NOEXCEPT
        {
            WLANTEST_ASSERT_TRUE(bufferSize > 0);

            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[bufferSize]);
            const int32_t TestConnectCount = lastCount;
            const int32_t TestClientCount = TestConnectCount - 1;
            const int32_t TestPutFrameCount = Time30s / Time5s;
            char macArray[nn::wlan::MacAddress::MacStringSize];
            LocalConnectsInfo conectsInfo;
            uint32_t seqNumber = 0;
            bool isSygnalWait = false;
            nn::Result result;

            WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
            char fromMac[nn::wlan::MacAddress::MacStringSize];
            char toMac[nn::wlan::MacAddress::MacStringSize];
#endif

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));

            // Master起動
            nn::wlan::MasterBssParameters testChangeMasterBss = {
                11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, GetMasterSsid(), 100
            };

            NN_LOG("             LocalMaster SSID : %s\n", GetMasterSsid().GetSsidData());

            WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testChangeMasterBss, TestMatchInfo,
                    sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
            NN_LOG("             LocalMaster Load(%s)\n", localMaster.m_macAddress.GetString(macArray));

            // MASTER受信スレッド起動
            WLANTEST_EXPECT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, bufferSize,
                    TestClientCount, TestClientCount, TestPutFrameCount));
            isSygnalWait = localMaster.LocalReceiveWait(Time600s, true);
            if (isStart != true)
            {
                outTime = nn::os::GetSystemTick();
                localMaster.LocalSetElapseTime();
                isStart = true;
            }

            // エージング時間が経過していたら終了条件。以外はローカルクライアント台数が揃ってのシグナルか判定
            if ((nn::os::GetSystemTick() - outTime).ToTimeSpan().GetMilliSeconds() > agingTime)
            {
                isSygnalWait = false;
            }
            else
            {
                WLANTEST_EXPECT_TRUE((isSygnalWait == true));
            }

            seqNumber = 0;
            while (isSygnalWait == true)
            {
                WLANTEST_EXPECT_TRUE(localMaster.LocalMakeFrame(putBuffer.get(), bufferSize,
                        nn::wlan::MacAddress::CreateBroadcastMacAddress(), localMaster.m_macAddress,
                        EtherTypes[0], TestMatchInfo[0], seqNumber));
                result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), bufferSize);

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
                NN_LOG("            LocalFunctionTest L115 Master PutFrameRaw From MacAddress[%s] To MacAddress[%s] : Seq Number : %lu\n",
                    localMaster.m_macAddress.GetString(fromMac), localMaster.m_clientMacArray[i].GetString(toMac), seqNumber);
#endif
                if (nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromMilliSeconds(Time5s)) == true)
                {
                    if (localMaster.GetStateClientCount(nn::wlan::ConnectionState_Connected) == 0)
                    {
                        break;
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(PutInterval));
                }

                // ブロードキャスト送信でエラーとなった場合イベントを見てから判定に変更
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
                seqNumber++;
            }

            localMaster.LocalTraceConnectsInfo(true);

            // MASTER受信スレッド停止
            localMaster.LocalReceiveFrameStop();

            // 通信結果ログ出力
            localMaster.LocalTraceConnectsInfo(true);

            localMaster.LocalGetConnectInfo(&conectsInfo);
            WLANTEST_EXPECT_TRUE(conectsInfo.connectCount >= TestClientCount);

            localMaster.LocalRelease();
        }

        void ClientFrameTest(bool& isStart, nn::os::Tick& outTime, size_t bufferSize, int64_t agingTime) NN_NOEXCEPT
        {
            char macArray[nn::wlan::MacAddress::MacStringSize];
            nn::wlan::MacAddress myMacAddress;
            LocalConnectsInfo conectsInfo;

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Client));

            TestLocalConnectParams clientChangeConnectParam = {
                GetMasterSsid(), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
            };

            NN_LOG("             LocalClient SSID : %s\n", GetMasterSsid().GetSsidData());
            WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(clientChangeConnectParam.ssid,
                    clientChangeConnectParam.security, TestMatchInfo,
                    sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
            Connect(clientChangeConnectParam, WlanTestMasterWaitTimeOut);

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&myMacAddress));
            NN_LOG("             LocalClient Load(%s)\n", myMacAddress.GetString(macArray));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

            // 受信スレッド起動
            NN_LOG("             LocalClient Receive Frame\n");
            WLANTEST_EXPECT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k,
                    bufferSize, 1, 1, WlanTestGetFrameCount));
            // MASTER接続待機
            NN_LOG("             LocalClient Master Start Frame Wait\n");
            bool isRcvStart = localClient.LocalReceiveWait(Time600s, Time1s, true);

            if (isStart != true)
            {
                outTime = nn::os::GetSystemTick();
                localClient.LocalSetElapseTime();
                isStart = true;
            }

            // エージング時間が経過していたら終了条件。以外はローカルクライアント台数が揃ってのシグナルか判定
            if ((nn::os::GetSystemTick() - outTime).ToTimeSpan().GetMilliSeconds() > agingTime)
            {
                isRcvStart = false;
            }
            else
            {
                WLANTEST_EXPECT_TRUE((isRcvStart == true));
            }

            // ローカルクライアント台数が揃わないまたは、エージング時間が経過していたら終了
            if (isRcvStart != true)
            {
                // 受信スレッド停止
                localClient.LocalReceiveFrameStop();
                Disconnect();
                localClient.LocalRelease();
                return;
            }

            // 送信スレッド起動
            NN_LOG("             LocalClient Send Frame\n");
            WLANTEST_EXPECT_TRUE(localClient.LocalSendFrameStart(bufferSize, Time30s, Time5s, true));

            nn::os::Tick sendTick = nn::os::GetSystemTick();
            while ((nn::os::GetSystemTick() - sendTick).ToTimeSpan().GetMilliSeconds() <= Time30s)
            {
                // MASTERからの切断イベント待機
                if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(PutInterval)) == true)
                {
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                    if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        break;
                    }
                }
            }

            // 送信スレッド停止
            localClient.LocalSendFrameStop();
            // 受信スレッド停止
            localClient.LocalReceiveFrameStop();
            // 通信結果ログ出力
            localClient.LocalTraceConnectsInfo(true);

            localClient.LocalGetConnectInfo(&conectsInfo);
            for (int32_t i = 0; i < conectsInfo.connectCount; i++)
            {
                WLANTEST_EXPECT_TRUE(conectsInfo.facing[i].isConnected);
                WLANTEST_EXPECT_TRUE(conectsInfo.facing[i].getFrameCount >= 1);
            }

            Disconnect();
            localClient.LocalRelease();
        }

        void MaseterAgingFrameTest(bool& isStart, size_t recvSize, size_t sendSize,
                int64_t sendInterval, int64_t connectTime) NN_NOEXCEPT
        {
            char macArray[nn::wlan::MacAddress::MacStringSize];
            LocalConnectsInfo conectsInfo;
            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[sendSize]);
            nn::os::Tick loopTick, sendTick;
            const int32_t TestConnectCount = lastCount;
            const int32_t TestClientCount = TestConnectCount - 1;
            uint32_t seqNumber = 0;

            WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));

            // Master起動
            nn::wlan::MasterBssParameters testChangeMasterBss = {
                6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, GetMasterSsid(), 100
            };

            NN_LOG("             LocalMaster SSID : %s\n", GetMasterSsid().GetSsidData());

            WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testChangeMasterBss,
                    TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
            NN_LOG("             LocalMaster Load(%s)\n", localMaster.m_macAddress.GetString(macArray));

            // MASTER受信スレッド起動
            localMaster.LocalReceiveFrameStart(BufferSize100k, recvSize,
                    TestClientCount, TestClientCount, WlanTestAgingGetFrameCount);

            localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true);
            if (isStart != true)
            {
                sysTick = nn::os::GetSystemTick();
                localMaster.LocalSetElapseTime();
                isStart = true;
            }

            seqNumber = 0;
            loopTick = nn::os::GetSystemTick();
            sendTick = loopTick;
            while ((nn::os::GetSystemTick() - loopTick).ToTimeSpan().GetMilliSeconds() <= connectTime)
            {
                if (((nn::os::GetSystemTick() - sendTick).ToTimeSpan().GetMilliSeconds()) > Time300msec)
                {
                    for (int32_t i = 0; i < localMaster.m_connectCount; i++)
                    {
                        WLANTEST_EXPECT_TRUE(localMaster.LocalMakeFrame(putBuffer.get(),
                                sendSize, localMaster.m_clientMacArray[i], localMaster.m_macAddress,
                                EtherTypes[0], TestMatchInfo[0]));
                        WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), sendSize, seqNumber));
                        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), sendSize));
                    }
                    sendTick = nn::os::GetSystemTick();
                    seqNumber++;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

                if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time600s)
                {
                    localMaster.LocalTraceConnectsInfo(true);
                    logOutTime = nn::os::GetSystemTick();
                }
            }

            // MASTER受信スレッド停止
            localMaster.LocalReceiveFrameStop();

            localMaster.LocalTraceConnectsInfo(true);

            localMaster.LocalGetConnectInfo(&conectsInfo);
            WLANTEST_EXPECT_TRUE(conectsInfo.connectCount >= TestClientCount);

            localMaster.LocalRelease();
            }

        void ClientAgingFrameTest(bool& isStart, size_t recvSize, size_t sendSize, int64_t sendInterval, int64_t connectTime) NN_NOEXCEPT
        {
            char macArray[nn::wlan::MacAddress::MacStringSize];
            nn::wlan::MacAddress myMacAddress;
            LocalConnectsInfo conectsInfo;
            const int32_t TestClientCount = lastCount - 1;

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Client));

            TestLocalConnectParams ConnectParam = {
                GetMasterSsid(), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
            };

            NN_LOG("             LocalClient SSID : %s\n", GetMasterSsid().GetSsidData());
            WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid,
                    ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
            Connect(ConnectParam, WlanTestMasterWaitTimeOut);

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&myMacAddress));
            NN_LOG("             LocalClient Load(%s)\n", myMacAddress.GetString(macArray));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

            // 受信スレッド起動
            WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k,
                    recvSize, 1, 1, WlanTestAgingGetFrameCount));
            // MASTER接続待機
            WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
            // 送信スレッド起動
            WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameStart(sendSize, connectTime + Time10s, Time300msec, true));

            if (isStart != true)
            {
                sysTick = nn::os::GetSystemTick();
                localClient.LocalSetElapseTime();
                isStart = true;
            }

            // MASTERからの接続が切られるまで待機
            while (NN_STATIC_CONDITION(1))
            {
                if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
                {
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                    if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        break;
                    }
                }

                if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time600s)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    logOutTime = nn::os::GetSystemTick();
                }
            }

            // 送信スレッド停止
            localClient.LocalSendFrameStop();
            // 受信スレッド停止
            localClient.LocalReceiveFrameStop();
            // 通信結果ログ出力
            localClient.LocalTraceConnectsInfo(true);

            localClient.LocalGetConnectInfo(&conectsInfo);
            WLANTEST_EXPECT_TRUE(conectsInfo.connectCount == TestClientCount);

            Disconnect();
            localClient.LocalRelease();
        }

        int GetRandom(int min, int max) NN_NOEXCEPT
        {
            static time_t check;
            int rand_ret;
            if (check != time(nullptr))
            {
                check = time(nullptr);
                srand(static_cast<unsigned int>(time(nullptr)));
            }
            rand_ret = min + static_cast<int>(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
            return rand_ret;
        }
    };// Class LocalClientStressTest

    typedef LocalModeChangeStressTest    LocalModeChangeMasterStressTest;
    typedef LocalModeChangeStressTest    LocalModeChangeClientStressTest;

    // 同一Masterに指定時間連続で接続と切断を繰り返し
    TEST_F(LocalModeChangeMasterStressTest, L115)
    {
        nn::os::Tick elapseTime;
        bool isStart = false;

        MasterSetup(waitClientCount);

        myNumber = GetSeqNumber();
        WLANTEST_ASSERT_TRUE(myNumber != 0);

        elapseTime = nn::os::GetSystemTick();
        GetFirstSeqNumber();
        while ((nn::os::GetSystemTick() - elapseTime).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            if (isMasterSeqNumber() == true)
            {
                MasterFrameTest(isStart, elapseTime, WlanTestPacketBufferSize, agingTime);
            }
            else
            {
                ClientFrameTest(isStart, elapseTime, WlanTestPacketBufferSize, agingTime);
            }
            GetNextSeqNumber();
        }

        WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));
    }

    TEST_F(LocalModeChangeClientStressTest, L115)
    {
        nn::os::Tick elapseTime;
        bool isStart = false;

        ClientSetup();

        myNumber = GetSeqNumber();
        WLANTEST_ASSERT_TRUE(myNumber != 0);

        elapseTime = nn::os::GetSystemTick();
        GetFirstSeqNumber();
        while ((nn::os::GetSystemTick() - elapseTime).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            if (isMasterSeqNumber() == true)
            {
                MasterFrameTest(isStart, elapseTime, WlanTestPacketBufferSize, agingTime);
            }
            else
            {
                ClientFrameTest(isStart, elapseTime, WlanTestPacketBufferSize, agingTime);
            }
            GetNextSeqNumber();
        }

        WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));
    }

}; // namespace WlanTest
