﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include "testWlan_localApiClass.h"
#include "testWlan_UnitTestCommon.h"

#include <nn/socket.h>
#include <nn/socket/socket_TypesPrivate.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>
#include <nn/nifm/nifm_ApiForNetworkMiddleware.h>

#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

namespace {

#define STATIC_IP

#ifdef STATIC_IP
#define STATIC_SUBNET_MASK      "255.255.255.0"
#define STATIC_GW_ADDR          "192.168.11.1"
#define STATIC_DNS_1            "8.8.8.8"
#define STATIC_DNS_2            "8.8.8.9"
#endif

    // TCP SERVER設定
#define WLAN_INTERFACE_NAME "wl0"

    NN_ALIGNAS(4096) uint8_t SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    // スレッドスタックサイズ定義
    const size_t StackSize                  = nn::os::ThreadStackAlignment * 4;

    const uint32_t ThreadWaitTime           = 10;

    const uint16_t FrameTypes[]             = { nn::wlan::ActionFrameType_Beacon };

    // BTMライブラリ関連変数
    const int64_t BtmBussyWait              = 100;

    bool IsBtmInit                          = false;

    const int64_t LocalSetupApWait          = 1800 * 1000;
    const int64_t LocalSetupConnectWait     = 1800 * 1000;

    nn::os::SystemEvent BtmSystemEvent;
}

namespace WlanTest {

    LocalAutoEvent::LocalAutoEvent() NN_NOEXCEPT
    {
        m_event._state = nn::os::EventType::State_NotInitialized;
        nn::os::InitializeMutex(&m_lock, false, 0);
    }

    LocalAutoEvent::~LocalAutoEvent() NN_NOEXCEPT
    {
        Release();
        nn::os::FinalizeMutex(&m_lock);
    }

    void LocalAutoEvent::SetEvent(nn::os::EventType& dscEvent) NN_NOEXCEPT
    {
        Release();
        if (dscEvent._state != nn::os::EventType::State_Initialized)
        {
            nn::os::InitializeEvent(&dscEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        }
        m_event = dscEvent;
    }

    void LocalAutoEvent::SetEvent(bool isSygnal, nn::os::EventClearMode mode) NN_NOEXCEPT
    {
        Release();
        nn::os::InitializeEvent(&m_event, false, mode);
    }

    void LocalAutoEvent::SetSygnal() NN_NOEXCEPT
    {
        if (m_event._state == nn::os::EventType::State_Initialized)
        {
            nn::os::SignalEvent(&m_event);
        }
    }

    void LocalAutoEvent::ClearSygnal() NN_NOEXCEPT
    {
        if (m_event._state == nn::os::EventType::State_Initialized)
        {
            nn::os::ClearEvent(&m_event);
        }
    }

    bool LocalAutoEvent::WaitSygnal() NN_NOEXCEPT
    {
        if (m_event._state == nn::os::EventType::State_Initialized)
        {
            nn::os::WaitEvent(&m_event);
            return true;
        }
        return false;
    }

    bool LocalAutoEvent::WaitSygnal(int64_t waitMsecTime) NN_NOEXCEPT
    {
        if (m_event._state == nn::os::EventType::State_Initialized)
        {
            return nn::os::TimedWaitEvent(&m_event, nn::TimeSpan::FromMilliSeconds(waitMsecTime));
        }
        return false;
    }

    bool LocalAutoEvent::TrySygnal() NN_NOEXCEPT
    {
        if (m_event._state == nn::os::EventType::State_Initialized)
        {
            return nn::os::TryWaitEvent(&m_event);
        }
        return false;
    }

    void LocalAutoEvent::Release() NN_NOEXCEPT
    {
        AutoLock lock(&m_lock);
        if (m_event._state == nn::os::EventType::State_Initialized)
        {
            nn::os::FinalizeEvent(&m_event);
        }
    }

    // Local通信ステート監視クラス
    LocalStateMonitoring::LocalStateMonitoring(bool isManagerInit) NN_NOEXCEPT
        :   m_isStateThread     (false)
    {
        m_isApiInit = isManagerInit;
        if (m_isApiInit != true)
        {
            nn::wlan::InitializeLocalManager();
        }

        nn::os::InitializeMutex(&m_lock, false, 0);
        m_func = nullptr;
    }

    LocalStateMonitoring::~LocalStateMonitoring() NN_NOEXCEPT
    {
        if (m_isApiInit != true)
        {
            nn::wlan::FinalizeLocalManager();
        }
        nn::os::FinalizeMutex(&m_lock);
    }

    nn::os::MutexType* LocalStateMonitoring::GetLock() NN_NOEXCEPT
    {
        return &m_lock;
    }

    nn::Result LocalStateMonitoring::GetState(nn::wlan::WlanState* pState) NN_NOEXCEPT
    {
        AutoLock lock(GetLock());

        return nn::wlan::Local::GetState(pState);
    }

    void LocalStateMonitoring::StateThreadStart(const int64_t interval, LocalStateCallBack pFunction) NN_NOEXCEPT
    {
        nn::Result result;
        m_func = pFunction;

        if (m_isStateThread == true)
        {
            return;
        }

        m_interval = interval;

        static NN_ALIGNAS(4096) uint8_t stateStack[StackSize];

        result = nn::os::CreateThread(&m_stateId, StateThread, this, stateStack, StackSize, nn::os::DefaultThreadPriority);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             LocalStateMonitoring CreateThread error(0x%08x)\n", result.GetInnerValueForDebug());
            return;
        }

        nn::os::StartThread(&m_stateId);
    }

    void LocalStateMonitoring::StateThreadStop()
    {
        if (m_isStateThread != true)
        {
            return;
        }

        m_isStateThread = false;
        nn::os::WaitThread(&m_stateId);
        nn::os::DestroyThread(&m_stateId);

        NN_LOG("             WLAN State Thread Free\n");
    }

    void LocalStateMonitoring::StateThread(void* arg) NN_NOEXCEPT
    {
        LocalStateMonitoring* pThis = static_cast<LocalStateMonitoring*>(arg);
        nn::wlan::WlanState oldState, newState;
        nn::os::Tick startTime;
        int64_t interval = pThis->m_interval;
        nn::Result result;

        result = pThis->GetState(&oldState);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             GetState Error(0x%08x)\n", result.GetInnerValueForDebug());
        }

        pThis->m_isStateThread = true;
        newState = oldState;
        while (pThis->m_isStateThread)
        {
            if ((nn::os::GetSystemTick() - startTime).ToTimeSpan().GetMilliSeconds() <= interval)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
                continue;
            }

            startTime = nn::os::GetSystemTick();
            result = pThis->GetState(&newState);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             GetState Error(0x%08x)\n", result.GetInnerValueForDebug());
                continue;
            }

            if (oldState != newState)
            {
                if (pThis->m_func != nullptr)
                {
                    pThis->m_func(oldState, newState);
                }
                oldState = newState;
            }
        }

    }

    // LocalApiClassクラス
    LocalApiClass::LocalApiClass(bool isManagerInit) NN_NOEXCEPT
    {
        m_isManagerInit = isManagerInit;
        m_setupMode = WLAN_TEST_LEFTSHIFT(TestMode_Init);
        m_eventType = EventType_Init;
        m_commandWaitTime = ThreadWaitTime;

        std::memset(&m_bssParam, 0x00, sizeof(nn::wlan::MasterBssParameters));
        std::memset(&m_etherList, 0x00, sizeof(m_etherList));

        m_etherList.etherTypes[0]  = TestEtherTypes[0];
        m_etherList.typeCount      = 1;

        if (m_isManagerInit != true)
        {
            nn::wlan::InitializeLocalManager();
        }

        nn::os::InitializeEvent(&m_commandEvent, false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeEvent(&m_receiveFrameEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_parseFrameEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);

        // ミューテックスを初期化する
        nn::os::InitializeMutex(&m_localApiLock, false, 0);
        nn::os::InitializeMutex(&m_localRcvLock, false, 0);

        m_elapsedTick = nn::os::GetSystemTick();

        m_tcpMasterEvent.SetEvent(false, nn::os::EventClearMode_ManualClear);

        std::memset(&m_tcpInfo, 0x00, sizeof(m_tcpInfo));
        m_tcpInfo.socket = -1;

        m_tcpInfo.tcpSize = 0;
        m_tcpInfo.pTcpBuffer = nullptr;

        for (int32_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++)
        {
            m_tcpInfo.clients[i].accept = -1;
        }

        m_isSendActionFrame = false;
        m_isActionFrame     = false;
        m_isSendFrame       = false;

        LocalClearSeqNumber();
    }

    LocalApiClass::~LocalApiClass() NN_NOEXCEPT
    {
        LocalRelease();

        if (m_isManagerInit != true)
        {
            nn::wlan::FinalizeLocalManager();
        }

        m_tcpMasterEvent.Release();

        nn::os::FinalizeMutex(&m_localApiLock);
        nn::os::FinalizeMutex(&m_localRcvLock);

        nn::os::FinalizeEvent(&m_commandEvent);
        nn::os::FinalizeEvent(&m_receiveFrameEvent);
        nn::os::FinalizeEvent(&m_parseFrameEvent);
    }

    void LocalApiClass::InitializeBtm() NN_NOEXCEPT
    {
        if (IsBtmInit != true)
        {
            NN_LOG("             nn::btm::InitializeBtmInterface START\n");
            nn::btm::InitializeBtmInterface();
            NN_LOG("             nn::btm::InitializeBtmInterface END\n\n");
            IsBtmInit = true;
            m_btmMode = nn::btm::WlanMode_None;
            NN_LOG("             nn::btm::RegisterSystemEventForConnectedDeviceCondition START\n");
            nn::btm::RegisterSystemEventForConnectedDeviceCondition(BtmSystemEvent.GetBase());
            NN_LOG("             nn::btm::RegisterSystemEventForConnectedDeviceCondition END\n\n");
        }
    }

    bool LocalApiClass::SetBtMode(int32_t btCount) NN_NOEXCEPT
    {
        return SetBtMode(btCount, LocalApiBtmWaitTime);
    }

    bool LocalApiClass::SetBtMode(int32_t btCount, int64_t waitMsecTime) NN_NOEXCEPT
    {
        if (btCount < LocalApiBtNodeNone || btCount > LocalApiBtNode8 || IsBtmInit != true)
        {
            NN_LOG("             LocalApi SetNode Parameter Argment Fail\n");
            return false;
        }

        nn::btm::WlanMode wlanMode = nn::btm::WlanMode_None;
        nn::os::Tick waitTick;
        nn::Result result;
        bool isResult = false;

        NN_LOG("             LocalApi %s ---> SetBtMode START\n", __FUNCTION__);
        // 設定済みの場合は一旦None設定で再起呼び出し(btCount引数がLocalApiBtNodeNoneの再起呼び出し時は無視)
        if (m_btmMode != nn::btm::WlanMode_None && btCount != LocalApiBtNodeNone)
        {
            // 再起呼び出しで設定ができない場合は終了
            if (SetBtMode(LocalApiBtNodeNone, waitMsecTime) == false)
            {
                NN_LOG("             LocalApi SetNode Recall Fail\n");
                return false;
            }
        }

        // BT最大接続設定
        if (btCount > LocalApiBtNode2)
        {
            // ジョイコン接続が２台を超える場合
            wlanMode = nn::btm::WlanMode_Local4;
        }
        else if (btCount > 0)
        {
            // ジョイコン接続が２台以下１台以上の場合
            wlanMode = nn::btm::WlanMode_Local8;
        }

        waitTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - waitTick).ToTimeSpan().GetMilliSeconds() <= waitMsecTime)
        {
            result = nn::btm::SetWlanMode(wlanMode);
            if (result.IsSuccess() == true)
            {
                isResult = true;
                break;
            }
            if (nn::btm::ResultBusy::Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(BtmBussyWait));
            }
        }
        NN_LOG("             LocalApi SetNode : %d Result : %08x\n", wlanMode, result.GetInnerValueForDebug());

        if (isResult == true)
        {
            NN_LOG("             LocalApi %s ---> SetBtMode Sygnal Wait\n", __FUNCTION__);
            nn::btm::DeviceConditionList condition;
            while ((nn::os::GetSystemTick() - waitTick).ToTimeSpan().GetMilliSeconds() < waitMsecTime)
            {
                BtmSystemEvent.Wait();
                nn::btm::GetConnectedDeviceCondition(&condition);
                isResult = (condition.wlanMode == wlanMode);
                if (isResult == true)
                {
                    m_btmMode = wlanMode;
                    break;
                }
            }
        }
        NN_LOG("             LocalApi %s ---> SetBtMode END\n", __FUNCTION__);

        return isResult;
    }

    LocalOpenModeType LocalApiClass::GetLocalOpenMode() NN_NOEXCEPT
    {
        nn::wlan::WlanState state;
        return GetLocalOpenMode(state);
    }

    LocalOpenModeType LocalApiClass::GetLocalOpenMode(nn::wlan::WlanState& state) NN_NOEXCEPT
    {
        nn::Result result;

#ifdef WLAN_TEST_STATE_ASSER_STOP
        result = this->GetState(&state);
#else
        result = nn::wlan::Local::GetState(&state);
#endif
        if (result.IsSuccess() != true)
        {
            return LocalOpenModeType::OpenMode_Init;
        }

        if (nn::wlan::WlanState_LocalMasterIdle <= state && state < nn::wlan::WlanState_LocalClientIdle)
        {
            return LocalOpenModeType::OpenMode_Master;
        }

        if (nn::wlan::WlanState_LocalClientIdle <= state && state < nn::wlan::WlanState_LocalSpectatorIdle)
        {
            return LocalOpenModeType::OpenMode_Client;
        }

        if (nn::wlan::WlanState_LocalSpectatorIdle <= state && state < nn::wlan::WlanState_Exit)
        {
            return LocalOpenModeType::OpenMode_Spectator;
        }

        return LocalOpenModeType::OpenMode_Init;
    }

    LocalOpenModeType LocalApiClass::LocalApiClass::GetLocalModeType() NN_NOEXCEPT
    {
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_MasterInit)) > 0)
        {
            return LocalOpenModeType::OpenMode_Master;
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ClientInit)) > 0)
        {
            return LocalOpenModeType::OpenMode_Client;
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_SpectatorInit)) > 0)
        {
            return LocalOpenModeType::OpenMode_Spectator;
        }

        return LocalOpenModeType::OpenMode_Init;
    }

    std::string LocalApiClass::GetStringOpenMode(const LocalOpenModeType mode) NN_NOEXCEPT
    {
        std::string strMode;
        switch (mode)
        {
        case LocalOpenModeType::OpenMode_Master:
            strMode = "OpenMaster";
            break;
        case LocalOpenModeType::OpenMode_Client:
            strMode = "OpenClient";
            break;
        case LocalOpenModeType::OpenMode_Spectator:
            strMode = "OpenSpectator";
            break;
        default:
            strMode = "Not Open";
            break;
        }

        return std::string(strMode.c_str());
    }

    bool LocalApiClass::AddEtherType(const uint16_t etherType) NN_NOEXCEPT
    {
        if (etherType < LocalApiEtherMinValue || m_etherList.typeCount >= LocalApiEtherMaxCount)
        {
            return false;
        }

        int32_t listPos = -1;

        for (int32_t i = 0; i < m_etherList.typeCount; i++)
        {
            if (etherType == m_etherList.etherTypes[i])
            {
                listPos = i;
                break;
            }
        }

        if (listPos == -1)
        {
            m_etherList.etherTypes[m_etherList.typeCount] = etherType;
            m_etherList.typeCount++;
        }

        return true;
    }

    void LocalApiClass::SetChannel(int16_t channel)
    {
        m_channel = channel;
    }

    void LocalApiClass::DeleteEhterType(const uint16_t etherType) NN_NOEXCEPT
    {
        if (etherType < LocalApiEtherMinValue || m_etherList.typeCount == 1)
        {
            return;
        }

        int32_t listPos = -1;

        for (int32_t i = 0; i < m_etherList.typeCount; i++)
        {
            if (etherType == m_etherList.etherTypes[i])
            {
                listPos = i;
                break;
            }
        }

        if (listPos != -1)
        {
            for (int32_t i = listPos; i < m_etherList.typeCount; i++)
            {
                if (i + 1 == m_etherList.typeCount)
                {
                    m_etherList.etherTypes[i] = 0;
                }
                else
                {
                    m_etherList.etherTypes[i] = m_etherList.etherTypes[i + 1];
                }
            }
            m_etherList.typeCount--;
        }

        return;
    }

    bool LocalApiClass::SetupLocalMaster(const nn::wlan::MasterBssParameters & bssParamer, const nn::wlan::ReceivedDataMatchInfo matchInfo[],
        uint32_t matchCount, bool isActionFrame) NN_NOEXCEPT
    {
        LocalRelease(true);
        nn::os::Tick bssTick;
        nn::Result result;

        bssTick = nn::os::GetSystemTick();
        m_isActionFrame = isActionFrame;

        if (m_isManagerInit != true)
        {
            result = nn::wlan::Local::OpenMasterMode();
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi Master OpenMode Result :0x%08x\n", result.GetInnerValueForDebug());
                return false;
            }
            NN_LOG("             LocalApi Master Open Time (%lld msec)\n", (nn::os::GetSystemTick() - bssTick).ToTimeSpan().GetMilliSeconds());
        }

        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_MasterInit));

        nn::wlan::Local::GetMacAddress(&m_macAddress);

        result = nn::wlan::Local::GetConnectionEvent(&m_connectEvent);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             LocalApi Master GetConnectionEvent Result :0x%08x\n", result.GetInnerValueForDebug());
            return false;
        }
        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_GetConnectEvent));

        std::memcpy(&m_bssParam, &bssParamer, sizeof(nn::wlan::MasterBssParameters));
        result = nn::wlan::Local::CreateBss(m_bssParam);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             LocalApi Master CreateBss Result :0x%08x\n", result.GetInnerValueForDebug());
            return false;
        }
        NN_LOG("             LocalApi Master CreateBss Time (%lld msec)\n\n", (nn::os::GetSystemTick() - bssTick).ToTimeSpan().GetMilliSeconds());

        LocalTraceBssInfo();
        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_MasterCreateBss));

        static NN_ALIGNAS(4096) uint8_t commandStack[StackSize];
        result = nn::os::CreateThread(&m_commandThread, CommandThread, this, commandStack, StackSize, nn::os::DefaultThreadPriority);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             LocalApi Master Command CreateThread Result :0x%08x\n", result.GetInnerValueForDebug());
            return false;
        }

        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_CreateThread));

        nn::os::StartThread(&m_commandThread);

        // DeleleRxEntryが呼ばれるまで使いまわせるため条件追加
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_CreateEntryId)) == 0)
        {
            result = nn::wlan::Local::CreateRxEntry(&m_entRxId, m_etherList.etherTypes, m_etherList.typeCount, 30);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi Master CreateRxEntry Result :0x%08x\n", result.GetInnerValueForDebug());
                return false;
            }
            SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_CreateEntryId));
            NN_LOG("             LocalApi Master CreateRxEntryID :%lu\n", m_entRxId);

            for (int32_t i = 0; i < matchCount; i++)
            {
                nn::wlan::Local::AddMatchingDataToRxEntry(m_entRxId, matchInfo[i]);
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalApi Master AddMatchingDataToRxEntry Result :0x%08x\n", result.GetInnerValueForDebug());
                    return false;
                }
            }
            NN_LOG("             LocalApi Master CreateRxEntry Time (%lld msec)\n", (nn::os::GetSystemTick() - bssTick).ToTimeSpan().GetMilliSeconds());
        }

        // DeleteRxEntryForActionFrameが呼ばれるまで使いまわせるため条件追加
        if (m_isActionFrame == true && (m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_CreateActionEntryId)) == 0)
        {
            result = nn::wlan::Local::CreateRxEntryForActionFrame(&m_actionRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi Master CreateRxEntryForActionFrame Result :0x%08x\n", result.GetInnerValueForDebug());
                return false;
            }
            SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_CreateActionEntryId));
            NN_LOG("             LocalApi Master CreateRxEntryForActionFrameID :%lu\n", m_actionRxId);
            NN_LOG("             LocalApi Master CreateRxEntryForActionFrame Time (%lld msec)\n", (nn::os::GetSystemTick() - bssTick).ToTimeSpan().GetMilliSeconds());
        }

        return true;
    }

    uint32_t LocalApiClass::LocalMasterConnectWait(int64_t waitMsecTime, uint32_t connectCount) NN_NOEXCEPT
    {
        nn::wlan::MacAddress clientArray[nn::wlan::ConnectableClientsCountMax];
        return LocalMasterConnectWait(clientArray, waitMsecTime, connectCount);
    }

    uint32_t LocalApiClass::LocalMasterConnectWait(nn::wlan::MacAddress clientArray[nn::wlan::ConnectableClientsCountMax], int64_t waitMsecTime, uint32_t connectCount) NN_NOEXCEPT
    {
        const int64_t eventWaitTime = 20;
        uint32_t clientConnectedCount = 0;
        nn::os::Tick waitTick;
        nn::Result result;

        waitTick = nn::os::GetSystemTick();

        NN_LOG("             Master Connect Wait Start(%lld msec)\n", (nn::os::GetSystemTick() - waitTick).ToTimeSpan().GetMilliSeconds());
        while ((nn::os::GetSystemTick() - waitTick).ToTimeSpan().GetMilliSeconds() < waitMsecTime)
        {
            if (nn::os::TimedWaitSystemEvent(&m_connectEvent, nn::TimeSpan::FromMilliSeconds(eventWaitTime)) == true)
            {
                clientConnectedCount = GetClientMacAddress(clientArray, nn::wlan::ConnectionState_Connected);
            }

            if (clientConnectedCount >= connectCount)
            {
                break;
            }
        }
        NN_LOG("             Master Connect Wait End(%lld msec)\n", (nn::os::GetSystemTick() - waitTick).ToTimeSpan().GetMilliSeconds());

        return clientConnectedCount;
    }

    uint32_t LocalApiClass::LocalMasterDisConnectWait(int64_t waitMsecTime) NN_NOEXCEPT
    {
        nn::wlan::MacAddress clientArray[nn::wlan::ConnectableClientsCountMax];
        return LocalMasterDisConnectWait(clientArray, waitMsecTime);
    }

    uint32_t LocalApiClass::LocalMasterDisConnectWait(nn::wlan::MacAddress clientArray[nn::wlan::ConnectableClientsCountMax], int64_t waitMsecTime) NN_NOEXCEPT
    {
        const int64_t eventWaitTime = 20;
        int64_t eventTime = std::min(eventWaitTime, waitMsecTime);
        uint32_t clientConnectedCount = 1;
        nn::os::Tick waitTick;
        nn::Result result;

        waitTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - waitTick).ToTimeSpan().GetMilliSeconds() < waitMsecTime)
        {
            if (nn::os::TimedWaitSystemEvent(&m_connectEvent, nn::TimeSpan::FromMilliSeconds(eventTime)) == true)
            {
                clientConnectedCount = GetClientMacAddress(clientArray, nn::wlan::ConnectionState_Connected);
            }
            if (clientConnectedCount == 0)
            {
                break;
            }
        }

        return clientConnectedCount;
    }

    bool LocalApiClass::GetMasterTcpMacAddress(nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax],
                                                uint32_t& macCount, IpV4Address masterIpAddr, uint32_t connectCount) NN_NOEXCEPT
    {
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[LocalApiFramePacketSize]);
        TcpMacAddressPacketFmt* pPacketData = reinterpret_cast<TcpMacAddressPacketFmt*>(pTestBuffer.get());
        nn::wlan::MacAddress putClientMacArray[nn::wlan::ConnectableClientsCountMax];
        bool isResult = false;

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_MasterInit)) != 0)
        {
            NN_LOG("             LocalApi GetMasterTcpMacAddress Setup mode Fail Line(%lu)\n", __LINE__);
            return false;
        }

        if (pTestBuffer.get() == nullptr)
        {
            NN_LOG("             LocalApi GetMasterTcpMacAddress Buffer Allocate Fail Line(%lu)\n", __LINE__);
            return false;
        }

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11bMask, nn::wlan::RateSetLegacy_11bMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid("LocalGetTcp"), 100
        };

        if (SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])) != true)
        {
            NN_LOG("             LocalApi SetupLocalMaster Error Line(%lu)\n", __LINE__);
            return false;
        }
        if (SetupTcpMaster(tcpMaster, connectCount) != true)
        {
            LocalRelease();
            NN_LOG("             LocalApi SetupTcpMaster Error Line(%lu)\n", __LINE__);
            return false;
        }

        // MASTER CLIENT接続待機
        if (LocalMasterConnectWait(putClientMacArray, LocalSetupApWait, connectCount) < connectCount)
        {
            TcpRelease();
            LocalRelease();
            NN_LOG("             LocalApi LocalMasterConnectWait TimeOut Line(%lu)\n", __LINE__);
            return false;
        }
        if (TcpMasterAcceptWait(LocalSetupConnectWait) != true)
        {
            TcpRelease();
            LocalRelease();
            NN_LOG("             LocalApi LocalSetupConnectWait TimeOut Line(%lu)\n", __LINE__);
            return false;
        }

        macCount = 0;
        pPacketData->len = 1;
        std::memcpy(pPacketData->macAddress[macCount], m_macAddress.GetOuiData(), sizeof(pPacketData->macAddress[macCount]));
        clientArray[macCount] = nn::wlan::MacAddress(pPacketData->macAddress[macCount]);
        macCount++;

        for (int32_t i = 0; i < connectCount; i++)
        {
            pPacketData->len++;
            std::memcpy(pPacketData->macAddress[macCount], putClientMacArray[i].GetOuiData(), sizeof(pPacketData->macAddress[macCount]));
            clientArray[macCount] = nn::wlan::MacAddress(pPacketData->macAddress[macCount]);
            macCount++;
        }

        isResult = TcpMasterSend(pTestBuffer.get(), LocalApiFramePacketSize);

        TcpRelease();
        LocalRelease();

        return isResult;
    }

    bool LocalApiClass::GetClientTcpMacAddress(nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax], uint32_t &macCount,
                                            IpV4Address masterIpAddr, IpV4Address clientIpAddr) NN_NOEXCEPT
    {
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[LocalApiFramePacketSize]);
        TcpMacAddressPacketFmt* pPacketData = reinterpret_cast<TcpMacAddressPacketFmt*>(pTestBuffer.get());
        bool isResult = false;

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ClientInit)) != 0)
        {
            NN_LOG("             LocalApi GetClientTcpMacAddress Setup mode Fail Line(%lu)\n", __LINE__);
            return false;
        }

        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid("LocalGetTcp"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        if (SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo,
            sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false) != true)
        {
            NN_LOG("             LocalApi SetupLocalClient Error Line(%lu)\n", __LINE__);
            return false;
        }

        if (LocalClientConnect(ConnectParam, static_cast<int64_t>(LocalSetupApWait)) != true)
        {
            LocalRelease();
            NN_LOG("             LocalApi LocalClientConnect Error Line(%lu)\n", __LINE__);
            return false;
        }

        if (SetupTcpClient(tcpMaster, clientIpAddr) != true)
        {
            TcpRelease();
            LocalRelease();
            NN_LOG("             LocalApi SetupTcpClient Error Line(%lu)\n", __LINE__);
            return false;
        }

        if (TcpClientConnectAndRecv(pTestBuffer.get(), LocalApiFramePacketSize, LocalSetupConnectWait) != true)
        {
            TcpRelease();
            LocalRelease();
            NN_LOG("             LocalApi TcpClientConnectAndRecv TimeOut Line(%lu)\n", __LINE__);
            return false;
        };

        macCount = pPacketData->len;
        for (int32_t i = 0; i < pPacketData->len; i++)
        {
            std::memcpy(clientArray[i].GetOuiData(), pPacketData->macAddress[i], nn::wlan::MacAddress::MacAddressSize);
        }

        isResult = LocalClientDisconnect(LocalSetupConnectWait);

        TcpRelease();
        LocalRelease();

        return isResult;
    }

    bool LocalApiClass::SetupTcpMaster(IpV4Address masterIpAddr, uint32_t connectCount) NN_NOEXCEPT
    {
        nn::socket::InAddr ipAddress = { 0 };
        std::string strIpAddress;
        nn::Result result;

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_MasterCreateBss)) == 0
            || (m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_TcpMasterInit)) != 0)
        {
            return false;
        }

        m_tcpInfo.socket = -1;
        std::memset(&m_tcpInfo.masterIp, 0x00, sizeof(m_tcpInfo.masterIp));
        std::memset(&m_tcpInfo.clientIp, 0x00, sizeof(m_tcpInfo.clientIp));
        std::memcpy(&m_tcpInfo.masterIp, &masterIpAddr, sizeof(IpV4Address));
        for (int32_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++)
        {
            m_tcpInfo.clients[i].accept = -1;
        }

        if (m_isManagerInit != true)
        {
            // socket ライブラリを初期化します。
            result = nn::socket::Initialize(
                SocketMemoryPoolBuffer,
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::DefaultSocketAllocatorSize,
                nn::socket::DefaultConcurrencyLimit);
            if (result.IsSuccess() != true)
            {
                return false;
            }
        }

        ConfigureInterface(WLAN_INTERFACE_NAME, m_tcpInfo.masterIp, true);

        if (nn::nifm::GetCurrentPrimaryIpAddress(&ipAddress).IsSuccess())
        {
            strIpAddress = nn::socket::InetNtoa(ipAddress);
        }
        else
        {
            strIpAddress = "No IpAddress";
        }
        NN_LOG("             Server IpAddress : %s (Port : %d)\n", strIpAddress.c_str(), m_tcpInfo.masterIp.port);

        static NN_ALIGNAS(4096) uint8_t tcpMasterStack[StackSize];

        m_tcpInfo.waitCount = connectCount;
        m_tcpMasterEvent.ClearSygnal();
        m_tcpInfo.pConnectEvent = m_tcpMasterEvent.GetEvent();
        m_tcpInfo.isThreadStop = false;

        result = nn::os::CreateThread(&m_tcpInfo.tcpThread, TcpMasterThread,
            &m_tcpInfo, tcpMasterStack, StackSize, nn::os::DefaultThreadPriority);

        if (result.IsSuccess() == true)
        {
            nn::os::StartThread(&m_tcpInfo.tcpThread);
            SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_TcpMasterInit));
            return true;
        }

        return false;
    }

    bool LocalApiClass::SetupTcpClient(IpV4Address masterIpAddr, IpV4Address clientIpAddr) NN_NOEXCEPT
    {
        nn::Result result;

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ClientConnect)) == 0
            || (m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_TcpClientInit)) != 0)
        {
            return false;
        }

        m_tcpInfo.socket = -1;
        std::memset(&m_tcpInfo.masterIp, 0x00, sizeof(m_tcpInfo.masterIp));
        std::memset(&m_tcpInfo.clientIp, 0x00, sizeof(m_tcpInfo.clientIp));

        std::memcpy(&m_tcpInfo.masterIp, &masterIpAddr, sizeof(IpV4Address));
        std::memcpy(&m_tcpInfo.clientIp, &clientIpAddr, sizeof(IpV4Address));

        m_tcpMasterEvent.ClearSygnal();
        m_tcpInfo.pConnectEvent = m_tcpMasterEvent.GetEvent();
        m_tcpInfo.isThreadStop = false;

        if (m_isManagerInit != true)
        {
            // socket ライブラリを初期化します。
            result = nn::socket::Initialize(
                SocketMemoryPoolBuffer,
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::DefaultSocketAllocatorSize,
                nn::socket::DefaultConcurrencyLimit);
            if (result.IsSuccess() != true)
            {
                return false;
            }
        }

        ConfigureInterface(WLAN_INTERFACE_NAME, m_tcpInfo.clientIp, true);

        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_TcpClientInit));
        return true;
    }

    bool LocalApiClass::TcpMasterAcceptWait(int64_t waitMsecTime) NN_NOEXCEPT
    {
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_MasterCreateBss)) == 0
            || (m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_TcpMasterInit)) == 0)
        {
            return false;
        }

        if (m_tcpInfo.connectCount >= nn::wlan::ConnectableClientsCountMax)
        {
            return false;
        }

        bool isResult = nn::os::TimedWaitEvent(m_tcpInfo.pConnectEvent, nn::TimeSpan::FromMilliSeconds(waitMsecTime));

        if (isResult == true)
        {
            SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_TcpMasterAccept));
        }
        else
        {
            m_tcpInfo.isThreadStop = true;
            for (int32_t i = 0; i < m_tcpInfo.waitCount; i++)
            {
                if (m_tcpInfo.clients[i].accept != -1)
                {
                    nn::socket::Shutdown(m_tcpInfo.clients[i].accept, nn::socket::ShutdownMethod::Shut_RdWr);
                    nn::socket::Close(m_tcpInfo.clients[i].accept);
                }
            }
            nn::socket::Close(m_tcpInfo.socket);
            m_tcpInfo.socket = -1;
            if (m_isManagerInit != true)
            {
                nn::socket::Finalize();
            }
        }

        nn::os::WaitThread(&m_tcpInfo.tcpThread);
        nn::os::DestroyThread(&m_tcpInfo.tcpThread);

        return isResult;
    }

    bool LocalApiClass::TcpMasterSend(uint8_t buffer[], size_t bufferSize) NN_NOEXCEPT
    {
        bool isResult = true;

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_MasterCreateBss)) == 0
            || (m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_TcpMasterAccept)) == 0)
        {
            return false;
        }

        for (int32_t i = 0; i < m_tcpInfo.connectCount; i++)
        {
            if (m_tcpInfo.clients[i].accept != -1)
            {
                int result = nn::socket::Send(m_tcpInfo.clients[i].accept, buffer, bufferSize, nn::socket::MsgFlag::Msg_None);
                if (result != bufferSize)
                {
                    NN_LOG("             LocalApi Tcp Send Error(%d)\n", nn::socket::GetLastError());
                    isResult = false;
                }
            }
        }

        return isResult;
    }

    bool LocalApiClass::TcpClientConnectAndRecv(uint8_t* pBuffer, size_t bufferSize, int64_t waitMsecTime) NN_NOEXCEPT
    {
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ClientConnect)) == 0
            || (m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_TcpClientInit)) == 0)
        {
            return false;
        }

        if (pBuffer == nullptr || bufferSize == 0)
        {
            return false;
        }

        nn::Result result;
        static NN_ALIGNAS(4096) uint8_t tcpClientStack[StackSize];

        m_tcpInfo.pTcpBuffer    = pBuffer;
        m_tcpInfo.tcpSize       = bufferSize;
        m_tcpInfo.isThreadStop  = false;
        m_tcpInfo.waitCount     = 0;
        result = nn::os::CreateThread(&m_tcpInfo.tcpThread, TcpClientThread,
                                        &m_tcpInfo, tcpClientStack, StackSize, nn::os::DefaultThreadPriority);
        nn::os::StartThread(&m_tcpInfo.tcpThread);

        bool isResult = nn::os::TimedWaitEvent(m_tcpInfo.pConnectEvent, nn::TimeSpan::FromMilliSeconds(waitMsecTime));

        m_tcpInfo.isThreadStop = true;
        if (isResult != true)
        {
            for (int32_t i = 0; i < m_tcpInfo.waitCount; i++)
            {
                if (m_tcpInfo.clients[i].accept != -1)
                {
                    nn::socket::Shutdown(m_tcpInfo.clients[i].accept, nn::socket::ShutdownMethod::Shut_RdWr);
                    nn::socket::Close(m_tcpInfo.clients[i].accept);
                }
            }
        }
        nn::os::WaitThread(&m_tcpInfo.tcpThread);
        nn::os::DestroyThread(&m_tcpInfo.tcpThread);

        return isResult;
    }

    void LocalApiClass::TcpRelease()
    {
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_TcpMasterInit)) == 0
            && (m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_TcpClientInit)) == 0)
        {
            return;
        }

        for (int32_t i = 0; i < m_tcpInfo.connectCount; i++)
        {
            if (m_tcpInfo.clients[i].accept != -1)
            {
                nn::socket::Shutdown(m_tcpInfo.clients[i].accept, nn::socket::ShutdownMethod::Shut_RdWr);
                nn::socket::Close(m_tcpInfo.clients[i].accept);
                m_tcpInfo.clients[i].accept = -1;
            }
        }

        if (m_tcpInfo.socket != -1)
        {
            nn::socket::Shutdown(m_tcpInfo.socket, nn::socket::ShutdownMethod::Shut_RdWr);
            nn::socket::Close(m_tcpInfo.socket);
            m_tcpInfo.socket = -1;
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_TcpClientInit)) != 0)
        {
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_TcpClientInit);
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_TcpMasterInit)) != 0)
        {
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_TcpMasterInit);
        }
    }

    bool  LocalApiClass::SetupLocalClient(const nn::wlan::Ssid& masterSsid, const nn::wlan::Security& security,
        const nn::wlan::ReceivedDataMatchInfo matchInfo[], uint32_t matchCount, bool isFrame, bool isActionFrame) NN_NOEXCEPT
    {
        nn::Result result;

        bool isSetupLocalResult = SetupLocalClient(masterSsid, security);

        m_isFrame = isFrame;
        m_isActionFrame = isActionFrame;

        if (isSetupLocalResult == true)
        {
            // DeleleRxEntryが呼ばれるまで使いまわせるため条件追加
            if (m_isFrame == true && (m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_CreateEntryId)) == 0)
            {
                result = nn::wlan::Local::CreateRxEntry(&m_entRxId, m_etherList.etherTypes, m_etherList.typeCount, 30);
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalApi Client CreateRxEntry Result :0x%08x\n", result.GetInnerValueForDebug());
                    return false;
                }
                SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_CreateEntryId));
                NN_LOG("             LocalApi Client CreateRxEntryID :%lu\n", m_entRxId);

                for (int32_t i = 0; i < matchCount; i++)
                {
                    result = nn::wlan::Local::AddMatchingDataToRxEntry(m_entRxId, matchInfo[i]);
                    if (result.IsSuccess() != true)
                    {
                        NN_LOG("             LocalApi Client AddMatchingDataToRxEntry Result :0x%08x\n", result.GetInnerValueForDebug());
                        return false;
                    }
                }
            }

            // DeleteRxEntryForActionFrameが呼ばれるまで使いまわせるため条件追加
            if (m_isActionFrame == true && (m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_CreateActionEntryId)) == 0)
            {
                result = nn::wlan::Local::CreateRxEntryForActionFrame(&m_actionRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10);
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalApi Client CreateRxEntryForActionFrame Result :0x%08x\n", result.GetInnerValueForDebug());
                    return false;
                }
                SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_CreateActionEntryId));
                NN_LOG("             LocalApi Client CreateRxEntryForActionFrameID :%lu\n", m_actionRxId);
            }
        }

        return isSetupLocalResult;
    }

    bool  LocalApiClass::SetupLocalClient(const nn::wlan::Ssid& masterSsid, const nn::wlan::Security& security) NN_NOEXCEPT
    {
        LocalRelease(true);

        nn::Result result;

        m_isActionFrame = false;
        m_isFrame = false;

        m_ssid = masterSsid;
        m_security = security;

        if (m_isManagerInit != true)
        {
            result = nn::wlan::Local::OpenClientMode();
            if (result.IsSuccess() != true)
            {
                return false;
            }
        }

        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_ClientInit));

        static NN_ALIGNAS(4096) uint8_t stack[StackSize];
        result = nn::os::CreateThread(&m_commandThread, CommandThread, this, stack, StackSize, nn::os::DefaultThreadPriority);
        if (result.IsSuccess() != true)
        {
            return false;
        }

        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_CreateThread));

        nn::os::StartThread(&m_commandThread);

        result = nn::wlan::Local::GetConnectionEvent(&m_connectEvent);
        if (result.IsSuccess() != true)
        {
            return false;
        }

        nn::wlan::Local::GetMacAddress(&m_macAddress);
        m_bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

        return true;
    }

    bool LocalApiClass::LocalPassiveScanResult(nn::wlan::Ssid compareSsid, uint16_t channel) NN_NOEXCEPT
    {
        return LocalPassiveScanResult(compareSsid, channel, 110);
    }

    bool LocalApiClass::LocalPassiveScanResult(nn::wlan::Ssid compareSsid, uint16_t channel, int32_t scnanTime) NN_NOEXCEPT
    {
        const size_t scanSize = 10 * 1024;
        std::unique_ptr<uint8_t[]> pScanBuffer(new uint8_t[scanSize]);
        nn::wlan::ScanIeMatchInfo info;
        nn::os::Tick scanTick;
        nn::Result result;
        bool isResult = false;

        nn::wlan::ScanParameters ScanParam = {
            nn::wlan::ScanType_Passive,{ 0 }, 1, scnanTime, 0, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };
        const uint16_t ScanAuto2GHz[] = {1, 6, 11};

        if (pScanBuffer == nullptr)
        {
            return false;
        }

        std::memset(pScanBuffer.get(), 0x00, scanSize);
        scanTick = nn::os::GetSystemTick();
        ScanParam.channelCount = 0;

        if (channel == nn::wlan::WirelessChannel_1ch ||
            channel == nn::wlan::WirelessChannel_6ch ||
            channel == nn::wlan::WirelessChannel_11ch)
        {
#ifdef WLAN_TEST_SCAN2GONECH
            NN_UNUSED(ScanAuto2GHz);
            ScanParam.channelList[ScanParam.channelCount] = channel;
            ScanParam.channelCount++;
#else
            // 2.4GHzスキャン
            for (auto setChannel : ScanAuto2GHz)
            {
                ScanParam.channelList[ScanParam.channelCount] = setChannel;
                ScanParam.channelCount++;
            }
#endif
        }
        else if (channel == nn::wlan::WirelessChannel_36ch ||
            channel == nn::wlan::WirelessChannel_40ch ||
            channel == nn::wlan::WirelessChannel_44ch ||
            channel == nn::wlan::WirelessChannel_48ch)
        {
            // 5GHzスキャン
            ScanParam.channelList[0] = channel;
            ScanParam.channelCount   = 1;
        }
        else
        {
            NN_LOG("             LocalApi Scan Channel Not Match Error CHANNEL %d\n", channel);
            return isResult;
        }

        result = nn::wlan::Local::StartScan(pScanBuffer.get(), scanSize, ScanParam, &info);
        if ((isResult = result.IsSuccess()) != true)
        {
            NN_LOG("             LocalApi StartScan Result :0x%08x Line : %d\n", result.GetInnerValueForDebug(), __LINE__);
            return isResult;
        }

        isResult = GetScanResult(compareSsid, pScanBuffer.get(), scanSize, channel);
        NN_LOG("             LocalApi StartScan Time : %lld msec\n\n", (nn::os::GetSystemTick() - scanTick).ToTimeSpan().GetMilliSeconds());
        return isResult;
    }

    bool LocalApiClass::LocalPassiveAllChannelScanResult(nn::wlan::Ssid compareSsid, uint16_t channel) NN_NOEXCEPT
    {
        const size_t scanSize = 10 * 1024;
        std::unique_ptr<uint8_t[]> pScanBuffer(new uint8_t[scanSize]);
        nn::wlan::ScanIeMatchInfo info;
        nn::os::Tick scanTick;
        nn::Result result;
        bool isResult = false;

        const nn::wlan::ScanParameters Scan2GHz = {
            nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 110, 0, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };
        const int16_t Scan5GHzChannel[] = {
            36, 40, 44, 48
        };

        if (pScanBuffer == nullptr)
        {
            return false;
        }

        // 2.4GHzスキャン
        std::memset(pScanBuffer.get(), 0x00, scanSize);
        scanTick = nn::os::GetSystemTick();
        result = nn::wlan::Local::StartScan(pScanBuffer.get(), scanSize, Scan2GHz, &info);
        if ((isResult = result.IsSuccess()) != true)
        {
            NN_LOG("             LocalApi AllChannel Result :0x%08x Line : %d\n", result.GetInnerValueForDebug(), __LINE__);
            return isResult;
        }

        isResult = GetScanResult(compareSsid, pScanBuffer.get(), scanSize);
        if (isResult != true)
        {
            // 5GHzスキャン（LDN仕様）
            for (auto nChannel : Scan5GHzChannel)
            {
                nn::wlan::ScanParameters Scan5GHz = {
                    nn::wlan::ScanType_Passive,{ nChannel }, 1, 110, 0, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
                };

                std::memset(pScanBuffer.get(), 0x00, scanSize);
                result = nn::wlan::Local::StartScan(pScanBuffer.get(), scanSize, Scan5GHz, &info);
                if ((isResult = result.IsSuccess()) != true)
                {
                    NN_LOG("             LocalApi AllChannel Result :0x%08x Line : %d\n", result.GetInnerValueForDebug(), __LINE__);
                    return isResult;
                }

                isResult = GetScanResult(compareSsid, pScanBuffer.get(), scanSize);
                if (isResult == true && channel == nChannel)
                {
                    break;
                }
            }
        }
        NN_LOG("             LocalApi AllChannel StartScan Time : %lld msec\n\n", (nn::os::GetSystemTick() - scanTick).ToTimeSpan().GetMilliSeconds());
        return isResult;
    }

    bool LocalApiClass::GetScanResult(nn::wlan::Ssid& ssid, uint8_t pScanBuffer[], size_t bufferSize) NN_NOEXCEPT
    {
        nn::wlan::BeaconScanResultReader resultReader(pScanBuffer);
        uint32_t bssCount = resultReader.GetCount();
        NN_UNUSED(bufferSize);

        for (uint32_t i = 0; i < bssCount; i++)
        {
            nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
            if (ssid == beacon.GetSsid())
            {
                return true;
            }
        }

        return false;
    }

    bool LocalApiClass::GetScanResult(nn::wlan::Ssid& ssid, uint8_t pScanBuffer[], size_t bufferSize, uint16_t channel) NN_NOEXCEPT
    {
        nn::wlan::BeaconScanResultReader resultReader(pScanBuffer);
        uint32_t bssCount = resultReader.GetCount();
        NN_UNUSED(bufferSize);

        for (uint32_t i = 0; i < bssCount; i++)
        {
            nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
            if (ssid == beacon.GetSsid() && beacon.GetChannel() == channel)
            {
                return true;
            }
        }

        return false;
    }

    bool LocalApiClass::LocalClientConnect(int64_t waitMsecTime, int16_t channel) NN_NOEXCEPT
    {
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ClientInit)) == 0)
        {
            return false;
        }

        LocalClientConnectParameter conParam;

        conParam.autoKeepAlive      = true;
        conParam.beaconLostTimeout  = 10;
        conParam.bssid              = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        conParam.channel            = channel;
        conParam.indication         = nn::wlan::BeaconIndication_Enable;
        conParam.ssid               = m_ssid;
        conParam.security           = m_security;

        return LocalClientConnect(conParam, waitMsecTime);
    }

    bool LocalApiClass::LocalClientConnect(LocalClientConnectParameter& conParam, int64_t waitMsecTime) NN_NOEXCEPT
    {
        if (GetLocalOpenMode() != LocalOpenModeType::OpenMode_Client)
        {
            return false;
        }

        nn::os::SystemEvent connectEvent;
        char strMacAddress[nn::wlan::MacAddress::MacStringSize];
        nn::wlan::ConnectionStatus connectionStatus;
        nn::os::Tick conTimeout;
        nn::Result result;

        result = nn::wlan::Local::GetConnectionEvent(connectEvent.GetBase());
        if (result.IsSuccess() != true)
        {
            NN_LOG("             LocalApi Client GetConnectionEvent Result :0x%08x Line : %d\n", result.GetInnerValueForDebug(), __LINE__);
            return false;
        }

        // 指定時間接続リトライ
        conTimeout = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - conTimeout).ToTimeSpan().GetMilliSeconds() <= waitMsecTime)
        {
            result = nn::wlan::Local::Connect(conParam.ssid,
                conParam.bssid,
                conParam.channel,
                conParam.security,
                conParam.autoKeepAlive,
                conParam.indication,
                conParam.beaconLostTimeout);

            if (result.IsSuccess() != true)
            {
                break;
            }

            if (nn::os::TimedWaitSystemEvent(connectEvent.GetBase(), nn::TimeSpan::FromMilliSeconds(LocalApiClientWait)) != true)
            {
                continue;
            }

            if (nn::wlan::Local::GetConnectionStatus(&connectionStatus).IsSuccess() == true)
            {
                if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_ClientConnect));
                    break;
                }
            }
        }

        conParam.bssid.GetString(strMacAddress);
        NN_LOG("             *** LocalClient Connect Parameters ***\n");
        NN_LOG("               SSID       : %s(LEN:%d)\n", conParam.ssid.GetSsidData(), conParam.ssid.GetLength());
        NN_LOG("               BSSID      : %s\n", strMacAddress);
        NN_LOG("               CHANNEL    : %d\n", conParam.channel);
        NN_LOG("               SECURITY G : %s\n", SecurityTypeString[conParam.security.groupPrivacyMode]);
        NN_LOG("               SECURITY P : %s\n", SecurityTypeString[conParam.security.privacyMode]);
        NN_LOG("               KEY INDEX  : %d\n", conParam.security.keyIdx);
        NN_LOG("               PS         : %s\n", conParam.security.key);
        NN_LOG("               AUTOKEEP   : %s\n", conParam.autoKeepAlive == true ? "Yes" : "No");
        NN_LOG("               INDICATION : %d\n", conParam.indication);
        NN_LOG("               STATE      : %s(%d)\n", WlanConnectionStateStr[connectionStatus.state], connectionStatus.state);
        NN_LOG("               Result     : 0x%x(Module:%d Desc:%d)\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription());
        NN_LOG("             *** LocalClient Connect Parameters ***\n\n");

        return true;
    }

    bool LocalApiClass::LocalClientAsyncConnect(int16_t channel) NN_NOEXCEPT
    {
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ClientConnect)) > 0)
        {
            return false;
        }
        m_channel = channel;
        SetCommandEvent(EventType_ClientConnect);

        return true;
    }

    bool LocalApiClass::LocalClientDisconnect(int64_t waitMsecTime) NN_NOEXCEPT
    {
        if (GetLocalOpenMode() != LocalOpenModeType::OpenMode_Client)
        {
            return false;
        }

        nn::os::SystemEvent connectEvent;
        nn::wlan::ConnectionStatus connectionStatus;
        nn::os::Tick conTimeout;
        nn::Result result;

        result = nn::wlan::Local::GetConnectionEvent(connectEvent.GetBase());
        if (result.IsSuccess() != true)
        {
            NN_LOG("             LocalApi Client GetConnectionEvent Result :0x%08x Line : %d\n", result.GetInnerValueForDebug(), __LINE__);
            return false;
        }

        result = nn::wlan::Local::GetConnectionStatus(&connectionStatus);
        if (result.IsSuccess() == true)
        {
            if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
            {
                return true;
            }
            result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi Client Disconnect Result :0x%08x\n", result.GetInnerValueForDebug());
                return false;
            }

            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_ClientConnect);

            // 指定時間切断待機
            conTimeout = nn::os::GetSystemTick();
            while ((connectionStatus.state == nn::wlan::ConnectionState_Connected) &&
                (nn::os::GetSystemTick() - conTimeout).ToTimeSpan().GetMilliSeconds() <= waitMsecTime)
            {
                if (nn::os::TimedWaitSystemEvent(connectEvent.GetBase(), nn::TimeSpan::FromMilliSeconds(LocalApiClientWait)) != true)
                {
                    continue;
                }

                result = nn::wlan::Local::GetConnectionStatus(&connectionStatus);
                if (result.IsSuccess() != true)
                {
                    break;
                }
            }
        }
        else
        {
            NN_LOG("             LocalApi Client GetConnectionStatus Result :0x%08x Line : %d\n", result.GetInnerValueForDebug(), __LINE__);
        }

        NN_LOG("             *** LocalClient Disconnect Result ***\n");
        NN_LOG("               State      : %s(%d)\n", WlanConnectionStateStr[connectionStatus.state], connectionStatus.state);
        NN_LOG("               Result     : 0x%x(Module:%d Desc:%d)\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription());
        NN_LOG("             *** LocalClient Disconnect Result ***\n");

        return true;
    }

    void LocalApiClass::LocalRelease() NN_NOEXCEPT
    {
        LocalRelease(false);
    }

    void LocalApiClass::LocalRelease(bool isSetup) NN_NOEXCEPT
    {
        LocalSendFrameStop();
        LocalReceiveFrameStop();
        TcpRelease();

        nn::Result result;
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_GetConnectEvent)) > 0)
        {
            nn::os::DestroySystemEvent(&m_connectEvent);
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_GetConnectEvent);
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_CreateEntryId)) > 0)
        {
            result = nn::wlan::Local::DeleteRxEntry(m_entRxId);
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_CreateEntryId);
            NN_LOG("             LocalApi DeleteRxEntry :%lu\n", m_entRxId);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi DeleteRxEntry Error Result(%d:%d) : 0x%08x\n",
                    result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            }
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_CreateActionEntryId)) > 0)
        {
            result = nn::wlan::Local::DeleteRxEntryForActionFrame(m_actionRxId);
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_CreateActionEntryId);
            NN_LOG("             LocalApi DeleteRxEntryForActionFrame :%lu\n", m_actionRxId);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi DeleteRxEntryForActionFrame Error Result(%d:%d) : 0x%08x\n",
                    result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            }
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_CreateThread)) > 0)
        {
            SetCommandEvent(EventType_Exit);
            nn::os::WaitThread(&m_commandThread);
            nn::os::DestroyThread(&m_commandThread);
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_CreateThread);
        }

        // Setupで呼ばれている以外
        if (isSetup == false)
        {
            // BtmModeがNone以外の場合はNoneに設定して終了する
            if (m_btmMode != nn::btm::WlanMode_None)
            {
                // エラー時は関数内でログ出力しているためbool結果判定はなし
                SetBtMode(LocalApiBtNodeNone, LocalApiBtmWaitTime);
            }
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_MasterCreateBss)) > 0)
        {
            result = nn::wlan::Local::DestroyBss();
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_MasterCreateBss);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi DestroyBss Error Result(%d:%d) : 0x%08x\n",
                    result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            }
        }

        if (m_isManagerInit != true)
        {
            if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_MasterInit)) > 0)
            {
                result = nn::wlan::Local::CloseMasterMode();
                m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_MasterInit);
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalApi CloseMasterMode Error Result(%d:%d) : 0x%08x\n",
                        result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
                }
            }
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ClientConnect)) > 0)
        {
            if (LocalClientDisconnect(LocalApiClientWaitTimeOut) == true)
            {
                m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_ClientConnect);
                NN_LOG("             LocalApi Client Disconnected\n");
            }
        }

        if (m_isManagerInit != true)
        {
            if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ClientInit)) > 0)
            {
                result = nn::wlan::Local::CloseClientMode();
                m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_ClientInit);
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalApi CloseClientMode Error Result(%d:%d) : 0x%08x\n",
                        result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
                }
            }
        }
    }

    bool LocalApiClass::LocalGetFrame(size_t size) NN_NOEXCEPT
    {
        m_getSize = size;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[m_getSize]);
        m_getBuffer = std::move(getBuffer);

        SetCommandEvent(EventType_GetFrame);
        return true;
    }

    bool LocalApiClass::LocalGetConnectInfo(LocalConnectsInfo* pInfo) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pInfo);

        nn::os::LockMutex(&m_localApiLock);
        std::memcpy(pInfo, &m_connectsInfo, sizeof(LocalConnectsInfo));
        nn::os::UnlockMutex(&m_localApiLock);
        return true;
    }

    bool LocalApiClass::LocalGetConnectInfo(LocalConnectsInfo* pInfo, const GetLocalFrameType type) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pInfo);

        uint32_t pos = 0;
        nn::os::LockMutex(&m_localApiLock);

        // フレームタイプ指定コピー
        pInfo->connectCount = m_connectsInfo.connectCount;
        pInfo->startRecvTick = m_connectsInfo.startRecvTick;
        pInfo->startSendTick = m_connectsInfo.startSendTick;
        pInfo->isReceiveStart = m_connectsInfo.isReceiveStart;
        pInfo->isSendStart = m_connectsInfo.isSendStart;
        pInfo->totalReceiveSize = m_connectsInfo.totalReceiveSize;
        pInfo->totalSendSize = m_connectsInfo.totalSendSize;
        pInfo->dbRecvRate = m_connectsInfo.dbRecvRate;
        pInfo->dbSendRate = m_connectsInfo.dbSendRate;

        for (int32_t i = 0; i < m_connectsInfo.connectCount; i++)
        {
            if (m_connectsInfo.facing[i].frameType == type)
            {
                pInfo->facing[pos].getFrameCount = m_connectsInfo.facing[i].getFrameCount;
                pInfo->facing[pos].isConnected = m_connectsInfo.facing[i].isConnected;
                pInfo->facing[pos].isGetFrame = m_connectsInfo.facing[i].isGetFrame;
                pInfo->facing[pos].lostFrameCount = m_connectsInfo.facing[i].lostFrameCount;
                pInfo->facing[pos].toMacAddres = m_connectsInfo.facing[i].toMacAddres;
                pInfo->facing[pos].fromMacAddres = m_connectsInfo.facing[i].fromMacAddres;
                pInfo->facing[pos].seqNum = m_connectsInfo.facing[i].seqNum;
                pInfo->facing[pos].totalFrameSize = m_connectsInfo.facing[i].totalFrameSize;
                pInfo->facing[pos].rttAverage = m_connectsInfo.facing[i].rttAverage;
                pInfo->facing[pos].rttMin = m_connectsInfo.facing[i].rttMin;
                pInfo->facing[pos].rttMax = m_connectsInfo.facing[i].rttMax;
                pInfo->facing[pos].rttCount = m_connectsInfo.facing[i].rttCount;
                pos++;
            }
        }
        nn::os::UnlockMutex(&m_localApiLock);
        return true;
    }

    bool LocalApiClass::LocalGetMacList(LoaclMacAddressInfo& macList) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_localApiLock);

        macList.macCount = m_resFrameMacAddress.macCount;
        for (int32_t i = 0; i < m_resFrameMacAddress.macCount; i++)
        {
            macList.macArray[i] = m_resFrameMacAddress.macArray[i];
        }

        nn::os::UnlockMutex(&m_localApiLock);

        return true;
    }

    bool LocalApiClass::LocalMakeFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac,
        const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo, const uint32_t seqNumber) NN_NOEXCEPT
    {
        bool isMakeFrame = LocalMakeFrame(frameBuffer, size, toMac, fromMac, etherType, matchInfo);

        if (isMakeFrame == true)
        {
            isMakeFrame = LocalSeqNumberAddFrame(frameBuffer, size, seqNumber);
        }

        return isMakeFrame;
    }


    bool LocalApiClass::LocalMakeFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac,
        const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo) NN_NOEXCEPT
    {
        size_t offsetSize = 0;
        size_t minFrameSize = nn::wlan::MacAddress::MacAddressSize * 2 + sizeof(etherType) + matchInfo.matchLength;

        if (size < minFrameSize)
        {
            NN_LOG("             LocalMakeFrame Buffer Size Small(%lld) header Size(%lld)\n", size, minFrameSize);
            return false;
        }

        std::memset(&frameBuffer[0], 0x00, size);

        std::memcpy(&frameBuffer[offsetSize], toMac.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
        offsetSize = nn::wlan::MacAddress::MacAddressSize;
        std::memcpy(&frameBuffer[offsetSize], fromMac.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
        offsetSize += nn::wlan::MacAddress::MacAddressSize;

        uint8_t* petherType = reinterpret_cast<uint8_t*>(const_cast<uint16_t*>(&etherType));
        frameBuffer[offsetSize]     = petherType[1];
        frameBuffer[offsetSize + 1] = petherType[0];
        offsetSize += sizeof(etherType);

        std::memcpy(&frameBuffer[offsetSize], matchInfo.matchData, matchInfo.matchLength);
        offsetSize += matchInfo.matchLength;

        return true;
    }

    bool LocalApiClass::LocalMakeFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac,
                                       const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo, const uint32_t seqNumber, bool isNeedReply) NN_NOEXCEPT
    {
        bool isMakeFrame = LocalMakeFrame(frameBuffer, size, toMac, fromMac, etherType, matchInfo, seqNumber);

        if ((isMakeFrame == true) && isNeedReply) {
            LocalFrameHeader* pFrameData = reinterpret_cast<LocalFrameHeader*>(frameBuffer);
            LocalPayloadData* pPayLoad = reinterpret_cast<LocalPayloadData*>(pFrameData->data);
            pPayLoad->seq = seqNumber;
            pPayLoad->reqType = FrameReqType::ReqType_RawPingReq;
            uint8_t *pData = pPayLoad->data;
            int64_t tick = nn::os::GetSystemTick().GetInt64Value();
            pData[0] = tick & 0xFF;
            pData[1] = tick>>8  & 0xFF;
            pData[2] = tick>>16 & 0xFF;
            pData[3] = tick>>24 & 0xFF;
            pData[4] = tick>>32 & 0xFF;
            pData[5] = tick>>40 & 0xFF;
            pData[6] = tick>>48 & 0xFF;
            pData[7] = tick>>56 & 0xFF;
        }

        return isMakeFrame;
    }

    bool LocalApiClass::LocalSeqNumberAddFrame(uint8_t frameBuffer[], size_t size, const uint32_t seqNumber) NN_NOEXCEPT
    {
        size_t minFrameSize = nn::wlan::MacAddress::MacAddressSize * 2 + sizeof(uint16_t) + sizeof(nn::wlan::ReceivedDataMatchInfo::matchData);

        if (minFrameSize > size)
        {
            NN_LOG("             LocalMakeFrame Buffer Size Small(%lld) header Size(%lld)\n", size, minFrameSize);
            return false;
        }

        LocalFrameHeader* pFrameData = reinterpret_cast<LocalFrameHeader*>(frameBuffer);
        LocalPayloadData* pPayLoad = reinterpret_cast<LocalPayloadData*>(pFrameData->data);

        pPayLoad->seq = seqNumber;
        pPayLoad->reqType = FrameReqType::ReqType_Frame;

        return true;
    }

    bool LocalApiClass::LocalMakeReqMacFrame(uint8_t frameBuffer[], size_t size) NN_NOEXCEPT
    {
        size_t minActionFrameSize = sizeof(LocalFrameHeader) + sizeof(LocalPayloadData) - 2;
        if (minActionFrameSize > size)
        {
            return false;
        }

        LocalFrameHeader* pFrameData = reinterpret_cast<LocalFrameHeader*>(frameBuffer);
        LocalPayloadData* pPayLoad = reinterpret_cast<LocalPayloadData*>(pFrameData->data);

        pPayLoad->seq = 0xFFFFFFFF;
        pPayLoad->reqType = FrameReqType::ReqType_MacAddressReq;

        return true;
    }

    bool LocalApiClass::LocalMakeResMacFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress setMacArray[nn::wlan::ConnectableClientsCountMax], const uint32_t macCount) NN_NOEXCEPT
    {
        size_t minActionFrameSize = sizeof(LocalFrameHeader) + sizeof(LocalPayloadData) + sizeof(LocalPayLoadMacReqData::len) + (nn::wlan::MacAddress::MacAddressSize * macCount) - 2;
        if (minActionFrameSize > size)
        {
            return false;
        }

        LocalFrameHeader* pFrameData = reinterpret_cast<LocalFrameHeader*>(frameBuffer);
        LocalPayloadData* pPayLoad = reinterpret_cast<LocalPayloadData*>(pFrameData->data);
        LocalPayLoadMacReqData* pMacData = reinterpret_cast<LocalPayLoadMacReqData*>(pPayLoad->data);

        pPayLoad->seq = 0xFFFFFFFF;
        pPayLoad->reqType = FrameReqType::ReqType_MacAddressRes;
        pMacData->len = 0;

        for (uint32_t i = 0; i < macCount; i++)
        {
            std::memcpy(&pMacData->macAddress[i][0], setMacArray[i].GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
            pMacData->len++;
        }
        return true;
    }

    bool LocalApiClass::LocalMakeActionFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac,
        const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo, const nn::wlan::ActionFrameType actionType, const uint32_t seqNumber) NN_NOEXCEPT
    {
        bool isMakeActionF = LocalMakeActionFrame(frameBuffer, size, toMac, fromMac, etherType, matchInfo, actionType);
        if (isMakeActionF == true)
        {
            isMakeActionF = LocalSeqNumberAddActionFrame(frameBuffer, size, seqNumber);
        }

        return isMakeActionF;
    }

    bool LocalApiClass::LocalMakeActionFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac,
        const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo, const nn::wlan::ActionFrameType actionType) NN_NOEXCEPT
    {
        size_t minActionFrameSize = sizeof(LocalActionFrameData) + sizeof(LocalActionFramePayloadData);
        if (minActionFrameSize > size)
        {
            return false;
        }

        bool isMakeFrame = LocalApiClass::LocalMakeFrame(frameBuffer, size, fromMac, toMac, etherType, matchInfo);
        if (isMakeFrame != true)
        {
            return false;
        }

        LocalActionFrameData *pActionFrame = reinterpret_cast<LocalActionFrameData*>(frameBuffer);
        // Category
        pActionFrame->categry = 0x7F;

        // OUI
        std::memcpy(&pActionFrame->oui, matchInfo.matchData, sizeof(pActionFrame->oui));

        // ActionFrameType
        pActionFrame->frameType = actionType;

        return true;
    }

    bool LocalApiClass::LocalSeqNumberAddActionFrame(uint8_t frameBuffer[], size_t size, const uint32_t seqNumber) NN_NOEXCEPT
    {
        size_t minActionFrameSize = sizeof(LocalActionFrameData) + sizeof(LocalActionFramePayloadData);
        if (minActionFrameSize > size)
        {
            return false;
        }

        LocalActionFrameData* pActionFrame = reinterpret_cast<LocalActionFrameData*>(frameBuffer);
        LocalActionFramePayloadData* pPayLoad = reinterpret_cast<LocalActionFramePayloadData*>(pActionFrame->data);

        pPayLoad->seq = seqNumber;

        return true;
    }

    void LocalApiClass::LocalSetElapseTime() NN_NOEXCEPT
    {
        m_elapsedTick = nn::os::GetSystemTick();
    }

    bool LocalApiClass::LocalPutFrame(uint8_t buffer[], size_t size) NN_NOEXCEPT
    {
        m_putSize = size;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[m_putSize]);

        m_putBuffer = std::move(putBuffer);
        std::memcpy(m_putBuffer.get(), buffer, size);

        SetCommandEvent(EventType_PutFrame);
        return true;
    }

    bool LocalApiClass::LocalSetActionFrameWithBeacon(uint8_t buffer[], size_t size) NN_NOEXCEPT
    {
        m_putSize = size;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[m_putSize]);
        m_putBuffer = std::move(putBuffer);
        std::memcpy(m_putBuffer.get(), buffer, size);

        SetCommandEvent(EventType_MasterSetActionFrameWithBeacon);

        return true;
    }

    bool LocalApiClass::LocalCancelFrameStart(uint32_t waitMsecTime) NN_NOEXCEPT
    {
        m_waitTime = waitMsecTime;
        m_isCancelFrame = true;

        SetCommandEvent(EventType_CancelFrame);
        return true;
    }

    void LocalApiClass::LocalCancelFrameStop() NN_NOEXCEPT
    {
        nn::os::Tick stopTick;
        m_isCancelFrame = false;

        stopTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - stopTick).ToTimeSpan().GetMilliSeconds() <= m_waitTime)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
    }

    bool LocalApiClass::LocalCancelActionFrameStart(uint32_t waitMsecTime) NN_NOEXCEPT
    {
        if (m_isActionFrame != true)
        {
            return false;
        }

        m_waitActionTime = waitMsecTime;
        m_isCancelActionFrame = true;

        SetCommandEvent(EventType_CancelActionFrame);
        return true;
    }

    void LocalApiClass::LocalCancelActionFrameStop() NN_NOEXCEPT
    {
        nn::os::Tick stopTick;
        m_isCancelActionFrame = false;

        stopTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - stopTick).ToTimeSpan().GetMilliSeconds() <= m_waitActionTime)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
    }

    void LocalApiClass::SetCommandEvent(WlanTest::LocalEvent eventType) NN_NOEXCEPT
    {
        SetEventMode(eventType);
        nn::os::SignalEvent(&m_commandEvent);

        while (m_eventType == eventType)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_commandWaitTime));
        }
    }

    void LocalApiClass::SetCommandTime(uint32_t msecTime) NN_NOEXCEPT
    {
        m_commandWaitTime = msecTime;
    }

    void LocalApiClass::SetEventMode(WlanTest::LocalEvent eventType) NN_NOEXCEPT
    {
        m_eventType = eventType;
    }

    void LocalApiClass::SetState(uint32_t state) NN_NOEXCEPT
    {
        m_setupMode = state;
    }

    void LocalApiClass::CommandThread(void* arg) NN_NOEXCEPT
    {
        WlanTest::LocalApiClass* pThis = static_cast<LocalApiClass*>(arg);
        nn::wlan::MacAddress outMacAddress;
        nn::os::Tick cancelTick;
        bool isLoop = true;
        size_t outSize = 0;
        LocalEvent testEvent;
        nn::Result result;

        while (isLoop == true)
        {
            nn::os::WaitEvent(&pThis->m_commandEvent);

            testEvent = pThis->m_eventType;
            pThis->SetEventMode(LocalEvent::EventType_Init);

            switch (testEvent)
            {
            case EventType_MasterSetActionFrameWithBeacon:
                pThis->m_ThreadResult = nn::wlan::Local::SetActionFrameWithBeacon(pThis->m_putBuffer.get(), pThis->m_putSize);

                break;
            case EventType_PutFrame:

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
                NN_LOG("            LocalApi Command Thread PutFrameRaw Non MacAddress\n");
#endif
                pThis->m_ThreadResult = nn::wlan::Local::PutFrameRaw(pThis->m_putBuffer.get(), pThis->m_putSize);
                if (pThis->m_ThreadResult.IsSuccess() != true)
                {
                    NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Line %d\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription(), __LINE__);
                }
                break;
            case EventType_GetFrame:

                if (pThis->m_isFrame == true)
                {
                    NN_LOG("             LocalApi Command Thread GetFrameRaw Start\n");
                    pThis->m_ThreadResult = nn::wlan::Local::GetFrameRaw(pThis->m_getBuffer.get(), pThis->m_getSize, &outSize, pThis->m_entRxId);
                    NN_LOG("             LocalApi Command Thread GetFrameRaw End\n");
                }

                if (pThis->m_isActionFrame == true)
                {
                    NN_LOG("             LocalApi Command Thread GetActionFrame Start\n");
                    pThis->m_ThreadResult = nn::wlan::Local::GetActionFrame(&outMacAddress, pThis->m_getBuffer.get(), pThis->m_getSize, &outSize, pThis->m_actionRxId);
                    NN_LOG("             LocalApi Command Thread GetActionFrame End\n");
                }

                break;
            case EventType_ClientConnect:
                NN_LOG("             LocalApi Command Thread Connect Start\n");
                if (nn::wlan::Local::Connect(pThis->m_ssid, pThis->m_bssid, -1, pThis->m_security,
                    false, nn::wlan::BeaconIndication_Disable).IsSuccess() == true)
                {
                    pThis->SetState(pThis->m_setupMode ^WLAN_TEST_LEFTSHIFT(TestMode_ClientConnect));
                }
                NN_LOG("             LocalApi Command Thread Connect End\n");

                break;
            case EventType_CancelFrame:

                cancelTick = nn::os::GetSystemTick();
                while ((nn::os::GetSystemTick() - cancelTick).ToTimeSpan().GetMilliSeconds() <= pThis->m_waitTime)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                    if (pThis->m_isCancelFrame != true)
                    {
                        break;
                    }
                }

                if (pThis->m_isCancelFrame == true)
                {
                    NN_LOG("             LocalApi Command Thread CancelGetFrame Start\n");
                    pThis->m_ThreadResult = nn::wlan::Local::CancelGetFrame(pThis->m_entRxId);
                    NN_LOG("             LocalApi Command Thread CancelGetFrame End Result : 0x%08x\n", pThis->m_ThreadResult.GetInnerValueForDebug());
                }
                pThis->m_waitTime = 0;
                break;
            case EventType_CancelActionFrame:

                cancelTick = nn::os::GetSystemTick();
                while ((nn::os::GetSystemTick() - cancelTick).ToTimeSpan().GetMilliSeconds() <= pThis->m_waitActionTime)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                    if (pThis->m_isCancelActionFrame != true)
                    {
                        break;
                    }
                }

                if (pThis->m_isCancelActionFrame == true)
                {
                    NN_LOG("             LocalApi Command Thread CancelGetActionFrame Start\n");
                    pThis->m_ThreadResult = nn::wlan::Local::CancelGetActionFrame(pThis->m_actionRxId);
                    NN_LOG("             LocalApi Command Thread CancelGetActionFrame End Result : 0x%08x\n", pThis->m_ThreadResult.GetInnerValueForDebug());
                }
                pThis->m_waitActionTime = 0;
                break;
            default:
                isLoop = false;
                break;
            }
        }
    }

    uint32_t LocalApiClass::GetStateClientCount(nn::wlan::ConnectionState State) NN_NOEXCEPT
    {
        nn::wlan::MacAddress clientMacAddress[nn::wlan::ConnectableClientsCountMax];
        return GetClientMacAddress(clientMacAddress, State);
    }

    uint32_t LocalApiClass::GetClientMacAddress(nn::wlan::MacAddress clientMacAddress[nn::wlan::ConnectableClientsCountMax], nn::wlan::ConnectionState State) NN_NOEXCEPT
    {
        uint32_t clientConnectedCount = 0;
        nn::Result result;

        result = nn::wlan::Local::GetClientStatus(m_clientStatus, &m_clientStatusBitMap);
        if (result.IsSuccess() != true)
        {
            return 0;
        }

        for (int8_t i = nn::wlan::ConnectableClientsCountMax - 1; i >= 0; i--)
        {
            if ((m_clientStatusBitMap >> i) & 0x00000001 && m_clientStatus[i].state == State)
            {
                clientMacAddress[clientConnectedCount] = m_clientStatus[i].clientMacAddress;
                clientConnectedCount++;
            }
            else
            {
                if (m_clientStatus[i].state == State)
                {
                    clientMacAddress[clientConnectedCount] = m_clientStatus[i].clientMacAddress;
                    clientConnectedCount++;
                }
            }
        }

        return clientConnectedCount;
    }

    bool LocalApiClass::SetPoolMngSetting(LocalPoolMng& poolMng, const size_t packetSize) NN_NOEXCEPT
    {
        if (packetSize > m_getSize)
        {
            return false;
        }

        uint8_t* pBuffer = const_cast<uint8_t*>(m_getBuffer.get());
        size_t actSize = m_getSize;

        std::memset(&poolMng, 0x00, sizeof(poolMng));

        poolMng.pBuffer = pBuffer;
        poolMng.bufferSize = m_getSize;
        poolMng.packetSize = packetSize;
        poolMng.maxCount = 0;

        while (actSize > packetSize)
        {
            poolMng.info[poolMng.maxCount].pData = pBuffer;
            poolMng.info[poolMng.maxCount].bufferSize = packetSize;
            poolMng.info[poolMng.maxCount].dataSize = 0;
            poolMng.maxCount++;
            actSize -= packetSize;
            pBuffer += packetSize;

            if (poolMng.maxCount >= LocalApiPoolMax)
            {
                break;
            }
        }

        return true;
    }

    LocalPoolMngDetail* LocalApiClass::GetFirstWritePool(LocalPoolMng& poolMng, const GetLocalFrameType frameType) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_localApiLock);

        poolMng.writeIdx = 0;
        poolMng.info[0].frameType = frameType;
        poolMng.info[0].dataSize = 0;

        nn::os::UnlockMutex(&m_localApiLock);

        return &poolMng.info[0];
    }

    LocalPoolMngDetail* LocalApiClass::GetNextWritePool(LocalPoolMng& poolMng, const GetLocalFrameType frameType) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_localApiLock);

        ++poolMng.writeIdx %= poolMng.maxCount;

        poolMng.info[poolMng.writeIdx].frameType = frameType;
        poolMng.info[poolMng.writeIdx].dataSize = 0;

        nn::os::UnlockMutex(&m_localApiLock);

        return &poolMng.info[poolMng.writeIdx];
    }

    bool LocalApiClass::LocalReceiveFrameStart(const size_t bufferSize, const size_t packetSize,
        const uint32_t waitConnectCount, const uint32_t endConnectCount, const uint32_t endPacketCount) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(bufferSize > 0 && bufferSize >= packetSize);

        static NN_ALIGNAS(4096) uint8_t localReceiveStack[StackSize];
        static NN_ALIGNAS(4096) uint8_t localReceiveActionFrameStack[StackSize];
        static NN_ALIGNAS(4096) uint8_t localParseStack[StackSize];
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[bufferSize]);
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[LocalApiFramePacketSize]);
        nn::Result result;

        NN_ABORT_UNLESS_NOT_NULL(getBuffer.get());
        NN_ABORT_UNLESS_NOT_NULL(putBuffer.get());

        std::memset(putBuffer.get(), 0x00, LocalApiFramePacketSize);

        LocalSendFrameStop();
        LocalReceiveFrameStop();

        m_getBuffer = std::move(getBuffer);
        m_getSize   = bufferSize;

        m_startPutBuffer    = std::move(putBuffer);
        m_startPutSize      = LocalApiFramePacketSize;

        m_waitConnectCount = waitConnectCount;
        m_endConnectCount = endConnectCount;
        m_endPacketCount = endPacketCount;

        if (SetPoolMngSetting(m_poolFrameMng, packetSize) != true)
        {
            return false;
        }

        // m_connectsInfo構造体にクラスメンバーを含むためループ初期化
        m_connectsInfo.connectCount = 0;
        m_connectsInfo.isReceiveStart = false;
        for (int32_t i = 0; i < LocalApiClientMaxCount; i++)
        {
            m_connectsInfo.facing[i].lostFrameCount = 0;
            m_connectsInfo.facing[i].getFrameCount  = 0;
            m_connectsInfo.facing[i].totalFrameSize = 0;
            m_connectsInfo.facing[i].seqNum         = 0;
            m_connectsInfo.facing[i].seqNumberGap   = 0;
            m_connectsInfo.facing[i].perRate        = 0.0;
            m_connectsInfo.facing[i].isGetFrame     = false;
            m_connectsInfo.facing[i].isConnected    = false;
            m_connectsInfo.facing[i].isRttPkt       = false;
            m_connectsInfo.facing[i].rttAverage     = 0;
            m_connectsInfo.facing[i].rttMin         = 0;
            m_connectsInfo.facing[i].rttMax         = 0;
            m_connectsInfo.facing[i].rttCount       = 0;
        }

#ifdef WLAN_TEST_RECV_TRACE
        m_masterToClientCount                       = 0;
        m_clientToClientCount                       = 0;
#endif

        nn::os::ClearEvent(&m_parseFrameEvent);
        nn::os::ClearEvent(&m_receiveFrameEvent);

        // フレーム受信スレッド生成
        m_isReceiveFrame = true;
        result = nn::os::CreateThread(&m_receiveThread, LocalReceiveFrameThread, this, localReceiveStack, StackSize, nn::os::DefaultThreadPriority - 5);
        if (result.IsSuccess() != true)
        {
            return false;
        }
        nn::os::StartThread(&m_receiveThread);
        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_ReceiveFrameThread));

        // 受信フレーム解析スレッド生成
        m_parseType = LocalParseType::ParseType_Parse;
        result = nn::os::CreateThread(&m_parseThread, LocalParseFrameThread, this, localParseStack, StackSize, nn::os::DefaultThreadPriority - 2);
        if (result.IsSuccess() != true)
        {
            return false;
        }
        nn::os::StartThread(&m_parseThread);
        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_ParseFrameThread));

        if (m_isActionFrame == true)
        {
            result = nn::os::CreateThread(&m_receiveActionFrameThread, LocalReceiveActionFrameThread, this, localReceiveActionFrameStack, StackSize, nn::os::DefaultThreadPriority);
            if (result.IsSuccess() != true)
            {
                return false;
            }
            nn::os::StartThread(&m_receiveActionFrameThread);
            SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_ReceiveActionFrameThread));
        }

        return true;
    }

    bool LocalApiClass::LocalReceiveWait(uint64_t waitMsecTime, uint64_t sendTime, bool isStart) NN_NOEXCEPT
    {
        // CLIENTのみAPI
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ClientInit)) == 0)
        {
            return false;
        }

        size_t putSize = LocalApiFramePacketSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::ConnectionStatus conStatus;
        uint64_t oneWaitTime = waitMsecTime;
        int32_t checkCnt = 1;
        nn::Result result;
        bool isSuccess;

        NN_ABORT_UNLESS_NOT_NULL(putBuffer);

        if ((result = nn::wlan::Local::GetConnectionStatus(&conStatus)).IsSuccess() != true)
        {
            NN_LOG("             LocalApi GetConnectionStatus  Error(0x%08x) Line(%d)\n", result.GetInnerValueForDebug(), __LINE__);
            return false;
        }

        isSuccess = LocalMakeFrame(putBuffer.get(), putSize, conStatus.bssid,
            m_macAddress, m_etherList.etherTypes[0], TestMatchInfo[0], 0);
        if (isSuccess != true)
        {
            NN_LOG("             LocalApi LocalMakeFrame Error\n");
            return false;
        }

        LocalMakeReqMacFrame(putBuffer.get(), putSize);
        if (waitMsecTime > sendTime)
        {
            checkCnt = static_cast<int32_t>(waitMsecTime / sendTime);
            oneWaitTime = sendTime;
        }

        for (int32_t i = 0; i < checkCnt; i++)
        {
            nn::wlan::Local::PutFrameRaw(putBuffer.get(), putSize);
            isSuccess = LocalReceiveWait(oneWaitTime, isStart);
            if (isSuccess == true)
            {
                break;
            }

            if (sendTime > 0 && (i + 1 == checkCnt))
            {
                oneWaitTime = waitMsecTime - (checkCnt * sendTime);
                isSuccess = LocalReceiveWait(oneWaitTime, isStart);
            }
        }

        return isSuccess;
    }

    bool LocalApiClass::LocalReceiveWait(uint64_t waitMsecTime, bool isStart) NN_NOEXCEPT
    {
        if (isStart == true && m_isReceiveFrame != true)
        {
            return false;
        }

        bool isSygnalEvent = nn::os::TimedWaitEvent(&m_receiveFrameEvent, nn::TimeSpan::FromMilliSeconds(waitMsecTime));
        if (isSygnalEvent == true)
        {
            nn::os::ClearEvent(&m_receiveFrameEvent);
        }

        return isSygnalEvent;
    }

    void LocalApiClass::LocalReceiveFrameStop() NN_NOEXCEPT
    {
        nn::Result result;
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ReceiveActionFrameThread)) > 0)
        {
            NN_LOG("             LocalApi Action Thread Free ActionRxID : %d\n", m_actionRxId);
            m_isReceiveFrame = false;
            result = nn::wlan::Local::CancelGetActionFrame(m_actionRxId);
            NN_LOG("             LocalApi CancelGetActionFrame Result : 0x%08x\n", result.GetInnerValueForDebug());
            nn::os::WaitThread(&m_receiveActionFrameThread);
            nn::os::DestroyThread(&m_receiveActionFrameThread);
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_ReceiveActionFrameThread);
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ReceiveFrameThread)) > 0)
        {
            NN_LOG("             LocalApi Frame Thread Free RxID : %d\n", m_entRxId);
            m_isReceiveFrame = false;
            result = nn::wlan::Local::CancelGetFrame(m_entRxId);
            NN_LOG("             LocalApi CancelGetFrame Result : 0x%08x\n", result.GetInnerValueForDebug());
            nn::os::WaitThread(&m_receiveThread);
            nn::os::DestroyThread(&m_receiveThread);
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_ReceiveFrameThread);
        }

        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_ParseFrameThread)) > 0)
        {
            NN_LOG("             LocalApi ParseFrame Thread Free\n");
            m_parseType = LocalParseType::ParseType_End;
            nn::os::SignalEvent(&m_parseFrameEvent);
            nn::os::WaitThread(&m_parseThread);
            nn::os::DestroyThread(&m_parseThread);
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_ParseFrameThread);
        }
    }

    bool LocalApiClass::LocalSendFrameStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval, bool isClientToClientSend) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(sendSize >= LocalApiFramePacketSize);
        static NN_ALIGNAS(4096) uint8_t localSendFrameStack[StackSize];
        nn::Result result;

        LocalSendFrameStop();

        m_putSize           = sendSize;
        m_sendTime          = sendTime;
        m_sendInterval      = sendInterval;
        m_isClientToClient  = isClientToClientSend;

        result = nn::os::CreateThread(&m_sendThread, LocalSendFrameThread, this, localSendFrameStack, StackSize, nn::os::DefaultThreadPriority);
        if (result.IsSuccess() != true)
        {
            return false;
        }

        m_isSendFrame = true;
        nn::os::StartThread(&m_sendThread);
        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_SendFrameThread));

        return true;
    }

    bool LocalApiClass::LocalSendFrameOneShotWaitStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval, bool isClientToClientSend) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(sendSize >= LocalApiFramePacketSize);
        static NN_ALIGNAS(4096) uint8_t localSendFrameStack[StackSize];
        nn::Result result;

        LocalSendFrameStop();

        m_putSize           = sendSize;
        m_sendTime          = sendTime;
        m_sendInterval      = sendInterval;
        m_isClientToClient  = isClientToClientSend;
        m_isNeedReply       = false;

        result = nn::os::CreateThread(&m_sendThread, LocalSendFrameOneShotWaitThread, this, localSendFrameStack, StackSize, nn::os::DefaultThreadPriority);
        if (result.IsSuccess() != true)
        {
            return false;
        }

        m_isSendFrame = true;
        nn::os::StartThread(&m_sendThread);
        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_SendFrameThread));

        return true;
    }

    // NTD追加関数
    bool LocalApiClass::LocalSendFrameOneShotWaitStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval, bool isClientToClientSend, bool isNeedReply) NN_NOEXCEPT
    {
        static NN_ALIGNAS(4096) uint8_t localSendFrameStack[StackSize];
        nn::Result result;

        LocalSendFrameStop();

        m_putSize           = sendSize;
        m_sendTime          = sendTime;
        m_sendInterval      = sendInterval;
        m_isClientToClient  = isClientToClientSend;
        m_isNeedReply       = isNeedReply;

        result = nn::os::CreateThread(&m_sendThread, LocalSendFrameOneShotWaitThread, this, localSendFrameStack, StackSize, nn::os::DefaultThreadPriority);
        if (result.IsSuccess() != true)
        {
            return false;
        }

        m_isSendFrame = true;
        nn::os::StartThread(&m_sendThread);
        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_SendFrameThread));

        return true;
    }

    bool LocalApiClass::LocalBroadcastSendFrameStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval) NN_NOEXCEPT
    {
        static NN_ALIGNAS(4096) uint8_t localBraodcastSendFrameStack[StackSize];
        nn::Result result;

        LocalSendFrameStop();

        m_putSize       = sendSize;
        m_sendTime      = sendTime;
        m_sendInterval  = sendInterval;

        result = nn::os::CreateThread(&m_sendThread, LocalBroadcastSendFrameThread, this, localBraodcastSendFrameStack, StackSize, nn::os::DefaultThreadPriority);
        if (result.IsSuccess() != true)
        {
            return false;
        }

        m_isSendFrame = true;
        nn::os::StartThread(&m_sendThread);
        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_SendFrameThread));

        return true;
    }

    bool LocalApiClass::LocalSendActionFrameStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval, nn::wlan::ActionFrameType frameType) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(sendSize >= LocalApiFramePacketSize);
        static NN_ALIGNAS(4096) uint8_t localSendActionFrameStack[StackSize];
        nn::Result result;

        LocalSendFrameStop();

        m_putSize = sendSize;
        m_sendTime = sendTime;
        m_sendInterval = sendInterval;
        m_frameType = frameType;

        result = nn::os::CreateThread(&m_sendActionThread, LocalSendActionFrameThread, this, localSendActionFrameStack, StackSize, nn::os::DefaultThreadPriority);
        if (result.IsSuccess() != true)
        {
            return false;
        }

        m_isSendActionFrame = true;
        nn::os::StartThread(&m_sendActionThread);
        SetState(m_setupMode | WLAN_TEST_LEFTSHIFT(TestMode_SendFrameThread));

        return true;
    }

    void LocalApiClass::LocalSendFrameStop() NN_NOEXCEPT
    {
        if ((m_setupMode & WLAN_TEST_LEFTSHIFT(TestMode_SendFrameThread)) > 0)
        {
            if (m_isSendActionFrame == true)
            {
                NN_LOG("             LocalApi SendActionFrame Thread Free\n");
                m_isSendActionFrame = false;
                nn::os::WaitThread(&m_sendActionThread);
                nn::os::DestroyThread(&m_sendActionThread);
            }

            if (m_isSendFrame == true)
            {
                NN_LOG("             LocalApi SendFrame Thread Free\n");
                m_isSendFrame = false;
                nn::os::WaitThread(&m_sendThread);
                nn::os::DestroyThread(&m_sendThread);
            }
            m_setupMode ^= WLAN_TEST_LEFTSHIFT(TestMode_SendFrameThread);
        }
    }

    void LocalApiClass::LocalClearSeqNumber() NN_NOEXCEPT
    {
        AutoLock lock(&m_localRcvLock);
        std::memset(&m_seqArray, 0x00, sizeof(m_seqArray));
        m_seqNumber = 0;
    }

    void LocalApiClass::LocalTraceBssInfo() NN_NOEXCEPT
    {
        char strMacAddress[nn::wlan::MacAddress::MacStringSize];
        char strSsid[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::wlan::ConnectionStatus conStatus;

        if (nn::wlan::Local::GetConnectionStatus(&conStatus).IsSuccess() == true)
        {
            NN_LOG("                 ***** LOCAL MASTER BSS INFO *****\n");
            NN_LOG("                   MAC ADDRESS : %s\n", conStatus.bssid.GetString(strMacAddress));
            NN_LOG("                   SSID        : %s\n", conStatus.ssid.GetHexString(strSsid));
            NN_LOG("                   CHANNEL     : %d\n", conStatus.channel);
            NN_LOG("                 ***** LOCAL MASTER BSS INFO *****\n\n");
        }
    }

    void LocalApiClass::LocalTraceCurrentInfo() NN_NOEXCEPT
    {
        nn::wlan::MacAddress myMacAddress;
        char strMacAddress[nn::wlan::MacAddress::MacStringSize];
        char strSsid[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::wlan::ConnectionStatus conStatus;
        nn::wlan::WlanState state;
        nn::wlan::LinkLevel link;
        int32_t rssi;
        nn::Result result;

#ifdef WLAN_TEST_RECV_TRACE
        NN_LOG("                 ***** LOCAL DEBUG TRACE INFO *****\n");
        NN_LOG("                   Master To Client GetFrame Count : %d\n", m_masterToClientCount);
        NN_LOG("                   Client To Client GetFrame Count : %d\n", m_clientToClientCount);
        NN_LOG("                 ***** LOCAL DEBUG TRACE INFO *****\n\n");
#endif

#ifdef WLAN_TEST_LINKLEVEL_ERROR
        NN_LOG("                 Call API GetMacAddress\n\n");
#endif
        nn::wlan::Local::GetMacAddress(&myMacAddress);
        myMacAddress.GetString(strMacAddress);
        NN_LOG("                 ***** LOCAL INFO *****\n");
        NN_LOG("                   OpenMode       : %s\n", GetStringOpenMode(GetLocalOpenMode()).c_str());
        NN_LOG("                   MacAddress     : %s\n", strMacAddress);

#ifdef WLAN_TEST_STATE_ASSER_STOP
        result = this->GetState(&state);
#else
#ifdef WLAN_TEST_LINKLEVEL_ERROR
        NN_LOG("                 Call API GetState\n\n");
#endif
        result = nn::wlan::Local::GetState(&state);
#endif
        if (result.IsSuccess() == true)
        {
            NN_LOG("                   State          : %d\n", state);
        }
        else
        {
            NN_LOG("                   State Result Error : 0x%08x\n", result.GetInnerValueForDebug());
        }

#ifdef WLAN_TEST_LINKLEVEL_ERROR
        NN_LOG("                 Call API GetLocalOpenMode\n\n");
#endif
        switch (GetLocalOpenMode())
        {
        case LocalOpenModeType::OpenMode_Master:
            NN_LOG("\n                   [MASTER MODE INFO]\n");
            NN_LOG("                       SSID             : %s\n", m_bssParam.ssid.GetHexString(strSsid));
            NN_LOG("                       CHANNEL          : %d\n", m_bssParam.channel);
            NN_LOG("                       STEALTH          : %d\n", m_bssParam.hiddenSsid);
            NN_LOG("                       INACTIVE PERIOD  : %lu\n", m_bssParam.inactivePeriod);
            NN_LOG("                       SUPPORTED RATES  : 0x%08x\n", m_bssParam.supportedRates);
            NN_LOG("                       BASIC RATES      : %lu\n", m_bssParam.basicRates);
            NN_LOG("                       SECURITY GROUP   : %s\n", SecurityTypeString[m_bssParam.security.groupPrivacyMode]);
            NN_LOG("                       SECURITY PRIVACY : %s\n", SecurityTypeString[m_bssParam.security.privacyMode]);
            NN_LOG("                       SECURITY KEYIDX  : %d\n\n", m_bssParam.security.keyIdx);
            break;
        case LocalOpenModeType::OpenMode_Client:
            if (state == nn::wlan::WlanState_LocalClient || state == nn::wlan::WlanState_LocalClientScan)
            {
#ifdef WLAN_TEST_LINKLEVEL_ERROR
                NN_LOG("                 Call API GetRssi\n\n");
#endif
                result = nn::wlan::Local::GetRssi(&rssi);
                if (result.IsSuccess() == true)
                {
                    NN_LOG("                   Rssi           : %d\n", rssi);
                }
                else
                {
                    NN_LOG("                   Rssi Result Error : 0x%08x\n", result.GetInnerValueForDebug());
                }
#ifdef WLAN_TEST_LINKLEVEL_WAIT
#ifdef WLAN_TEST_LINKLEVEL_ERROR
                NN_LOG("                 Call API GetLinkLevel\n\n");
#endif
                result = nn::wlan::Local::GetLinkLevel(&link);
                if (result.IsSuccess() == true)
                {
                    NN_LOG("                   Link Level     : %d\n", link);
                }
                else
                {
                    NN_LOG("                   Link Level Result Error : 0x%08x\n", result.GetInnerValueForDebug());
                }
#endif
            }
            else
            {
                NN_LOG("\n                   [CLIENT MODE SETTING INFO]\n");
                NN_LOG("                       SSID             : %s\n", m_ssid.GetHexString(strSsid));
                NN_LOG("                       CHANNEL          : %d\n", m_channel);
                NN_LOG("                       SECURITY GROUP   : %s\n", SecurityTypeString[m_security.groupPrivacyMode]);
                NN_LOG("                       SECURITY PRIVACY : %s\n", SecurityTypeString[m_security.privacyMode]);
                NN_LOG("                       SECURITY KEYIDX  : %d\n", m_security.keyIdx);
                NN_LOG("                   [CLIENT MODE SETTING INFO]\n");
            }

#ifdef WLAN_TEST_LINKLEVEL_ERROR
            NN_LOG("                 Call API GetConnectionStatus\n\n");
#endif
            NN_LOG("\n                   [CLIENT MODE CONNECTED INFO]\n");
            result = nn::wlan::Local::GetConnectionStatus(&conStatus);
            if (result.IsSuccess() == true)
            {
                NN_LOG("                       SSID             : %s\n", conStatus.ssid.GetHexString(strSsid));
                NN_LOG("                       MAC ADDRESS      : %s\n", conStatus.bssid.GetString(strMacAddress));
                NN_LOG("                       CHANNEL          : %d\n", conStatus.channel);
                NN_LOG("                       CAUSE            : %d\n", conStatus.cause);
                NN_LOG("                       AID              : %d\n", conStatus.aid);
                NN_LOG("                       SATUS REASON CODE: %d\n", conStatus.statusReasonCode);
                NN_LOG("                       CAPABILITYINFO   : %d\n", conStatus.capabilityInfo);
                NN_LOG("                       BEACON INTERVAL  : %d\n", conStatus.beaconInterval);
                NN_LOG("                       CONNECT STATUS   : %d\n\n", conStatus.state);
            }
            else
            {
                NN_LOG("                   GetConnectionStatus Result Error : 0x%08x\n", result.GetInnerValueForDebug());
            }
            NN_LOG("                   [CLIENT MODE CONNECT INFO]\n");
            break;
        default:
            break;
        }
        NN_LOG("                 ***** LOCAL INFO *****\n\n");
    } // NOLINT(impl/function_size) WLANフレーム調査ログ削除でチェックは通ります

    void LocalApiClass::LocalTracceClientInfo(LocalConnectsInfo& info) NN_NOEXCEPT
    {
        double rateTime;
        int64_t elapsedTime = (nn::os::GetSystemTick() - m_elapsedTick).ToTimeSpan().GetMilliSeconds();
        uint32_t msecTime   = elapsedTime % 1000;
        uint32_t secTime    = (elapsedTime / 1000) % 60;
        uint32_t minuteTime = (elapsedTime / 1000) % (60 * 60) / 60;
        uint32_t hourTime   = (elapsedTime / 1000) / (60 * 60);

        NN_LOG("                   Elapsed Time          : %lu:%02lu:%02lu.%03lu\n", hourTime, minuteTime, secTime, msecTime);
        NN_LOG("                   Recieve Client Count  : %d\n", info.connectCount);
        rateTime = static_cast<double>((nn::os::GetSystemTick() - info.startRecvTick).ToTimeSpan().GetSeconds());
        if (info.isSendStart == true && rateTime > 0)
        {
            info.dbSendRate = 0;
            info.dbSendRate = (info.totalSendSize * 8.0) / rateTime;
            info.dbSendRate /= (1024.0 * 1024.0);
            if (info.dbSendRate < 0.0001)
            {
                info.dbSendRate = 0.0;
            }
            NN_LOG("                   Frame Total Send Size : %llu Byte\n", info.totalSendSize);
            NN_LOG("                   Frame Send Rate       : %.3lf Mbps\n", info.dbSendRate);
        }

        rateTime = static_cast<double>((nn::os::GetSystemTick() - info.startRecvTick).ToTimeSpan().GetSeconds());
        if (info.isReceiveStart == true && rateTime > 0)
        {
            info.dbRecvRate = 0;
            info.dbRecvRate = (info.totalReceiveSize * 8.0) / rateTime;
            info.dbRecvRate /= (1024.0 * 1024.0);
            if (info.dbRecvRate < 0.0001)
            {
                info.dbRecvRate = 0.0;
            }
            NN_LOG("                   Frame Total Recv Size : %llu Byte\n", info.totalReceiveSize);
            NN_LOG("                   Frame Recv Rate       : %.3lf Mbps\n", info.dbRecvRate);
        }

    }

    void LocalApiClass::LocalTraceConnectsInfo(bool isModeLog) NN_NOEXCEPT
    {
        LocalConnectsInfo info;
        char strMacAddress[nn::wlan::MacAddress::MacStringSize];
        uint16_t* pEtherType;

        NN_LOG("\n                 --------------------------------\n");
        if (isModeLog == true)
        {
            LocalTraceCurrentInfo();
        }

        LocalGetConnectInfo(&info);
        NN_LOG("                 ***** CONNECT CLIENT INFO *****\n");
        LocalTracceClientInfo(info);

        for (uint32_t i = 0; i < info.connectCount; i++)
        {
            info.facing[i].fromMacAddres.GetString(strMacAddress);
            pEtherType = reinterpret_cast<uint16_t*>(&info.facing[i].etherType[0]);
            switch (info.facing[i].frameType)
            {
            case GetLocalFrameType::FrameType_Frame:
                NN_LOG("                   [FRAME INFO]\n");
                break;
            case GetLocalFrameType::FrameType_ActionFrame:
                NN_LOG("                   [ACTION FRAME INFO]\n");
                break;
            default:
                NN_LOG("                   [NO FRAME TYPE] %d\n", info.facing[i].frameType);
                continue;
            }
            NN_LOG("                   Mac Address       : %s\n", strMacAddress);
            NN_LOG("                   EtherType         : %04x\n", *pEtherType);
            NN_LOG("                   Get Frame Count   : %lu\n", info.facing[i].getFrameCount);
            NN_LOG("                   Total Frame Size  : %lld Byte\n", info.facing[i].totalFrameSize);
            NN_LOG("                   Lost Frame Count  : %lu\n", info.facing[i].lostFrameCount);
            NN_LOG("                   Per Rate          : %.3lf %\n\n", info.facing[i].perRate);
        }
        NN_LOG("                 ***** CONNECT CLIENT INFO *****\n\n");
    }

    // NTD追加ログ（受け入れ検証と分離）
    void LocalApiClass::LocalTraceConnetsRttInfo(bool isModeLog) NN_NOEXCEPT
    {
        LocalConnectsInfo info;
        char strMacAddress[nn::wlan::MacAddress::MacStringSize];
        uint16_t* pEtherType;

        NN_LOG("\n                 --------------------------------\n");
        if (isModeLog == true)
        {
            LocalTraceCurrentInfo();
        }

        LocalGetConnectInfo(&info);
        NN_LOG("                 ***** CONNECT CLIENT INFO *****\n");
        LocalTracceClientInfo(info);

        for (uint32_t i = 0; i < info.connectCount; i++)
        {
            info.facing[i].fromMacAddres.GetString(strMacAddress);
            pEtherType = reinterpret_cast<uint16_t*>(&info.facing[i].etherType[0]);
            switch (info.facing[i].frameType)
            {
            case GetLocalFrameType::FrameType_Frame:
                NN_LOG("                   [FRAME INFO]\n");
                break;
            case GetLocalFrameType::FrameType_ActionFrame:
                NN_LOG("                   [ACTION FRAME INFO]\n");
                break;
            default:
                NN_LOG("                   [NO FRAME TYPE] %d\n", info.facing[i].frameType);
                continue;
            }
            NN_LOG("                   Mac Address       : %s\n", strMacAddress);
            NN_LOG("                   EtherType         : %04x\n", *pEtherType);
            NN_LOG("                   Get Frame Count   : %lu\n", info.facing[i].getFrameCount);
            NN_LOG("                   Total Frame Size  : %lld Byte\n", info.facing[i].totalFrameSize);
            NN_LOG("                   Lost Frame Count  : %lu\n", info.facing[i].lostFrameCount);
            NN_LOG("                   Per Rate          : %.3lf %\n", info.facing[i].perRate);
            NN_LOG("                   SeqNum Gap        : %lu\n", info.facing[i].seqNumberGap);
            NN_LOG("                   RTT SeqNum Recent : %lu\n", info.facing[i].seqNumberRttRecent);
            NN_LOG("                   RTT SeqNum Gap    : %lu\n", info.facing[i].seqNumberRttGap);
            NN_LOG("                   RTT Average       : %.3lf msec\n", info.facing[i].rttAverage);
            NN_LOG("                   RTT Minimum       : %lu msec\n", info.facing[i].rttMin);
            NN_LOG("                   RTT Maximum       : %lu msec\n", info.facing[i].rttMax);
            NN_LOG("                   Pkt Count(RTT)    : %lu\n\n", info.facing[i].rttCount);
        }
        NN_LOG("                 ***** CONNECT CLIENT INFO *****\n\n");
    }

    bool LocalApiClass::LocalStartTriggerPutFrame(uint8_t frameBuffer[], size_t size, nn::wlan::MacAddress& toMac, nn::wlan::MacAddress& fromMac) NN_NOEXCEPT
    {
        nn::Result result;
        bool isSuccess = false;

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
        char fromMacArray[nn::wlan::MacAddress::MacStringSize];
        char toMacArray[nn::wlan::MacAddress::MacStringSize];
#endif

        for (int32_t i = 0; i < m_etherList.typeCount; i++)
        {
            isSuccess = LocalMakeFrame(frameBuffer, size, toMac, fromMac, m_etherList.etherTypes[i], TestMatchInfo[0]);
            if (isSuccess != true)
            {
                NN_LOG("             LocalApi MakeFrame Error\n");
                return false;
            }

            isSuccess = LocalMakeResMacFrame(frameBuffer, size, m_clientMacArray, m_connectCount);
            if (isSuccess != true)
            {
                NN_LOG("             LocalApi MakeReqMacFrame Error\n");
                return false;
            }

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
            NN_LOG("            LocalApi StartTrigger PutFrameRaw From MacAddress[%s] To MacAddress[%s]\n",
                fromMac.GetString(fromMacArray), toMac.GetString(toMacArray));
#endif
            result = nn::wlan::Local::PutFrameRaw(frameBuffer, size);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Line %d\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription(), __LINE__);
                return false;
            }
        }

        m_resFrameMacAddress.macCount = 0;
        for (int32_t i = 0; i < m_connectCount; i++)
        {
            m_resFrameMacAddress.macArray[i] = nn::wlan::MacAddress(m_clientMacArray[i]);
            m_resFrameMacAddress.macCount++;
        }

        return true;
    }

    bool LocalApiClass::LocalStartTriggerPutActionFrame(uint8_t frameBuffer[], size_t size, nn::wlan::MacAddress& toMac, nn::wlan::MacAddress& fromMac) NN_NOEXCEPT
    {
        nn::Result result;
        bool isSuccess = false;

        for (int32_t i = 0; i < m_etherList.typeCount; i++)
        {
            isSuccess = LocalMakeActionFrame(frameBuffer, size, toMac, fromMac, m_etherList.etherTypes[i], TestMatchInfo[0], m_frameType);
            if (isSuccess != true)
            {
                NN_LOG("             LocalApi LocalMakeActionFrame Error\n");
                return false;
            }

            isSuccess = LocalMakeResMacFrame(frameBuffer, size, m_clientMacArray, m_connectCount);
            if (isSuccess != true)
            {
                NN_LOG("             LocalApi MakeReqMacFrame Error\n");
                return false;
            }

            result = nn::wlan::Local::PutFrameRaw(frameBuffer, size);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Line %d\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription(), __LINE__);
                return false;
            }
        }

        m_resFrameMacAddress.macCount = 0;
        for (int32_t i = 0; i < m_connectCount; i++)
        {
            m_resFrameMacAddress.macArray[i] = nn::wlan::MacAddress(m_clientMacArray[i]);
            m_resFrameMacAddress.macCount++;
        }

        return true;
    }

    void LocalApiClass::LocalResGetMacAddress(LocalPayloadData* pPayLoad, LoaclMacAddressInfo* pMacInfo) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pPayLoad);
        NN_ABORT_UNLESS_NOT_NULL(pMacInfo);

        LocalPayLoadMacReqData* pMacData = reinterpret_cast<LocalPayLoadMacReqData*>(pPayLoad->data);

        pMacInfo->macCount = 0;
        for (uint32_t i = 0; i < pMacData->len; i++)
        {
            pMacInfo->macArray[i] = nn::wlan::MacAddress(pMacData->macAddress[i]);
            pMacInfo->macCount++;
        }
    }

    void LocalApiClass::LocalResGetMacAddress(LocalActionFramePayloadData* pPayLoad, LoaclMacAddressInfo* pMacInfo) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pPayLoad);
        NN_ABORT_UNLESS_NOT_NULL(pMacInfo);

        LocalPayLoadMacReqData* pMacData = reinterpret_cast<LocalPayLoadMacReqData*>(pPayLoad->data);

        pMacInfo->macCount = 0;
        for (uint32_t i = 0; i < pMacData->len; i++)
        {
            pMacInfo->macArray[i] = nn::wlan::MacAddress(pMacData->macAddress[i]);
            pMacInfo->macCount++;
        }
    }

    bool LocalApiClass::LocalParseFrame(LocalConnectsInfo& connectsInfo, uint8_t frameBuffer[], size_t size) NN_NOEXCEPT
    {
        if (size < WlanTestGetFramMineSize)
        {
            NN_LOG("             LocalApi Parse Frame Data Size Mine(%lu)\n", size);
            return false;
        }

        LocalFrameHeader* pFrameHeader = reinterpret_cast<LocalFrameHeader*>(frameBuffer);
        LocalPayloadData* pPayLoad = reinterpret_cast<LocalPayloadData*>(pFrameHeader->data);
        int32_t clientPos = 0;
        bool isMatchMacAddress = false;

        if (nn::wlan::MacAddress(0x00, 0x00, 0x00, 0x00, 0x00, 0x00) == nn::wlan::MacAddress(pFrameHeader->toMacAddress))
        {
            char strMacAddress[nn::wlan::MacAddress::MacStringSize];
            nn::wlan::MacAddress(pFrameHeader->toMacAddress).GetString(strMacAddress);
            NN_LOG("             MacAddress Nothing Frame : %s\n", strMacAddress);
            return true;
        }

        // CLIENT検索(MACアドレス比較)
        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            if (connectsInfo.facing[i].fromMacAddres == nn::wlan::MacAddress(pFrameHeader->fromMacAddress)
                && connectsInfo.facing[i].frameType == GetLocalFrameType::FrameType_Frame)
            {
                isMatchMacAddress = true;
                clientPos = i;
                break;
            }
        }

#ifdef WLAN_TEST_STATE_ASSER_STOP
        LocalOpenModeType openMode = GetLocalOpenMode();
#else
        LocalOpenModeType openMode = GetLocalModeType();
#endif
        // モード条件
        switch (openMode)
        {
        case LocalOpenModeType::OpenMode_Master:
            // MASTERフレーム解析
            if (LocalMasterParseFrame(&connectsInfo.facing[clientPos], pFrameHeader, pPayLoad, isMatchMacAddress) != true)
            {
                return true;
            }

            break;
        case LocalOpenModeType::OpenMode_Client:
            // CLIENTフレーム解析
            if (LocalClientParseFrame(pPayLoad) != true)
            {
                return true;
            }
            break;
        default:
            NN_LOG("             LocalApi Parse Frame Open Mode found(%d)\n", openMode);
            return false;
        }

        if (pPayLoad->reqType == FrameReqType::ReqType_RawPingReq)
        {
            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[size]);
            uint8_t tmpBuf[100];
            nn::Result result;
            nn::wlan::MacAddress myMacAddress;

            NN_ABORT_UNLESS_NOT_NULL(putBuffer);
            std::memcpy(putBuffer.get(), frameBuffer, size);

            LocalFrameHeader* pFrameData    = reinterpret_cast<LocalFrameHeader*>(putBuffer.get());
            LocalPayloadData* pRawPayLoad   = reinterpret_cast<LocalPayloadData*>(pFrameData->data);
            nn::wlan::Local::GetMacAddress(&myMacAddress);

            /* Exchange Mac Address */
            std::memcpy(tmpBuf, pFrameData->fromMacAddress, nn::wlan::MacAddress::MacAddressSize);
            std::memcpy(pFrameData->fromMacAddress, myMacAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
            std::memcpy(pFrameData->toMacAddress, tmpBuf, nn::wlan::MacAddress::MacAddressSize);

            pRawPayLoad->reqType = FrameReqType::ReqType_RawPingRes;

            result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), size);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi PutFrame Error(module:%d destription:%d) Line %d\n", result.GetModule(), result.GetDescription(), __LINE__);
            }
        }

        if (connectsInfo.connectCount >= LocalApiClientMaxCount)
        {
            NN_LOG("             LocalApi Parse Frame connect Array Overflow\n");
            return false;
        }

        nn::os::LockMutex(&m_localApiLock);
        // 新規追加
        if (isMatchMacAddress != true)
        {
            clientPos = connectsInfo.connectCount;
            connectsInfo.facing[clientPos].fromMacAddres = nn::wlan::MacAddress(pFrameHeader->fromMacAddress);
            connectsInfo.facing[clientPos].frameType = GetLocalFrameType::FrameType_Frame;
            std::memcpy(connectsInfo.facing[clientPos].etherType, pFrameHeader->etherType, sizeof(connectsInfo.facing[clientPos].etherType));
            connectsInfo.connectCount++;
        }

        // NTD追加箇所
        if (pPayLoad->reqType == FrameReqType::ReqType_RawPingRes)
        {
            if (connectsInfo.facing[clientPos].isRttPkt != true)
            {
                NN_LOG("***+++ RTT PACKET RECV +++***\n");
                connectsInfo.facing[clientPos].isRttPkt = true;
            }
            uint8_t* pData = pPayLoad->data;
            int64_t tickOld = (int64_t)pData[0] | (int64_t)pData[1] << 8 |
                    (int64_t)pData[2] << 16 | (int64_t)pData[3] << 24 |
                    (int64_t)pData[4] << 32 | (int64_t)pData[5] << 40 |
                    (int64_t)pData[6] << 48 | (int64_t)pData[7] << 56;
            nn::os::Tick packetTick(tickOld);
            nn::TimeSpan rtt = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - packetTick);
            uint16_t rttTemp  = rtt.GetMilliSeconds();
            if (connectsInfo.facing[clientPos].rttCount == 0)
            {
                connectsInfo.facing[clientPos].rttAverage = rttTemp;
                connectsInfo.facing[clientPos].rttMin = rttTemp;
                connectsInfo.facing[clientPos].rttMax = rttTemp;
                connectsInfo.facing[clientPos].seqNumberRttRecent = pPayLoad->seq;
                connectsInfo.facing[clientPos].seqNumberRttGap = 0;
            }
            else
            {
                if (rttTemp < connectsInfo.facing[clientPos].rttMin)
                {
                    connectsInfo.facing[clientPos].rttMin = rttTemp;
                }
                if (rttTemp > connectsInfo.facing[clientPos].rttMax)
                {
                    connectsInfo.facing[clientPos].rttMax = rttTemp;
                }
                connectsInfo.facing[clientPos].rttAverage = (connectsInfo.facing[clientPos].rttAverage * connectsInfo.facing[clientPos].rttCount + rttTemp) / (connectsInfo.facing[clientPos].rttCount + 1);
                if (pPayLoad->seq - connectsInfo.facing[clientPos].seqNumberRttRecent > connectsInfo.facing[clientPos].seqNumberRttGap)
                {
                    connectsInfo.facing[clientPos].seqNumberRttGap = pPayLoad->seq - connectsInfo.facing[clientPos].seqNumberRttRecent;
                }
                connectsInfo.facing[clientPos].seqNumberRttRecent = pPayLoad->seq;
            }
            connectsInfo.facing[clientPos].rttCount++;
            connectsInfo.facing[clientPos].isGetFrame = true;
            connectsInfo.facing[clientPos].isConnected = true;
            nn::os::UnlockMutex(&m_localApiLock);
            return true;
        }
        else if (pPayLoad->reqType == FrameReqType::ReqType_RawPingReq)
        {
            if (connectsInfo.facing[clientPos].isRttPkt != true)
            {
                NN_LOG("***+++ RTT PACKET RECV +++***\n");
                connectsInfo.facing[clientPos].isRttPkt = true;
            }
            connectsInfo.facing[clientPos].getFrameCount++;
            connectsInfo.facing[clientPos].isGetFrame = true;
            connectsInfo.facing[clientPos].isConnected = true;
            nn::os::UnlockMutex(&m_localApiLock);
            return true;
        }
        else
        {
            // 処理なし(コーディング規約修正)
        }

        // 初回取得判定
        if (connectsInfo.facing[clientPos].isGetFrame != true)
        {
            connectsInfo.facing[clientPos].seqNum = pPayLoad->seq;
        }
        else
        {
            connectsInfo.facing[clientPos].seqNum++;
        }

        // ロストフレーム算出
        bool isLostFrame = false;
        if (connectsInfo.facing[clientPos].seqNum != pPayLoad->seq)
        {
            char fromMac[nn::wlan::MacAddress::MacStringSize];
            char toMac[nn::wlan::MacAddress::MacStringSize];
            uint32_t compareSeq = 0;
            if (connectsInfo.facing[clientPos].seqNum > pPayLoad->seq)
            {
                if ((connectsInfo.facing[clientPos].seqNum - 1) == pPayLoad->seq)
                {
                    // 重複受信パケット条件
                    NN_LOG("             LocalApi From MacAddress[%s] To MacAddress[%s] Duplicate FramePacket Compare Seq : %lu Frame Seq: %lu\n",
                        nn::wlan::MacAddress(pFrameHeader->fromMacAddress).GetString(fromMac),
                        nn::wlan::MacAddress(pFrameHeader->toMacAddress).GetString(toMac),
                        connectsInfo.facing[clientPos].seqNum, pPayLoad->seq);
                    compareSeq++;
                }
                else if ((connectsInfo.facing[clientPos].seqNum - pPayLoad->seq) > 0x8FFFFFFF)
                {
                    // シーケンス番号巻き戻り対策(純粋に桁MAXからの反転なら0x8FFFFFFFを超える条件)
                    isLostFrame = true;
                    compareSeq = ((0xFFFFFFFF - connectsInfo.facing[clientPos].seqNum) + pPayLoad->seq);
                }
                else
                {
                    // シーケンス番号の巻き戻りが発生したらここに来る
                    NN_LOG("             LocalApi From MacAddress[%s] To MacAddress[%s] Back SeqNumber Lost FramePacket Compare Seq : %lu Frame Seq: %lu\n",
                        nn::wlan::MacAddress(pFrameHeader->fromMacAddress).GetString(fromMac),
                        nn::wlan::MacAddress(pFrameHeader->toMacAddress).GetString(toMac),
                        connectsInfo.facing[clientPos].seqNum, pPayLoad->seq);
                    compareSeq++;
                }
            }
            else
            {
                isLostFrame = true;
                compareSeq = (pPayLoad->seq - connectsInfo.facing[clientPos].seqNum);
            }

            if (compareSeq > connectsInfo.facing[clientPos].seqNumberGap)
            {
                connectsInfo.facing[clientPos].seqNumberGap = compareSeq;
            }
            connectsInfo.facing[clientPos].lostFrameCount += compareSeq;
            connectsInfo.facing[clientPos].seqNum = pPayLoad->seq;
        }
#ifdef WLAN_TEST_SENDRECVDATA_TRACE
        else
        {
            char fromMac[nn::wlan::MacAddress::MacStringSize];
            char toMac[nn::wlan::MacAddress::MacStringSize];
            NN_LOG("             LocalApi From MacAddress[%s] To MacAddress[%d] Pass FramePacket Frame Seq: %lu\n",
                    nn::wlan::MacAddress(pFrameHeader->fromMacAddress).GetString(fromMac),
                    nn::wlan::MacAddress(pFrameHeader->toMacAddress).GetString(toMac),
                    pPayLoad->seq);
        }
#endif
#ifdef WLAN_TEST_LOSTFRAME_TRACE
        if (isLostFrame == true)
        {
            char fromMac[nn::wlan::MacAddress::MacStringSize];
            char toMac[nn::wlan::MacAddress::MacStringSize];
            NN_LOG("             LocalApi From MacAddress[%s] To MacAddress[%s] Lost FramePacket Compare Seq : %lu Frame Seq: %lu\n",
                nn::wlan::MacAddress(pFrameHeader->fromMacAddress).GetString(fromMac),
                nn::wlan::MacAddress(pFrameHeader->toMacAddress).GetString(toMac),
                connectsInfo.facing[clientPos].seqNum, pPayLoad->seq);
        }
#endif

        connectsInfo.facing[clientPos].getFrameCount++;
        connectsInfo.facing[clientPos].totalFrameSize += size;
        connectsInfo.facing[clientPos].isGetFrame = true;
        connectsInfo.facing[clientPos].isConnected = true;
        if (connectsInfo.facing[clientPos].lostFrameCount > 0)
        {
            double totalFrame   = static_cast<double>(connectsInfo.facing[clientPos].getFrameCount + connectsInfo.facing[clientPos].lostFrameCount);
            double lostFrame    = static_cast<double>(connectsInfo.facing[clientPos].lostFrameCount);
            connectsInfo.facing[clientPos].perRate = (lostFrame / totalFrame) * 100;
            if (connectsInfo.facing[clientPos].perRate < 0.001)
            {
                connectsInfo.facing[clientPos].perRate = 0.001;
            }
        }
        nn::os::UnlockMutex(&m_localApiLock);

        return true;
    }// NOLINT(impl/function_size) WLANパケット解析が試験毎に必要なため

    bool LocalApiClass::LocalMasterParseFrame(LocalClientInfoDetail* pDetailInfo, LocalFrameHeader *pHeader, LocalPayloadData* pPayload, bool isFirstFrame) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pDetailInfo);
        NN_ABORT_UNLESS_NOT_NULL(pHeader);
        NN_ABORT_UNLESS_NOT_NULL(pPayload);

        if (isFirstFrame != true)
        {
            bool isConnectMatch = false;
            for (int32_t i = 0; i < m_connectCount; i++)
            {
                if (m_clientMacArray[i] == nn::wlan::MacAddress(pHeader->fromMacAddress))
                {
                    isConnectMatch = true;
                    break;
                }
            }

            // MASTER待機で取得したMacAddressと一致しない場合に開始フレーム送信
            if (isConnectMatch != true)
            {
                nn::os::LockMutex(&m_localApiLock);
                pPayload->reqType = FrameReqType::ReqType_MacAddressReq;
                nn::os::UnlockMutex(&m_localApiLock);

                m_connectCount = GetClientMacAddress(m_clientMacArray, nn::wlan::ConnectionState_Connected);
            }
        }

        // Macアドレス一覧要求があれば一覧を返す
        if (pPayload->reqType == FrameReqType::ReqType_MacAddressReq)
        {
            nn::wlan::MacAddress putMac(pHeader->fromMacAddress);
            char clientMacArray[nn::wlan::MacAddress::MacStringSize];
            nn::wlan::Local::GetMacAddress(&m_macAddress);

            NN_LOG("             LocalApi Client Request MacAddress List PutFrame MacAddress : %s\n", putMac.GetString(clientMacArray));
            LocalStartTriggerPutFrame(m_startPutBuffer.get(), m_startPutSize, putMac, m_macAddress);
            return false;
        }

        return true;
    }

    bool LocalApiClass::LocalClientParseFrame(LocalPayloadData* pPayload) NN_NOEXCEPT
    {
        if (pPayload->reqType == FrameReqType::ReqType_MacAddressRes)
        {
            nn::os::LockMutex(&m_localApiLock);
            LocalResGetMacAddress(pPayload, &m_resFrameMacAddress);
            nn::os::UnlockMutex(&m_localApiLock);

            nn::os::SignalEvent(&m_receiveFrameEvent);
            return false;
        }
        return true;
    }

    bool LocalApiClass::LocalParseActionFrame(LocalConnectsInfo& connectsInfo, uint8_t frameBuffer[], size_t size) NN_NOEXCEPT
    {
        if (size < WlanTestGetFramMineSize)
        {
            return false;
        }

        LocalActionFrameData* pFrameHeader = reinterpret_cast<LocalActionFrameData*>(frameBuffer);
        LocalActionFramePayloadData* pPayLoad = reinterpret_cast<LocalActionFramePayloadData*>(pFrameHeader->data);

        int32_t clientPos = 0;
        bool isMatchMacAddress = false;

        // CLIENT検索(MACアドレス比較)
        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            if (connectsInfo.facing[i].fromMacAddres == nn::wlan::MacAddress(pFrameHeader->fromMacAddress)
                && connectsInfo.facing[i].frameType == GetLocalFrameType::FrameType_ActionFrame)
            {
                isMatchMacAddress = true;
                clientPos = i;
                break;
            }
        }

#ifdef WLAN_TEST_STATE_ASSER_STOP
        LocalOpenModeType openMode = GetLocalOpenMode();
#else
        LocalOpenModeType openMode = GetLocalModeType();
#endif

        // モード条件
        switch (openMode)
        {
        case LocalOpenModeType::OpenMode_Master:
            // MASTERフレーム解析
            if (LocalMasterParseActionFrame(&connectsInfo.facing[clientPos], pFrameHeader, pPayLoad, isMatchMacAddress) != true)
            {
                return true;
            }
            break;
        case LocalOpenModeType::OpenMode_Client:
            // CLIENTフレーム解析
            if (LocalClientParseActionFrame(pPayLoad) != true)
            {
                return true;
            }
            break;
        default:
            NN_LOG("             LocalApi Parse Frame Open Mode found(%d)\n", openMode);
            return false;
        }

        // CLIENT検索(MACアドレス比較)
        for (int32_t i = 0; i < m_actionMaxConnect; i++)
        {
            if (connectsInfo.facing[i].fromMacAddres == nn::wlan::MacAddress(pFrameHeader->fromMacAddress)
                && connectsInfo.facing[i].frameType == GetLocalFrameType::FrameType_ActionFrame)
            {
                isMatchMacAddress = true;
                clientPos = i;
                break;
            }
        }

        nn::os::LockMutex(&m_localApiLock);

        // CLIENT新規追加
        if (isMatchMacAddress != true)
        {
            clientPos = connectsInfo.connectCount;
            connectsInfo.facing[clientPos].fromMacAddres = nn::wlan::MacAddress(pFrameHeader->fromMacAddress);
            connectsInfo.facing[clientPos].frameType = GetLocalFrameType::FrameType_ActionFrame;
            connectsInfo.connectCount++;
        }

        if (connectsInfo.facing[clientPos].isGetFrame != true)
        {
            connectsInfo.facing[clientPos].seqNum = pPayLoad->seq;
        }
        else
        {
            connectsInfo.facing[clientPos].seqNum++;
        }

        // ロストフレーム算出
        if (connectsInfo.facing[clientPos].seqNum != pPayLoad->seq)
        {
            uint32_t compareSeq = 0;
#ifdef WLAN_TEST_LOSTFRAME_TRACE
            char fromMac[nn::wlan::MacAddress::MacStringSize];
            NN_LOG("             LocalApi From MacAddress[%s] Lost ActionFramePacket Compare Seq : %lu Frame Seq: %lu\n",
                nn::wlan::MacAddress(pFrameHeader->fromMacAddress).GetString(fromMac),
                connectsInfo.facing[clientPos].seqNum, pPayLoad->seq);
#endif
            if (connectsInfo.facing[clientPos].seqNum > pPayLoad->seq)
            {
                // シーケンス番号巻き戻り対策(純粋に桁MAXからの反転なら0x8FFFFFFFを超える条件)
                if ((connectsInfo.facing[clientPos].seqNum - pPayLoad->seq) > 0x8FFFFFFF)
                {
                    compareSeq = ((0xFFFFFFFF - connectsInfo.facing[clientPos].seqNum) + pPayLoad->seq);
                }
                else
                {
                    compareSeq++;
                }
            }
            else
            {
                compareSeq = (pPayLoad->seq - connectsInfo.facing[clientPos].seqNum);
            }

            if (compareSeq > connectsInfo.facing[clientPos].seqNumberGap)
            {
                connectsInfo.facing[clientPos].seqNumberGap = compareSeq;
            }
            connectsInfo.facing[clientPos].lostFrameCount += compareSeq;
            connectsInfo.facing[clientPos].seqNum = pPayLoad->seq;
        }

        connectsInfo.facing[clientPos].getFrameCount++;
        connectsInfo.facing[clientPos].totalFrameSize += size;
        connectsInfo.facing[clientPos].isGetFrame = true;
        connectsInfo.facing[clientPos].isConnected = true;
        if (connectsInfo.facing[clientPos].lostFrameCount > 0)
        {
            double totalFrame = static_cast<double>(connectsInfo.facing[clientPos].getFrameCount + connectsInfo.facing[clientPos].lostFrameCount);
            double lostFrame = static_cast<double>(connectsInfo.facing[clientPos].lostFrameCount);
            connectsInfo.facing[clientPos].perRate = (lostFrame / totalFrame) * 100;
        }

        nn::os::UnlockMutex(&m_localApiLock);
        return true;
    }// NOLINT(impl/function_size) WLANパケット解析が試験毎に必要なため

    bool LocalApiClass::LocalMasterParseActionFrame(LocalClientInfoDetail *pDetailInfo, LocalActionFrameData *pHeader, LocalActionFramePayloadData* pPayload, bool isFirstFrame) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pDetailInfo);
        NN_ABORT_UNLESS_NOT_NULL(pHeader);
        NN_ABORT_UNLESS_NOT_NULL(pPayload);

        if (isFirstFrame != true)
        {
            bool isConnectMatch = false;
            for (int32_t i = 0; i < m_connectCount; i++)
            {
                if (m_clientMacArray[i] == nn::wlan::MacAddress(pHeader->fromMacAddress))
                {
                    isConnectMatch = true;
                    break;
                }
            }
        }

        return true;
    }

    bool LocalApiClass::LocalClientParseActionFrame(LocalActionFramePayloadData* pPayload) NN_NOEXCEPT
    {
        if (pPayload->reqType == FrameReqType::ReqType_MacAddressRes)
        {
            nn::os::LockMutex(&m_localApiLock);
            LocalResGetMacAddress(pPayload, &m_resFrameMacAddress);
            nn::os::UnlockMutex(&m_localApiLock);

            nn::os::SignalEvent(&m_receiveFrameEvent);
            return false;
        }
        return true;
    }

    bool LocalApiClass::LocalClientSendActionFrame(uint8_t* pPutBuffer, size_t putSize,
        uint16_t channel, nn::wlan::ActionFrameType frameType) NN_NOEXCEPT
    {
        nn::Result result;
        bool isSuccess;

        // ActionFrame送信
        LoaclMacAddressInfo macList;
        LocalGetMacList(macList);
        for (int32_t i = 0; i < macList.macCount; i++)
        {
            nn::wlan::MacAddress sendMacAddress = macList.macArray[i];
            if (sendMacAddress == m_macAddress)
            {
                continue;
            }

            for (int32_t cnt = 0; cnt < m_etherList.typeCount; cnt++)
            {
                isSuccess = LocalMakeActionFrame(pPutBuffer, putSize, sendMacAddress,
                    m_macAddress, m_etherList.etherTypes[cnt], TestMatchInfo[0], frameType, m_seqArray[i]);
                if (isSuccess != true)
                {
                    NN_LOG("             LocalApi Client LocalMakeActionFrame Error\n");
                    break;
                }

                result = nn::wlan::Local::PutActionFrameOneShot(sendMacAddress, pPutBuffer, putSize, channel, m_sendTimeOut);
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalApi Client PutActionFrameOneShot Error(0x%08x)\n", result.GetInnerValueForDebug());
                }
                else
                {
                    if ((cnt + 1) == m_etherList.typeCount)
                    {
                        m_seqArray[i]++;
                    }

                    m_connectsInfo.totalSendSize += putSize;
                }
            }
        }

        return isSuccess;
    }

    bool LocalApiClass::LocalMasterSendActionFrame(uint8_t* pPutBuffer, size_t putSize,
        uint16_t channel, nn::wlan::ActionFrameType frameType) NN_NOEXCEPT
    {
        // ActionFrame送信
        LoaclMacAddressInfo macList;
        nn::wlan::ConnectionStatus conStatus;
        nn::Result result;
        bool isSuccess;

        LocalGetMacList(macList);

        result = nn::wlan::Local::GetConnectionStatus(&conStatus);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             LocalApi Master GetConnectionStatus Error(0x%08x) Line(%d)\n", result.GetInnerValueForDebug(), __LINE__);
            return false;
        }

        // Master送信
        for (int32_t cnt = 0; cnt < m_etherList.typeCount; cnt++)
        {
            isSuccess = LocalMakeActionFrame(pPutBuffer, putSize, conStatus.bssid,
                m_macAddress, m_etherList.etherTypes[cnt], TestMatchInfo[0], frameType, m_seqNumber);
            if (isSuccess != true)
            {
                NN_LOG("             LocalApi Master LocalMakeActionFrame Error\n");
                break;
            }

            result = nn::wlan::Local::PutActionFrameOneShot(conStatus.bssid, pPutBuffer, putSize, channel, m_sendTimeOut);
            if (result.IsSuccess() != true)
            {
                NN_LOG("            LocalApi Master PutActionFrameOneShot Error(0x%08x)\n", result.GetInnerValueForDebug());
            }
            else
            {
                if ((cnt + 1) == m_etherList.typeCount)
                {
                    m_seqNumber++;
                }
                m_connectsInfo.totalSendSize += putSize;
            }
        }

        return isSuccess;
    }

    void LocalApiClass::ConfigureInterface(const char* interfaceName, IpV4Address ipAddr, bool enable) NN_NOEXCEPT
    {
        nn::Result result;

        nn::socket::IfReq ifr = { WLAN_INTERFACE_NAME };
        ifr.ifr_flags = nn::socket::IfrFlag::Iff_Up;
        int socket = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        NN_ABORT_UNLESS(socket >= 0);
        nn::socket::Ioctl(socket, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocSIfFlags), &ifr, sizeof(ifr));

        if (enable)
        {
            nn::bsdsocket::cfg::IfSettings ifcfg;
            memset(&ifcfg, 0, sizeof(ifcfg));

            if (ipAddr.isDhsp != true)
            {
                // static ip
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
                ifcfg.mtu = 1500;

                nn::socket::InetAton(ipAddr.strIpAddress, &ifcfg.u.modeStatic.addr);
                nn::socket::InetAton(ipAddr.strGateWay, &ifcfg.u.modeStatic.gatewayAddr);
                nn::socket::InetAton(ipAddr.strSubNetMask, &ifcfg.u.modeStatic.subnetMask);
                ifcfg.u.modeStatic.broadcastAddr.S_addr =
                    (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
                    ~ifcfg.u.modeStatic.subnetMask.S_addr;
                nn::socket::InetAton(ipAddr.strDns1, &ifcfg.dnsAddrs[0]);
                nn::socket::InetAton(ipAddr.strDns2, &ifcfg.dnsAddrs[1]);
            }
            else
            {
                // dhcp
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
                ifcfg.mtu = 1500;
            }

            result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
            if (result.IsFailure())
            {
                NN_LOG("             failed to configure interface %s - %d:%d\n",
                    interfaceName,
                    result.GetModule(),
                    result.GetDescription());
            }
        }
        else
        {
            nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
        }

        nn::socket::Close(socket);
    }

    void LocalApiClass::LocalSendFrameThread(void* arg) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(arg);
        LocalApiClass* pThis = static_cast<LocalApiClass*>(arg);
        nn::wlan::ConnectionStatus conStatus;
        size_t putSize = pThis->m_putSize;
        bool isClientToClient = pThis->m_isClientToClient;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        int64_t sendInterval = pThis->m_sendInterval;
        int64_t sendTime = pThis->m_sendTime;
        nn::os::Tick startTick, interValTick;
        uint32_t sendCnt;
        uint32_t errorCount;
        nn::Result result;
        bool isSuccess;
#ifdef WLAN_TEST_SENDRECVDATA_TRACE
        char fromMac[nn::wlan::MacAddress::MacStringSize];
        char toMac[nn::wlan::MacAddress::MacStringSize];
#endif
        NN_ABORT_UNLESS_NOT_NULL(putBuffer);

        std::memset(putBuffer.get(), 0x00, putSize);

        result = nn::wlan::Local::GetConnectionStatus(&conStatus);
        if (result.IsSuccess() != true)
        {
            NN_LOG("            LocalApi GetConnectionStatus Error(0x%08x) Line(%d)\n", result.GetInnerValueForDebug(), __LINE__);
            return;
        }

#ifdef WLAN_TEST_STATE_ASSER_STOP
        LocalOpenModeType openMode = pThis->GetLocalOpenMode();
#else
        LocalOpenModeType openMode = pThis->GetLocalModeType();
#endif
        errorCount = 0;
        sendCnt = 0;
        pThis->m_sendResult = nn::ResultSuccess();
        pThis->m_connectsInfo.isSendStart = false;
        pThis->m_connectsInfo.totalSendSize = 0;
        pThis->LocalClearSeqNumber();
        startTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds() < sendTime)
        {
            if (pThis->m_isSendFrame != true)
            {
                break;
            }

            if (pThis->m_connectsInfo.isSendStart != true)
            {
                pThis->m_connectsInfo.isSendStart = true;
                pThis->m_connectsInfo.startSendTick = nn::os::GetSystemTick();
            }

            // CLIENT送信
            if (openMode == OpenMode_Master || isClientToClient == true)
            {
                LoaclMacAddressInfo macList;
                pThis->LocalGetMacList(macList);
                for (int32_t i = 0; i < macList.macCount; i++)
                {
                    nn::wlan::MacAddress sendMacAddress = macList.macArray[i];
                    if (sendMacAddress == pThis->m_macAddress)
                    {
                        continue;
                    }

                    for (int32_t cnt = 0; cnt < pThis->m_etherList.typeCount; cnt++)
                    {
                        isSuccess = pThis->LocalMakeFrame(putBuffer.get(), putSize, sendMacAddress,
                            pThis->m_macAddress, pThis->m_etherList.etherTypes[cnt], TestMatchInfo[0], pThis->m_seqArray[i]);
                        if (isSuccess != true)
                        {
                            NN_LOG("            LocalApi LocalMakeFrame Error\n");
                            break;
                        }

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
                        NN_LOG("             LocalApi PutFrameRaw From MacAddress[%s] To MacAddress[%s] : Seq Number : %lu\n",
                            pThis->m_macAddress.GetString(fromMac), sendMacAddress.GetString(toMac), seqNumber);
#endif
                        sendCnt++;
                        result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), putSize);
                        if (pThis->m_isSendFrame != true)
                        {
                            continue;
                        }
                        if (result.IsSuccess() != true)
                        {
                            if (pThis->m_sendResult.GetInnerValueForDebug() != result.GetInnerValueForDebug())
                            {
                                if (pThis->m_sendResult.IsSuccess() != true)
                                {
                                    NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Error Count(%lu/%lu) Line %d\n",
                                        pThis->m_sendResult.GetInnerValueForDebug(), pThis->m_sendResult.GetModule(), pThis->m_sendResult.GetDescription(), errorCount, sendCnt, __LINE__);
                                    errorCount = 0;
                                }
                                NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Line %d\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription(), __LINE__);
                                pThis->m_sendResult = result;
                            }
                            errorCount++;
                        }
                        else
                        {
                            if ((cnt + 1) == pThis->m_etherList.typeCount)
                            {
                                pThis->m_seqArray[i]++;
                            }
                            pThis->m_connectsInfo.totalSendSize += putSize;

                            if (sendInterval > 0)
                            {
                                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sendInterval));
                            }
                        }
                    }
                }
            }

            // Master以外送信
            if (openMode != OpenMode_Master)
            {
                // Master送信
                for (int32_t cnt = 0; cnt < pThis->m_etherList.typeCount; cnt++)
                {
                    isSuccess = pThis->LocalMakeFrame(putBuffer.get(), putSize, conStatus.bssid,
                        pThis->m_macAddress, pThis->m_etherList.etherTypes[cnt], TestMatchInfo[0], pThis->m_seqNumber);
                    if (isSuccess != true)
                    {
                        NN_LOG("            LocalApi LocalMakeFrame Error\n");
                        break;
                    }
#ifdef WLAN_TEST_SENDRECVDATA_TRACE
                    NN_LOG("             LocalApi PutFrameRaw From MacAddress[%s] To MacAddress[%s] : Seq Number : %lu\n",
                        pThis->m_macAddress.GetString(fromMac), conStatus.bssid.GetString(toMac), mSeqNumber);
#endif
                    sendCnt++;
                    result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), putSize);
                    if (pThis->m_isSendFrame != true)
                    {
                        continue;
                    }
                    if (result.IsSuccess() != true)
                    {
                        if (pThis->m_sendResult.GetInnerValueForDebug() != result.GetInnerValueForDebug())
                        {
                            if (pThis->m_sendResult.IsSuccess() != true)
                            {
                                NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Error Count(%lu/%lu) Line %d\n",
                                    pThis->m_sendResult.GetInnerValueForDebug(), pThis->m_sendResult.GetModule(), pThis->m_sendResult.GetDescription(), errorCount, sendCnt, __LINE__);
                                errorCount = 0;
                            }
                            NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Line %d\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription(), __LINE__);
                            pThis->m_sendResult = result;
                        }
                        errorCount++;
                    }
                    else
                    {
                        if ((cnt + 1) == pThis->m_etherList.typeCount)
                        {
                            pThis->m_seqNumber++;
                        }
                        pThis->m_connectsInfo.totalSendSize += putSize;

                        if (sendInterval > 0)
                        {
                            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sendInterval));
                        }
                    }
                }
            }
        }
        if (errorCount > 0)
        {
            NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Error Count(%lu/%lu) Line %d\n",
                pThis->m_sendResult.GetInnerValueForDebug(), pThis->m_sendResult.GetModule(), pThis->m_sendResult.GetDescription(), errorCount, sendCnt, __LINE__);
        }
    }// NOLINT(impl/function_size) WLANフレーム調査ログ削除でチェックは通ります

    void LocalApiClass::LocalSendFrameOneShotWaitThread(void* arg) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(arg);
        LocalApiClass* pThis = static_cast<LocalApiClass*>(arg);
        nn::wlan::ConnectionStatus conStatus;
        size_t putSize = pThis->m_putSize;
        bool isClientToClient = pThis->m_isClientToClient;
        bool isNeedReply = pThis->m_isNeedReply;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        int64_t sendInterval = pThis->m_sendInterval;
        int64_t sendTime = pThis->m_sendTime;
        nn::os::Tick startTick;
        nn::Result result;
        uint32_t sendCnt;
        uint32_t errorCount;
        bool isSuccess;
#ifdef WLAN_TEST_SENDRECVDATA_TRACE
        char fromMac[nn::wlan::MacAddress::MacStringSize];
        char toMac[nn::wlan::MacAddress::MacStringSize];
#endif
        NN_ABORT_UNLESS_NOT_NULL(putBuffer);

        std::memset(putBuffer.get(), 0x00, putSize);

        result = nn::wlan::Local::GetConnectionStatus(&conStatus);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             LocalApi GetConnectionStatus Error(0x%08x) Line(%d)\n", result.GetInnerValueForDebug(), __LINE__);
            return;
        }

#ifdef WLAN_TEST_STATE_ASSER_STOP
        LocalOpenModeType openMode = pThis->GetLocalOpenMode();
#else
        LocalOpenModeType openMode = pThis->GetLocalModeType();
#endif
        pThis->m_isRttSnd = false;
        errorCount = 0;
        pThis->m_sendResult = nn::ResultSuccess();
        pThis->m_connectsInfo.isSendStart = false;
        pThis->m_connectsInfo.totalSendSize = 0;
        pThis->LocalClearSeqNumber();
        startTick = nn::os::GetSystemTick();
        sendCnt = 0;
        while ((nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds() < sendTime)
        {
            if (pThis->m_isSendFrame != true)
            {
                break;
            }

            if (pThis->m_connectsInfo.isSendStart != true)
            {
                pThis->m_connectsInfo.isSendStart = true;
                pThis->m_connectsInfo.startSendTick = nn::os::GetSystemTick();
            }

            // MASTER又はCLIENT間送信
            if (openMode == OpenMode_Master || isClientToClient == true)
            {
                LoaclMacAddressInfo macList;
                pThis->LocalGetMacList(macList);
                for (int32_t i = 0; i < macList.macCount; i++)
                {
                    nn::wlan::MacAddress sendMacAddress = macList.macArray[i];
                    if (sendMacAddress == pThis->m_macAddress)
                    {
                        continue;
                    }

                    for (int32_t cnt = 0; cnt < pThis->m_etherList.typeCount; cnt++)
                    {
                        // NTD追加箇所(条件は省略しないコーディング規約修正)
                        if (isNeedReply == true)
                        {
                            if (pThis->m_isRttSnd != true)
                            {
                                NN_LOG("***+++ RTT PACKET SEND +++***\n\n");
                                pThis->m_isRttSnd = true;
                            }
                            isSuccess = pThis->LocalMakeFrame(putBuffer.get(), putSize, sendMacAddress,
                                                              pThis->m_macAddress, pThis->m_etherList.etherTypes[cnt], TestMatchInfo[0], pThis->m_seqArray[i], isNeedReply);
                        }
                        else
                        {
                            isSuccess = pThis->LocalMakeFrame(putBuffer.get(), putSize, sendMacAddress,
                                                              pThis->m_macAddress, pThis->m_etherList.etherTypes[cnt], TestMatchInfo[0], pThis->m_seqArray[i]);
                        }
                        if (isSuccess != true)
                        {
                            NN_LOG("             LocalApi LocalMakeFrame Error\n");
                            break;
                        }

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
                        NN_LOG("             LocalApi PutFrameRaw From MacAddress[%s] To MacAddress[%s] : Seq Number : %lu\n",
                            pThis->m_macAddress.GetString(fromMac), sendMacAddress.GetString(toMac), seqNumber);
#endif
                        sendCnt++;
                        result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), putSize);
                        if (pThis->m_isSendFrame != true)
                        {
                            continue;
                        }
                        if (result.IsSuccess() != true)
                        {
                            if (pThis->m_sendResult.GetInnerValueForDebug() != result.GetInnerValueForDebug())
                            {
                                if (pThis->m_sendResult.IsSuccess() != true)
                                {
                                    NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Error Count(%lu/%lu) Line %d\n",
                                        pThis->m_sendResult.GetInnerValueForDebug(), pThis->m_sendResult.GetModule(), pThis->m_sendResult.GetDescription(), errorCount, sendCnt, __LINE__);
                                    errorCount = 0;
                                }
                                NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Line %d\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription(), __LINE__);
                                pThis->m_sendResult = result;
                            }
                            errorCount++;
                        }
                        else
                        {
                            pThis->m_connectsInfo.totalSendSize += putSize;
                            if ((cnt + 1) == pThis->m_etherList.typeCount)
                            {
                                pThis->m_seqArray[i]++;
                            }

                            if (sendInterval > 0)
                            {
                                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sendInterval));
                            }
                        }
                    }
                }
            }

            // Master以外送信
            if (openMode != OpenMode_Master)
            {
                // Master送信
                for (int32_t cnt = 0; cnt < pThis->m_etherList.typeCount; cnt++)
                {
                    // NTD追加箇所(条件は省略しないコーディング規約修正)
                    if (isNeedReply == true)
                    {
                        if (pThis->m_isRttSnd != true)
                        {
                            NN_LOG("***+++ RTT PACKET SEND +++***\n\n");
                            pThis->m_isRttSnd = true;
                        }
                        isSuccess = pThis->LocalMakeFrame(putBuffer.get(), putSize, conStatus.bssid,
                                                          pThis->m_macAddress, pThis->m_etherList.etherTypes[cnt], TestMatchInfo[0], pThis->m_seqNumber, isNeedReply);
                    }
                    else
                    {
                        isSuccess = pThis->LocalMakeFrame(putBuffer.get(), putSize, conStatus.bssid,
                                                          pThis->m_macAddress, pThis->m_etherList.etherTypes[cnt], TestMatchInfo[0], pThis->m_seqNumber);
                    }
                    if (isSuccess != true)
                    {
                        NN_LOG("             LocalApi LocalMakeFrame Error\n");
                        break;
                    }
#ifdef WLAN_TEST_SENDRECVDATA_TRACE
                    NN_LOG("             LocalApi PutFrameRaw From MacAddress[%s] To MacAddress[%s] : Seq Number : %lu\n",
                        pThis->m_macAddress.GetString(fromMac), conStatus.bssid.GetString(toMac), seqNumber);
#endif
                    sendCnt++;
                    result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), putSize);
                    if (pThis->m_isSendFrame != true)
                    {
                        continue;
                    }
                    if (result.IsSuccess() != true)
                    {
                        if (pThis->m_sendResult.GetInnerValueForDebug() != result.GetInnerValueForDebug())
                        {
                            if (pThis->m_sendResult.IsSuccess() != true)
                            {
                                NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Error Count(%lu/%lu) Line %d\n",
                                    pThis->m_sendResult.GetInnerValueForDebug(), pThis->m_sendResult.GetModule(), pThis->m_sendResult.GetDescription(), errorCount, sendCnt, __LINE__);
                                errorCount = 0;
                            }
                            NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Line %d\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription(), __LINE__);
                            pThis->m_sendResult = result;
                        }
                        errorCount++;
                    }
                    else
                    {
                        if ((cnt + 1) == pThis->m_etherList.typeCount)
                        {
                            pThis->m_seqNumber++;
                        }
                        pThis->m_connectsInfo.totalSendSize += putSize;

                        if (sendInterval > 0)
                        {
                            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sendInterval));
                        }
                    }
                }
            }
        }
        if (errorCount > 0)
        {
            NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Error Count(%lu/%lu) Line %d\n",
                pThis->m_sendResult.GetInnerValueForDebug(), pThis->m_sendResult.GetModule(), pThis->m_sendResult.GetDescription(), errorCount, sendCnt, __LINE__);
        }
    }// NOLINT(impl/function_size) WLANフレーム調査ログ削除でチェックは通ります

    void LocalApiClass::LocalBroadcastSendFrameThread(void* arg) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(arg);
        LocalApiClass* pThis = static_cast<LocalApiClass*>(arg);
        size_t putSize = pThis->m_putSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        int64_t sendInterval = pThis->m_sendInterval;
        int64_t sendTime = pThis->m_sendTime;
        nn::os::Tick startTick;
        uint32_t errorCount, sendCnt;
        nn::Result result;
        bool isSuccess;
#ifdef WLAN_TEST_SENDRECVDATA_TRACE
        char fromMac[nn::wlan::MacAddress::MacStringSize];
        char toMac[nn::wlan::MacAddress::MacStringSize];
#endif
        NN_ABORT_UNLESS_NOT_NULL(putBuffer);
        std::memset(putBuffer.get(), 0x00, putSize);

        if (pThis->m_connectsInfo.isSendStart != true)
        {
            pThis->m_connectsInfo.isSendStart = false;
        }

        sendCnt = 0;
        pThis->m_connectsInfo.totalSendSize = 0;
        pThis->LocalClearSeqNumber();
        startTick = nn::os::GetSystemTick();
        pThis->m_sendResult = nn::ResultSuccess();
        while ((nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds() < sendTime)
        {
            if (pThis->m_isSendFrame != true)
            {
                break;
            }

            if (pThis->m_connectsInfo.isSendStart != true)
            {
                pThis->m_connectsInfo.isSendStart = true;
                pThis->m_connectsInfo.startSendTick = nn::os::GetSystemTick();
            }

            // ブロードキャスト送信
            for (int32_t cnt = 0; cnt < pThis->m_etherList.typeCount; cnt++)
            {
                isSuccess = pThis->LocalMakeFrame(putBuffer.get(), putSize, nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                    pThis->m_macAddress, pThis->m_etherList.etherTypes[cnt], TestMatchInfo[0], pThis->m_seqNumber);
                if (isSuccess != true)
                {
                    NN_LOG("             LocalApi Broadcast LocalMakeFrame Error\n");
                    break;
                }

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
                NN_LOG("             LocalApi Broadcast PutFrameRaw From MacAddress[%s] To MacAddress[%s] : Seq Number : %lu\n",
                    pThis->m_macAddress.GetString(fromMac), nn::wlan::MacAddress::CreateBroadcastMacAddress().GetString(toMac), seqNumber);
#endif
                result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), putSize);
                if (result.IsSuccess() != true)
                {
                    if (pThis->m_sendResult.GetInnerValueForDebug() != result.GetInnerValueForDebug())
                    {
                        if (pThis->m_sendResult.IsSuccess() != true)
                        {
                            NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Error Count(%lu/%lu) Line %d\n",
                                pThis->m_sendResult.GetInnerValueForDebug(), pThis->m_sendResult.GetModule(), pThis->m_sendResult.GetDescription(), errorCount, sendCnt, __LINE__);
                            errorCount = 0;
                        }
                        NN_LOG("             LocalApi PutFrame Error(0x%08x:module:%d destription:%d) Line %d\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription(), __LINE__);
                        pThis->m_sendResult = result;
                    }
                    errorCount++;
                }
                else
                {
                    if ((cnt + 1) == pThis->m_etherList.typeCount)
                    {
                        pThis->m_seqNumber++;
                    }
                    sendCnt++;
                    pThis->m_connectsInfo.totalSendSize += putSize;

                    if (sendInterval > 0)
                    {
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sendInterval));
                    }
                }
            }
        }
    }

    void LocalApiClass::LocalSendActionFrameThread(void* arg) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(arg);
        LocalApiClass* pThis = static_cast<LocalApiClass*>(arg);
        nn::wlan::ConnectionStatus conStatus;
        size_t putSize = pThis->m_putSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        int64_t sendInterval = pThis->m_sendInterval;
        int64_t sendTime = pThis->m_sendTime;
        nn::os::Tick startTick;
        nn::Result result;
        bool isSuccess;

        NN_ABORT_UNLESS_NOT_NULL(putBuffer);
        std::memset(putBuffer.get(), 0x00, putSize);

        result = nn::wlan::Local::GetConnectionStatus(&conStatus);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             LocalApi GetConnectionStatus Error(0x%08x) Line(%d)\n", result.GetInnerValueForDebug(), __LINE__);
            return;
        }

        if (pThis->m_connectsInfo.isSendStart != true)
        {
            pThis->m_connectsInfo.isSendStart = false;
            pThis->m_connectsInfo.totalSendSize = 0;
        }

#ifdef WLAN_TEST_STATE_ASSER_STOP
        LocalOpenModeType openMode = pThis->GetLocalOpenMode();
#else
        LocalOpenModeType openMode = pThis->GetLocalModeType();
#endif

        pThis->LocalClearSeqNumber();
        startTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds() < sendTime)
        {
            if (pThis->m_isSendActionFrame != true)
            {
                break;
            }

            if (pThis->m_connectsInfo.isSendStart != true)
            {
                pThis->m_connectsInfo.isSendStart = true;
                pThis->m_connectsInfo.startSendTick = nn::os::GetSystemTick();
            }

            if (openMode == OpenMode_Master)
            {
                // CLIENT送信
                isSuccess = pThis->LocalClientSendActionFrame(putBuffer.get(), putSize, pThis->m_channel, pThis->m_frameType);
            }
            else
            {
                // MASTER送信
                isSuccess = pThis->LocalMasterSendActionFrame(putBuffer.get(), putSize, conStatus.channel, pThis->m_frameType);
            }

            if (isSuccess == true)
            {
                pThis->m_connectsInfo.totalSendSize += putSize;
                if (sendInterval > 0)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sendInterval));
                }
            }
            else
            {
                NN_LOG("             LocalApi Thread PutActionFrameOneShot Error\n");
            }
        }

    }

    void LocalApiClass::LocalReceiveFrameThread(void* arg) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(arg);
        nn::wlan::MacAddress myMacAddress, toMacAddress;
        LocalApiClass* pThis = static_cast<LocalApiClass*>(arg);
        LocalPoolMngDetail *pPoolMng = nullptr;
        LocalOpenModeType openMode;
        nn::Result result;

        nn::wlan::Local::GetMacAddress(&myMacAddress);

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
        char fromMac[nn::wlan::MacAddress::MacStringSize];
        char toMac[nn::wlan::MacAddress::MacStringSize];
#endif

#ifdef WLAN_TEST_STATE_ASSER_STOP
        openMode = pThis->GetLocalOpenMode();
#else
        openMode = pThis->GetLocalModeType();
#endif
        // MASTER準備
        if (openMode == LocalOpenModeType::OpenMode_Master)
        {
            pThis->m_connectCount = 0;

            // 指定台数接続するまで待機
            while (pThis->m_connectCount < pThis->m_waitConnectCount)
            {
                pThis->m_connectCount = pThis->LocalMasterConnectWait(pThis->m_clientMacArray, WlanTestClientWaitTimeOut, pThis->m_waitConnectCount);

                if (pThis->m_isReceiveFrame != true)
                {
                    break;
                }
            }

            // 受信開始フレーム送信
            if (pThis->m_isReceiveFrame == true)
            {
                char clientMacArray[nn::wlan::MacAddress::MacStringSize];
                NN_LOG("             LocalApi Client Connected Client Count : %d\n\n", pThis->m_connectCount);
                NN_LOG("             *** START FRAME PACKET LIST ***\n");
                for (int32_t i = 0; i < pThis->m_connectCount; i++)
                {
                    pThis->m_clientMacArray[i].GetString(clientMacArray);
                    NN_LOG("             LocalApi Client Start Trigger PutFrame MacAddress : %s\n", clientMacArray);
                }
                NN_LOG("             *** START FRAME PACKET LIST ***\n\n");

                toMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
                if (pThis->LocalStartTriggerPutFrame(pThis->m_startPutBuffer.get(), pThis->m_startPutSize, toMacAddress, myMacAddress) != true)
                {
                    NN_LOG("             LocalApi LocalStartTriggerPutFrame Error\n");
                }
            }
            nn::os::SignalEvent(&pThis->m_receiveFrameEvent);
        }

        pPoolMng = pThis->GetFirstWritePool(pThis->m_poolFrameMng, GetLocalFrameType::FrameType_Frame);

        pThis->m_connectsInfo.totalReceiveSize = 0;
        pThis->m_connectsInfo.isReceiveStart = false;

        while (pThis->m_isReceiveFrame == true)
        {
            if (pThis->m_connectsInfo.isReceiveStart != true)
            {
                pThis->m_connectsInfo.startRecvTick = nn::os::GetSystemTick();
                pThis->m_connectsInfo.isReceiveStart = true;
            }

            result = nn::wlan::Local::GetFrameRaw(pPoolMng->pData, pPoolMng->bufferSize, &pPoolMng->dataSize, pThis->m_entRxId);
            if (result.IsSuccess() != true)
            {
                if (nn::wlan::ResultGetFrameCancelled::Includes(result) != true)
                {
                    NN_LOG("             LocalApi GetFrame Error(0x%08x:module:%d destription:%d) Entry ID : %lu\n",
                            result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription(), pThis->m_entRxId);
                }
                else
                {
                    NN_LOG("             LocalApi GetFrame Cancel Stop RxID : %d\n", pThis->m_entRxId);
                }
                continue;
            }

            if (pPoolMng->dataSize <= 0)
            {
                NN_LOG("             LocalApi GetFrame Zero Size Receive Break\n");
                continue;
            }

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
            if (pPoolMng->frameType == FrameType_Frame)
            {
                LocalFrameHeader* pHeader = reinterpret_cast<LocalFrameHeader*>(pPoolMng->pData);
                LocalPayloadData* pPayload = reinterpret_cast<LocalPayloadData*>(pHeader->data);
                if (nn::wlan::MacAddress(pHeader->toMacAddress) != myMacAddress)
                {
                    NN_LOG("             LocalApi MAC Address Destination Invalid.\n");
                }
                NN_LOG("             LocalApi GetFrameRaw Data From MacAddress[%s] To MacAddress[%s] Seq : %lu\n",
                    nn::wlan::MacAddress(pHeader->fromMacAddress).GetString(fromMac),
                    nn::wlan::MacAddress(pHeader->toMacAddress).GetString(toMac),
                    pPayload->seq);
            }
#endif

            pThis->m_connectsInfo.totalReceiveSize += pPoolMng->dataSize;

#ifdef WLAN_TEST_RECV_TRACE
            LocalFrameHeader *pFrameHeader = reinterpret_cast<LocalFrameHeader*>(pPoolMng->pData);

            if (nn::wlan::MacAddress(pFrameHeader->fromMacAddress) == nn::wlan::MacAddress(TestTraceMasterMac))
            {
                pThis->m_masterToClientCount++;
            }
            else
            {
                pThis->m_clientToClientCount++;
            }
#endif
            pPoolMng = pThis->GetNextWritePool(pThis->m_poolFrameMng, GetLocalFrameType::FrameType_Frame);
            nn::os::SignalEvent(&pThis->m_parseFrameEvent);
        }

        nn::os::SignalEvent(&pThis->m_parseFrameEvent);

        // MASTER後処理
        if (openMode == LocalOpenModeType::OpenMode_Master && pThis->GetStateClientCount(nn::wlan::ConnectionState_Connected) > 0)
        {
            // 各クライアントのフレームが全て取得できたらMASTERから切断
            nn::wlan::DisconnectClient disClient;
            disClient.clientMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
            disClient.reasonCode = nn::wlan::Dot11ReasonCode::Dot11ReasonCode_UnspecifiedFailure;
            // 全クライアント切断
            result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &disClient);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             LocalApi Master All Client Disconnect Error(0x%08x)\n", result.GetInnerValueForDebug());
            }
            else
            {
                NN_LOG("             LocalApi Master All Client Disconnected\n");
            }
        }

        pThis->m_isReceiveFrame = false;
        nn::os::SignalEvent(&pThis->m_receiveFrameEvent);
    }// NOLINT(impl/function_size) WLANフレーム調査ログ削除でチェックは通ります

    void LocalApiClass::LocalReceiveActionFrameThread(void* arg) NN_NOEXCEPT
    {
        nn::wlan::MacAddress myMacAddress, outMacAddress;
        LocalApiClass* pThis = static_cast<LocalApiClass*>(arg);
        LocalPoolMngDetail *pPoolMng = nullptr;
        nn::Result result;

        pPoolMng = pThis->GetFirstWritePool(pThis->m_poolFrameMng, GetLocalFrameType::FrameType_ActionFrame);
        while (pThis->m_isReceiveFrame == true)
        {
            result = nn::wlan::Local::GetActionFrame(&outMacAddress, pPoolMng->pData, pPoolMng->bufferSize, &pPoolMng->dataSize, pThis->m_actionRxId);
            if (result.IsSuccess() != true)
            {
                if (nn::wlan::ResultGetFrameCancelled::Includes(result) != true)
                {
                    NN_LOG("             LocalApi GetActionFrame Error(0x%08x)\n", result.GetInnerValueForDebug());
                }
                else
                {
                    NN_LOG("             LocalApi GetActionFrame Cancel Stop(Stop:%d)\n", pThis->m_isReceiveFrame);
                }
                continue;
            }

            pPoolMng = pThis->GetNextWritePool(pThis->m_poolFrameMng, GetLocalFrameType::FrameType_ActionFrame);
            nn::os::SignalEvent(&pThis->m_parseFrameEvent);
        }

        pThis->m_isReceiveFrame = false;
        nn::os::SignalEvent(&pThis->m_receiveFrameEvent);
    }

    void LocalApiClass::LocalParseFrameThread(void* arg) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(arg);

        LocalApiClass* pThis = static_cast<LocalApiClass*>(arg);
        uint32_t parseRange = 0;
        uint32_t parsePos = 0;
        uint32_t writePos = 0;
        bool isParseResult = false;

        while (true)
        {
            nn::os::WaitEvent(&pThis->m_parseFrameEvent);
            nn::os::ClearEvent(&pThis->m_parseFrameEvent);
            switch (pThis->m_parseType)
            {
            case LocalParseType::ParseType_Parse:
                break;
            case LocalParseType::ParseType_End:
                NN_LOG("             LocalApi ParseFrameThread Stop Event Sygnal\n");
                return;
            default:
                NN_LOG("             LocalApi Parse Frame Type not found(%d)\n", pThis->m_parseType);
                continue;
            }

            writePos = pThis->m_poolFrameMng.writeIdx;
            parsePos = pThis->m_poolFrameMng.readIdx;

            // バッファプール範囲条件：書き込み位置、読み込み位置
            if (pThis->m_poolFrameMng.readIdx > writePos)
            {
                parseRange = pThis->m_poolFrameMng.maxCount - parsePos + writePos;
            }
            else
            {
                parseRange = writePos - parsePos;
            }

            if (writePos == parsePos)
            {
                NN_LOG("             LocalApi Buffer Pool Over Run(%d)\n", parseRange);
            }

            for (int32_t i = 0; i < parseRange; i++)
            {
                if (pThis->m_poolFrameMng.info[parsePos].frameType == GetLocalFrameType::FrameType_Frame)
                {
                    // Frameパケット解析取得
                    isParseResult = pThis->LocalParseFrame(pThis->m_connectsInfo, pThis->m_poolFrameMng.info[parsePos].pData, pThis->m_poolFrameMng.info[parsePos].dataSize);
                }
                else
                {
                    // ActionFrameパケット解析取得
                    isParseResult = pThis->LocalParseActionFrame(pThis->m_connectsInfo, pThis->m_poolFrameMng.info[parsePos].pData, pThis->m_poolFrameMng.info[parsePos].dataSize);
                }

                if (isParseResult != true)
                {
                    NN_LOG("             LocalApi Frame Parse Error\n");
                }

                nn::os::LockMutex(&pThis->m_localApiLock);
                pThis->m_poolFrameMng.info[parsePos].dataSize = 0;
                nn::os::UnlockMutex(&pThis->m_localApiLock);

                ++pThis->m_poolFrameMng.readIdx %= pThis->m_poolFrameMng.maxCount;
                parsePos = pThis->m_poolFrameMng.readIdx;
                if (parsePos == writePos)
                {
                    break;
                }
            }

            // 指定のCLIENT台数が接続されているか判定
            if (pThis->m_connectsInfo.connectCount < pThis->m_endConnectCount)
            {
                continue;
            }

            // 全CLIENTでフレームが指定パケット以上受けているか判定
            uint32_t endPacketCount = 0;
            for (int32_t i = 0; i < pThis->m_connectsInfo.connectCount; i++)
            {
                if (pThis->m_connectsInfo.facing[i].getFrameCount >= pThis->m_endPacketCount)
                {
                    endPacketCount++;
                }
            }

            // 全CLIENTで指定のフレーム数を受信済みかの判定（全台数で且つ指定MAXパケットに到達した場合にログ出力）
            if (endPacketCount >= pThis->m_connectsInfo.connectCount)
            {
                NN_LOG("             LocalApi Max Packet(%lu)\n", pThis->m_endPacketCount);
            }
        }
    }

    void LocalApiClass::TcpMasterThread(void* arg) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(arg);

        SocketInfo* pTcpInfo = static_cast<SocketInfo*>(arg);
        SocketClientInfo *pClientInfo = &pTcpInfo->clients[pTcpInfo->connectCount];
        nn::socket::SockLenT len = sizeof(nn::socket::SockAddrIn);
        nn::socket::SockAddrIn addr;
        int32_t ret;

        std::memset(&addr, 0x00, sizeof(addr));
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(pTcpInfo->masterIp.port);
        addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);

        pTcpInfo->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        if (0 > pTcpInfo->socket)
        {
            NN_LOG("             LocalApi Socekt Error Line(%lu)\n", __LINE__);
            return;
        }

        ret = nn::socket::Bind(pTcpInfo->socket, reinterpret_cast<nn::socket::SockAddr *>(&addr), sizeof(addr));

        // ソケットを受信待機モードにし，保留接続キューのサイズを確保
        ret = nn::socket::Listen(pTcpInfo->socket, 5);
        if (0 > ret)
        {
            NN_LOG("             LocalApi Socekt Listen Error Line(%lu)\n", __LINE__);
            return;
        }

        while (pTcpInfo->isThreadStop != true)
        {
            if (pTcpInfo->connectCount >= pTcpInfo->waitCount)
            {
                NN_LOG("             LocalApi Max Connected(%lu)\n", pTcpInfo->connectCount);
                break;
            }

            // サーバーソケットを接続要求待ち状態
            len = sizeof(pClientInfo->clientAddr);
            pClientInfo->accept = nn::socket::Accept(pTcpInfo->socket, reinterpret_cast<nn::socket::SockAddr *>(&pClientInfo->clientAddr), &len);
            if (pClientInfo->accept > -1)
            {
                pTcpInfo->connectCount++;
                pClientInfo++;
            }
        }

        if (pTcpInfo->connectCount >= pTcpInfo->waitCount &&
            pTcpInfo->pConnectEvent != nullptr)
        {
            nn::os::SignalEvent(pTcpInfo->pConnectEvent);
        }
    }

    void LocalApiClass::TcpClientThread(void* arg) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(arg);

        SocketInfo*             pTcpInfo = static_cast<SocketInfo*>(arg);
        nn::socket::SockAddrIn  addr;
        nn::socket::Errno       error = nn::socket::Errno::ESuccess,
                                oldError = nn::socket::Errno::ESuccess;

        //std::memset(&addr, 0x00, sizeof(addr));
        addr.sin_family = nn::socket::Family::Pf_Inet;
        addr.sin_port = nn::socket::InetHtons(pTcpInfo->masterIp.port);
        addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);

        while (pTcpInfo->isThreadStop != true)
        {
            // 接続先IP
            addr.sin_family = nn::socket::Family::Pf_Inet;
            nn::socket::InetAton(pTcpInfo->masterIp.strIpAddress, &addr.sin_addr);
            addr.sin_port = nn::socket::InetHtons(pTcpInfo->masterIp.port);

            pTcpInfo->socket = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
            if (0 > pTcpInfo->socket)
            {
                return;
            }

            if (nn::socket::Connect(pTcpInfo->socket, (nn::socket::SockAddr *)&addr, sizeof(addr)) == 0)
            {
                NN_LOG("             Tcp Connected\n\n");
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            error = nn::socket::GetLastError();
            if (error != oldError)
            {
                NN_LOG("             Connect error : %d\n", (oldError = error));
            }
            nn::socket::Close(pTcpInfo->socket);
            pTcpInfo->socket = -1;
        }

        if (nn::socket::Recv(pTcpInfo->socket, pTcpInfo->pTcpBuffer, pTcpInfo->tcpSize, nn::socket::MsgFlag::Msg_None) > 0)
        {
            nn::os::SignalEvent(pTcpInfo->pConnectEvent);
        }
    }
} // namespace
