﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @brief   nn::util の文字エンコード API のテストです。
 * @details 以下の関数のテストをします。
 *          - ConvertStringUtf8ToUtf16Native (srcLength あり/なし)
 *          - ConvertStringUtf16NativeToUtf8 (srcLength あり/なし)
 *          - GetLengthOfConvertedStringUtf8ToUtf16Native
 *          - GetLengthOfConvertedStringUtf16NativeToUtf8
 *          - ConvertStringUtf8ToUtf32
 *          - ConvertStringUtf32ToUtf8
 *          - GetLengthOfConvertedStringUtf8ToUtf32
 *          - GetLengthOfConvertedStringUtf32ToUtf8
 *          - ConvertCharacterUtf8ToUtf16Native
 *          - ConvertCharacterUtf16NativeToUtf8
 *          - ConvertCharacterUtf8ToUtf32
 *          - ConvertCharacterUtf32ToUtf8
 *          - PickOutCharacterFromUtf8String
 */

#include <cstdlib>
#include <random>
#include <nnt/nntest.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nnt/nnt_Compiler.h>

/* ----------------------------------------------------------------------------- *
 * (参考)文字エンコード表
 *
 * 文字 | UTF-8  | UTF-16 | UTF-32 | SJIS |  EUC
 *   A  |     41 |    41  |    41  |   41 |   41
 *   B  |     42 |    42  |    42  |   42 |   42
 *   C  |     43 |    43  |    43  |   43 |   43
 *  あ  | E38182 |  3042  |  3042  | 82A0 | A4A2
 *  い  | E38184 |  3044  |  3044  | 82A2 | A4A4
 *  う  | E38186 |  3046  |  3046  | 82A4 | A4A6
 *  え  | E38188 |  3048  |  3048  | 82A6 | A4A8
 *  お  | E3818A |  304A  |  304A  | 82A8 | A4AA
 *
 * (参考)サロゲートペア文字列
 *
 * 文字 |  UTF-8  | 上位サロゲート | 下位サロゲート |  UTF-32  |
 * 𠀋   | F0A0808B|      D840      |      DC0B      | 0002000B |
 * 𣑋   | F0A3918B|      D84D      |      DC4B      | 0002344B |
 * 𥱤   | F0A5B1A4|      D857      |      DC64      | 00025C64 |
 * ----------------------------------------------------------------------------- */

namespace {

// パターン1: ひらがな "あいうえお"
const int g_Utf8Length1 = 16;
const char g_Utf8Str1[g_Utf8Length1] = {
    '\xE3', '\x81', '\x82', '\xE3', '\x81', '\x84', '\xE3', '\x81', '\x86',
    '\xE3', '\x81', '\x88', '\xE3', '\x81', '\x8A', '\0'
};

// パターン 2 : 英字 "ABCDE"
const int g_Utf8Length2 = 6;
const char g_Utf8Str2[g_Utf8Length2] = {
    '\x41', '\x42', '\x43', '\x44', '\x45', '\0'
};

// パターン 3 : サロゲートペア "𠀋 𣑋 𥱤"
const int g_Utf8Length3 = 13;
const char g_Utf8Str3[g_Utf8Length3] = {
    '\xF0', '\xA0', '\x80', '\x8B',
    '\xF0', '\xA3', '\x91', '\x8B',
    '\xF0', '\xA5', '\xB1', '\xA4', '\0'
};

// ---------------
// UTF-8 -> UTF-16
// ---------------

// テストフィクスチャ
template <typename T>
class EncSmokeUtf16Test : public ::testing::Test
{
protected:

    // テストで利用する文字列
    // パターン 1 : ひらがな
    const int m_Utf16Length1;
    T* m_Utf16Str1;
    // パターン 2 : 英数字
    const int m_Utf16Length2;
    T* m_Utf16Str2;
    // パターン 3 : サロゲートペア
    const int m_Utf16Length3;
    T* m_Utf16Str3;
protected:

    EncSmokeUtf16Test() : m_Utf16Length1(6), m_Utf16Length2(6), m_Utf16Length3(7)
    {
        // テストで利用する文字列を準備
        m_Utf16Str1 = reinterpret_cast<T*>(std::malloc(sizeof(T) * m_Utf16Length1));
        m_Utf16Str2 = reinterpret_cast<T*>(std::malloc(sizeof(T) * m_Utf16Length2));
        m_Utf16Str3 = reinterpret_cast<T*>(std::malloc(sizeof(T) * m_Utf16Length3));

        m_Utf16Str1[0] = 0x3042;    // あ
        m_Utf16Str1[1] = 0x3044;    // い
        m_Utf16Str1[2] = 0x3046;    // う
        m_Utf16Str1[3] = 0x3048;    // え
        m_Utf16Str1[4] = 0x304A;    // お
        m_Utf16Str1[5] = '\0';

        m_Utf16Str2[0] = 0x0041;    // A
        m_Utf16Str2[1] = 0x0042;    // B
        m_Utf16Str2[2] = 0x0043;    // C
        m_Utf16Str2[3] = 0x0044;    // D
        m_Utf16Str2[4] = 0x0045;    // E
        m_Utf16Str2[5] = '\0';

        m_Utf16Str3[0] = 0xD840; m_Utf16Str3[1] = 0xDC0B;   // 𠀋
        m_Utf16Str3[2] = 0xD84D; m_Utf16Str3[3] = 0xDC4B;   // 𣑋
        m_Utf16Str3[4] = 0xD857; m_Utf16Str3[5] = 0xDC64;   // 𥱤
        m_Utf16Str3[6] = '\0';
    }

    ~EncSmokeUtf16Test()
    {
        std::free(reinterpret_cast<void*>(m_Utf16Str1));
        std::free(reinterpret_cast<void*>(m_Utf16Str2));
        std::free(reinterpret_cast<void*>(m_Utf16Str3));
    }

    /**
     * @brief   テスト開始時に呼び出される関数
     */
    virtual void SetUp()
    {
    }

    /**
     * @brief   テスト終了時に呼び出される関数
     */
    virtual void TearDown()
    {
    }

    /**
     * @brief   文字列が一致しているかどうか確認します。
     */
    template <typename CharType>
    void CheckCorrespondence(const CharType* pStr1, const CharType* pStr2, int length)
    {
        for(int i = 0; i < length; ++i)
        {
            EXPECT_EQ(pStr1[i], pStr2[i]);
        }
    }

    /**
     * @brief   UTF-8 と UTF-16 の文字を 1 文字ずつ格納するための構造体
     */
    struct CharacterSet
    {
        char pUtf8[4];      // UTF-8 の文字
        int utf8Length;     // UTF-8 の文字のバイト数
        T pUtf16[2]; // UTF-16 の文字
        int utf16Length;    // UTF-16 の文字のバイト数
    };

NNT_DISABLE_OPTIMIZATION
    /**
     * @brief       指定された長さの配列を文字列で埋めます。
     *
     * @param[out]  pUtf8Str        UTF-8 文字列で埋まるバッファ
     * @param[out]  pUtf16Str       UTF-16 文字列で埋まるバッファ
     * @param[out]  outUtf8Length   pUtf8Str に入れられた値の長さ
     * @param[out]  outUtf16Length  pUtf16Str に入れられた値の長さ
     * @param[in]   length          埋める文字数
     * @param[in]   utf8Length      pUtf8Str のバッファサイズ(長さ)
     * @param[in]   utf16Length     pUtf16Str のバッファサイズ(長さ)
     *
     * @return      正常に埋められたら true が、エラーの場合は false が返ります。
     *
     * @details     pUtf8Str と pUtf16Str には文字コード違いの同じ文字列が入ります。
     */
    bool MakeRandomString(char* pUtf8Str, T* pUtf16Str, int* outUtf8Length, int* outUtf16Length, int length, int utf8Length, int utf16Length)
    {
        const int seed = 1234567;   // 初期シード
        std::mt19937 engine(seed);
        std::uniform_int_distribution<int> distribution(0, 2);

        CharacterSet pattern[] = {
           // 「A」
           { { '\x41', '\x00', '\x00', '\x00'},
             1,
             {0x0041, 0x0000},
             1
           },
           // 「あ」
           { { '\xE3', '\x81', '\x82', '\x00'},
             3,
             {0x3042, 0x0000},
             1
           },
           // 「𠀋」
           { { '\xF0', '\xA0', '\x80', '\x8B'},
             4,
             {0xD840, 0xDC0B},
             2
           }
        };

        int utf8Index = 0;
        int utf16Index = 0;
        *outUtf8Length = 0;
        *outUtf16Length = 0;

        // ランダムに選択した文字を pUtf8Str と pUtf16Str に入れていく
        for(int i = 0; i < length - 1; ++i)
        {
            int index = distribution(engine);
            for(int j = 0; j < pattern[index].utf8Length; ++j)
            {
                pUtf8Str[utf8Index] = pattern[index].pUtf8[j];
                ++utf8Index;
                if(utf8Index >= utf8Length)
                {
                    return false;
                }
            }
            *outUtf8Length += pattern[index].utf8Length;
            for(int j = 0; j < pattern[index].utf16Length; ++j)
            {
                pUtf16Str[utf16Index] = pattern[index].pUtf16[j];
                ++utf16Index;
                if(utf16Index >= utf16Length)
                {
                    return false;
                }
            }
            *outUtf16Length += pattern[index].utf16Length;
        }

        // 終端文字の挿入
        pUtf8Str[utf8Index] = '\0';
        *outUtf8Length += 1;
        pUtf16Str[utf16Index] = '\0';
        *outUtf16Length += 1;
        return true;
    }
NNT_RESTORE_OPTIMIZATION
};

// 型付けテストで指定する型のリスト
typedef ::testing::Types<uint16_t, char16_t> Utf16Types;
TYPED_TEST_CASE(EncSmokeUtf16Test, Utf16Types);

}   // unnamed namespace

// UTF-8 の文字列を UTF-16 に変換
TYPED_TEST(EncSmokeUtf16Test, Utf8ToUtf16Simple)
{
    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&dstLength,
                                                                        g_Utf8Str1,
                                                                        g_Utf8Length1);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);

    ASSERT_EQ(dstLength, this->m_Utf16Length1);
    TypeParam* pDst = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * dstLength));

    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst, dstLength, g_Utf8Str1, g_Utf8Length1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, this->m_Utf16Str1, dstLength - 1);   // 終端文字は付加されない

    // srcLength なしの変換もチェック
    memset(pDst, 0xFF, sizeof(TypeParam) * dstLength);       // 0 以外の適当な数で埋める
    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst, dstLength, g_Utf8Str1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, this->m_Utf16Str1, dstLength);      // 終端文字が付加される

    std::free(reinterpret_cast<void*>(pDst));
}

// 終端文字が文中にある場合の挙動
TYPED_TEST(EncSmokeUtf16Test, Utf8ToUtf16TerminateValue)
{
    // 入力(UTF-8)
    const int srcLength = 18;
    char pSrc[srcLength] = {'\xE3', '\x81', '\x82',    // あ
                            '\xE3', '\x81', '\x84',    // い
                            '\xE3', '\x81', '\x86',    // う
                            '\0',
                            '\xE3', '\x81', '\x88',    // え
                            '\xE3', '\x81', '\x8A',    // お
                            '\0',
                            '\0'};
    const int correctLength = 8;
    TypeParam pCorrect[correctLength] = {0x3042,
                                        0x3044,
                                        0x3046,
                                        '\0',
                                        0x3048,
                                        0x304A,
                                        '\0',
                                        '\0'};

    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&dstLength,
                                                                        pSrc,
                                                                        srcLength - 1);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(dstLength, 7);

    result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&dstLength,
                                                                   pSrc,
                                                                   srcLength);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(dstLength, 8);

    TypeParam* pDst = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * dstLength));
    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst, dstLength, pSrc, srcLength);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, pCorrect, dstLength);

    // srcLength なしの変換もチェック
    memset(pDst, 0xFF, sizeof(TypeParam) * dstLength);
    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst, dstLength, pSrc);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, pCorrect, 4);
    // srcLength を指定しなかった場合は最初の \0 で変換が終了しているはず
    for (int i = 5; i < correctLength; ++i)
    {
        EXPECT_NE(pDst[i], pCorrect[i]);
    }

    std::free(reinterpret_cast<void*>(pDst));
}

// サロゲートペア文字列の変換
TYPED_TEST(EncSmokeUtf16Test, Utf8ToUtf16Surrogate)
{
    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&dstLength,
                                                                        g_Utf8Str3,
                                                                        g_Utf8Length3);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);

    TypeParam* pDst = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * dstLength));
    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst, dstLength, g_Utf8Str3, g_Utf8Length3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, this->m_Utf16Str3, dstLength - 1);

    memset(pDst, 0xFA, sizeof(TypeParam) * dstLength);
    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst, dstLength, g_Utf8Str3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, this->m_Utf16Str3, dstLength);

    std::free(reinterpret_cast<void*>(pDst));

    // pSrc に正しい値が入っていても、 srcLength で指定した長さが小さい場合は失敗する
    char pStr[5] = { '\xF0', '\xA0', '\x80', '\x8B', 0 };
    TypeParam pDst2[3] = { 0 };
    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst2, 1, pStr, 3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);
    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst2, 2, pStr, 4);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(pDst2[0], 0xD840);
    EXPECT_EQ(pDst2[1], 0xDC0B);
    // srcLength をとらない変換は、終端文字を付加するため dstLength = 2 では足りない
    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst2, 2, pStr);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InsufficientLength);
    result = nn::util::ConvertStringUtf8ToUtf16Native(pDst2, 3, pStr);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(pDst2[0], 0xD840);
    EXPECT_EQ(pDst2[1], 0xDC0B);
}

// 1 文字変換
TYPED_TEST(EncSmokeUtf16Test, Utf8ToUtf16Chacacter)
{
    char pSrc[4] = { 0 };
    TypeParam pDst[2];

    pSrc[0] = g_Utf8Str1[0];
    pSrc[1] = g_Utf8Str1[1];
    pSrc[2] = g_Utf8Str1[2];

    auto result = nn::util::ConvertCharacterUtf8ToUtf16Native(pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(pDst[0], this->m_Utf16Str1[0]);

    // 文字の後にノイズが入っている場合
    pSrc[3] = g_Utf8Str1[0];
    result = nn::util::ConvertCharacterUtf8ToUtf16Native(pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);

    // 不正文字の場合
    pSrc[0] = 0xFFu;
    result = nn::util::ConvertCharacterUtf8ToUtf16Native(pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);

    // サロゲートペアの場合
    pSrc[0] = g_Utf8Str3[0];
    pSrc[1] = g_Utf8Str3[1];
    pSrc[2] = g_Utf8Str3[2];
    pSrc[3] = g_Utf8Str3[3];
    result = nn::util::ConvertCharacterUtf8ToUtf16Native(pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, this->m_Utf16Str3, 2);

    // pSrc に 2 文字以上文字が入ってしまっている場合
    pSrc[0] = g_Utf8Str2[0];
    pSrc[1] = g_Utf8Str2[1];
    result = nn::util::ConvertCharacterUtf8ToUtf16Native(pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(pDst[0], this->m_Utf16Str2[0]);
    EXPECT_NE(pDst[1], this->m_Utf16Str2[1]);     // 2 文字目は変換されない
}

// ---------------
// UTF-16 -> UTF-8
// ---------------

// UTF-16 の文字列を UTF-8 に変換
TYPED_TEST(EncSmokeUtf16Test, Utf16ToUtf8Simple)
{
    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&dstLength,
                                                                        this->m_Utf16Str1,
                                                                        this->m_Utf16Length1);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    char* pDst = reinterpret_cast<char*>(std::malloc(sizeof(char) * dstLength));

    result = nn::util::ConvertStringUtf16NativeToUtf8(pDst, dstLength, this->m_Utf16Str1, this->m_Utf16Length1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, const_cast<char*>(g_Utf8Str1), dstLength - 1);

    memset(pDst, 0xFF, dstLength);
    result = nn::util::ConvertStringUtf16NativeToUtf8(pDst, dstLength, this->m_Utf16Str1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, const_cast<char*>(g_Utf8Str1), dstLength);

    std::free(reinterpret_cast<void*>(pDst));
}

// 終端文字が文中にある場合の挙動
TYPED_TEST(EncSmokeUtf16Test, Utf16ToUtf8TerminateValue)
{
    // 入力(UTF-16)
    const int srcLength = 8;
    TypeParam pSrc[srcLength] = {0x3042,    // あ -> UTF-8 だと 3 バイト
                                0x3044,    // い
                                0x3046,    // う
                                '\0',
                                0x3048,    // え
                                0x304A,    // お
                                '\0',
                                '\0'};
    const int correctLength = 18;
    char pCorrect[correctLength] = { '\xE3', '\x81', '\x82',    // あ
                                     '\xE3', '\x81', '\x84',    // い
                                     '\xE3', '\x81', '\x86',    // う
                                     '\0',
                                     '\xE3', '\x81', '\x88',    // え
                                     '\xE3', '\x81', '\x8A',    // お
                                     '\0',
                                     '\0' };

    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&dstLength,
                                                                        pSrc,
                                                                        srcLength - 1);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(dstLength, 17);
    result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&dstLength,
                                                                        pSrc,
                                                                        srcLength);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(dstLength, 18);

    char* pDst = reinterpret_cast<char*>(std::malloc(sizeof(char) * dstLength));
    result = nn::util::ConvertStringUtf16NativeToUtf8(pDst, dstLength, pSrc, srcLength);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, pCorrect, dstLength);

    memset(pDst, 0xFF, dstLength);
    result = nn::util::ConvertStringUtf16NativeToUtf8(pDst, dstLength, pSrc);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, pCorrect, 10);
    // srcLength を指定しなかった場合は最初の \0 で変換が終了しているはず
    for (int i = 11; i < correctLength; ++i)
    {
        EXPECT_NE(pDst[i], pCorrect[i]);
    }

    std::free(reinterpret_cast<void*>(pDst));
}

// サロゲートペアを渡したときの挙動
TYPED_TEST(EncSmokeUtf16Test, Utf16ToUtf8Surrogate)
{
    int dstLength = 0;

    enum ByteCheck
    {
        ByteCheck_Correspondence,   // 文字列の完全一致チェック
        ByteCheck_Invalid,          // 文字列の完全不一致チェック
        ByteCheck_UpperSurrogate    // 文字列の上位サロゲートのみ変換されているチェック
    };

    auto ExecuteEncoding = [&](int checkFlg , bool isTerminateNull) -> nn::util::CharacterEncodingResult
    {
        char* pDst = reinterpret_cast<char*>(std::calloc(sizeof(char), dstLength));
        nn::util::CharacterEncodingResult result;
        if (isTerminateNull)
        {
            result = nn::util::ConvertStringUtf16NativeToUtf8(pDst,
                                                              dstLength,
                                                              this->m_Utf16Str3,
                                                              this->m_Utf16Length3);
        }
        else
        {
            result = nn::util::ConvertStringUtf16NativeToUtf8(pDst, dstLength, this->m_Utf16Str3);
        }

        switch(checkFlg)
        {
        case ByteCheck_Correspondence:
            this->CheckCorrespondence(pDst, const_cast<char*>(g_Utf8Str3), isTerminateNull ? dstLength : dstLength - 1);
            break;
        case ByteCheck_Invalid:
            EXPECT_NE(g_Utf8Str3[8], pDst[8]);
            EXPECT_NE(g_Utf8Str3[9], pDst[9]);
            EXPECT_NE(g_Utf8Str3[10], pDst[10]);
            EXPECT_NE(g_Utf8Str3[11], pDst[11]);
            break;
        case ByteCheck_UpperSurrogate:
            // 上位サロゲートのみ部分的に変換されている
            EXPECT_EQ(g_Utf8Str3[8], pDst[8]);
            EXPECT_NE(g_Utf8Str3[9], pDst[9]);
            EXPECT_NE(g_Utf8Str3[10], pDst[10]);
            EXPECT_NE(g_Utf8Str3[11], pDst[11]);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
            break;
        }

        std::free(reinterpret_cast<void*>(pDst));

        return result;
    };

    auto result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&dstLength,
                                                                        this->m_Utf16Str3,
                                                                        this->m_Utf16Length3);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);

    // 変換が成功する場合
    EXPECT_EQ(ExecuteEncoding(ByteCheck_Correspondence, true), nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(ExecuteEncoding(ByteCheck_Correspondence, false), nn::util::CharacterEncodingResult_Success);

    // 上位サロゲートで終わる場合
    this->m_Utf16Str3[5] = '\0';
    EXPECT_EQ(ExecuteEncoding(ByteCheck_UpperSurrogate, true), nn::util::CharacterEncodingResult_InvalidFormat);
    EXPECT_EQ(ExecuteEncoding(ByteCheck_UpperSurrogate, false), nn::util::CharacterEncodingResult_InvalidFormat);

    // 下位サロゲートしかない場合
    this->m_Utf16Str3[4] = 0xDC64;
    this->m_Utf16Str3[5] = '\0';
    EXPECT_EQ(ExecuteEncoding(ByteCheck_Invalid, true), nn::util::CharacterEncodingResult_InvalidFormat);
    EXPECT_EQ(ExecuteEncoding(ByteCheck_Invalid, false), nn::util::CharacterEncodingResult_InvalidFormat);

    // 上位サロゲートは正しく、下位サロゲートは不正な場合
    this->m_Utf16Str3[4] = 0xD857;
    this->m_Utf16Str3[5] = 0xD800;
    EXPECT_EQ(ExecuteEncoding(ByteCheck_UpperSurrogate, true), nn::util::CharacterEncodingResult_InvalidFormat);
    EXPECT_EQ(ExecuteEncoding(ByteCheck_UpperSurrogate, false), nn::util::CharacterEncodingResult_InvalidFormat);

    // 下位サロゲート、上位サロゲートの順番で来た場合
    this->m_Utf16Str3[4] = 0xDC64;
    this->m_Utf16Str3[5] = 0xD857;
    EXPECT_EQ(ExecuteEncoding(ByteCheck_Invalid, true), nn::util::CharacterEncodingResult_InvalidFormat);
    EXPECT_EQ(ExecuteEncoding(ByteCheck_Invalid, false), nn::util::CharacterEncodingResult_InvalidFormat);

}

// 1 文字変換
TYPED_TEST(EncSmokeUtf16Test, Utf16ToUtf8Chacacter)
{
    TypeParam pSrc[2] = { 0 };
    char pDst[4];

    pSrc[0] = this->m_Utf16Str1[0];

    auto result = nn::util::ConvertCharacterUtf16NativeToUtf8(pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, const_cast<char*>(g_Utf8Str1), 3);

    // サロゲートペアの場合
    pSrc[0] = this->m_Utf16Str3[0];
    pSrc[1] = this->m_Utf16Str3[1];
    result = nn::util::ConvertCharacterUtf16NativeToUtf8(pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, const_cast<char*>(g_Utf8Str3), 4);

    // 不正文字の場合
    pSrc[1] = this->m_Utf16Str1[0];
    result = nn::util::ConvertCharacterUtf16NativeToUtf8(pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);

    // pSrc に 2 文字以上文字が入ってしまっている場合
    pSrc[0] = this->m_Utf16Str2[0];
    pSrc[1] = this->m_Utf16Str2[1];
    result = nn::util::ConvertCharacterUtf16NativeToUtf8(pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(pDst[0], g_Utf8Str2[0]);
    EXPECT_NE(pDst[1], g_Utf8Str2[1]);     // 2 文字目は変換されない
}

#if !defined(NN_BUILD_CONFIG_OS_WIN) || !defined(NN_BUILD_CONFIG_ADDRESS_64)
// 変換に必要な配列の長さ取得関数のテスト
TYPED_TEST(EncSmokeUtf16Test, GetLength)
{
    // パターン 4 : 1 ～ 3 の混合 + 長い文字列
    int utf8Length4, utf16Length4;
    int strLength = 1024;                       // 欲しい文字列の長さ
    int bufferLength = strLength * 4 + 1;       // 確保する長さ。最大でも * 4 + 終端文字があれば事足りる

    char* utf8Str4 = reinterpret_cast<char*>(std::malloc(sizeof(char) * bufferLength));
    TypeParam* utf16Str4 = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * bufferLength));

    bool strResult = this->MakeRandomString(utf8Str4, utf16Str4, &utf8Length4, &utf16Length4, strLength, bufferLength, bufferLength);
    ASSERT_NE(strResult, false);

    int dstLength = 0;

    // UTF-8変換後に必要な長さの取得チェック
    auto CheckUtf8Length = [&](TypeParam* pSrc, int srcLength, int correct)
    {
        auto result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&dstLength,
            pSrc,
            srcLength);
        EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
        EXPECT_EQ(dstLength, correct);
    };

    // UTF-16変換後に必要な長さの取得チェック
    auto CheckUtf16Length = [&](const char* pSrc, int srcLength, int correct)
    {
        auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&dstLength,
            pSrc,
            srcLength);
        EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
        EXPECT_EQ(dstLength, correct);
    };

    // UTF-16 -> UTF-8 変換時に必要な配列の長さが正しいかチェック
    CheckUtf8Length(this->m_Utf16Str1, this->m_Utf16Length1, g_Utf8Length1);
    CheckUtf8Length(this->m_Utf16Str2, this->m_Utf16Length2, g_Utf8Length2);
    CheckUtf8Length(this->m_Utf16Str3, this->m_Utf16Length3, g_Utf8Length3);
    CheckUtf8Length(utf16Str4, utf16Length4, utf8Length4);

    // UTF-8 -> UTF-16 変換時に必要な配列の長さが正しいかチェック
    CheckUtf16Length(g_Utf8Str1, g_Utf8Length1, this->m_Utf16Length1);
    CheckUtf16Length(g_Utf8Str2, g_Utf8Length2, this->m_Utf16Length2);
    CheckUtf16Length(g_Utf8Str3, g_Utf8Length3, this->m_Utf16Length3);
    CheckUtf16Length(utf8Str4, utf8Length4, utf16Length4);
}
#endif

// 長さの取得関数のテスト2
TYPED_TEST(EncSmokeUtf16Test, GetLength2)
{
    // 配列の長さより短い srcLength を与えた場合、 srcLength で指定した場所までの長さを返す
    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&dstLength, g_Utf8Str2, 3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(dstLength, 3);

    dstLength = 0;
    result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&dstLength, this->m_Utf16Str2, 3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(dstLength, 3);
}

// ---------------
// UTF-8 -> UTF-32
// ---------------

namespace {

// テストフィクスチャ
template <typename T>
class EncSmokeUtf32Test : public ::testing::Test
{
protected:

    // テストで利用する文字列
    // パターン 1 : ひらがな
    const int m_Utf32Length1;
    T* m_Utf32Str1;
    // パターン 2 : 英数字
    const int m_Utf32Length2;
    T* m_Utf32Str2;
    // パターン 3 : サロゲートペア
    const int m_Utf32Length3;
    T* m_Utf32Str3;

protected:

    EncSmokeUtf32Test() : m_Utf32Length1(6), m_Utf32Length2(6), m_Utf32Length3(4)
    {
        // テストで利用する文字列を準備
        m_Utf32Str1 = reinterpret_cast<T*>(std::malloc(sizeof(T) * m_Utf32Length1));
        m_Utf32Str2 = reinterpret_cast<T*>(std::malloc(sizeof(T) * m_Utf32Length2));
        m_Utf32Str3 = reinterpret_cast<T*>(std::malloc(sizeof(T) * m_Utf32Length3));

        m_Utf32Str1[0] = 0x00003042;    // あ
        m_Utf32Str1[1] = 0x00003044;    // い
        m_Utf32Str1[2] = 0x00003046;    // う
        m_Utf32Str1[3] = 0x00003048;    // え
        m_Utf32Str1[4] = 0x0000304A;    // お
        m_Utf32Str1[5] = '\0';

        m_Utf32Str2[0] = 0x00000041;    // A
        m_Utf32Str2[1] = 0x00000042;    // B
        m_Utf32Str2[2] = 0x00000043;    // C
        m_Utf32Str2[3] = 0x00000044;    // D
        m_Utf32Str2[4] = 0x00000045;    // E
        m_Utf32Str2[5] = '\0';

        m_Utf32Str3[0] = 0x0002000B;    // 𠀋
        m_Utf32Str3[1] = 0x0002344B;    // 𣑋
        m_Utf32Str3[2] = 0x00025C64;    // 𥱤
        m_Utf32Str3[3] = '\0';
    }

    ~EncSmokeUtf32Test()
    {
        std::free(reinterpret_cast<void*>(m_Utf32Str1));
        std::free(reinterpret_cast<void*>(m_Utf32Str2));
        std::free(reinterpret_cast<void*>(m_Utf32Str3));
    }

    /**
    * @brief   テスト開始時に呼び出される関数
    */
    virtual void SetUp()
    {
    }

    /**
    * @brief   テスト終了時に呼び出される関数
    */
    virtual void TearDown()
    {
    }

    /**
    * @brief   文字列が一致しているかどうか確認します。
    */
    template <typename CharType>
    void CheckCorrespondence(const CharType* pStr1, const CharType* pStr2, int length)
    {
        for (int i = 0; i < length; ++i)
        {
            EXPECT_EQ(pStr1[i], pStr2[i]);
        }
    }

};

// 型付けテストで指定する型のリスト
typedef ::testing::Types<uint32_t, char32_t> Utf32Types;
TYPED_TEST_CASE(EncSmokeUtf32Test, Utf32Types);

}   // unnamed namespace

// UTF-8 の文字列を UTF-32 に変換
TYPED_TEST(EncSmokeUtf32Test, Utf8ToUtf32Simple)
{
    char utf8str1[g_Utf8Length1];
    memcpy(utf8str1, g_Utf8Str1, g_Utf8Length1);
    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&dstLength,
        utf8str1);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);

    ASSERT_EQ(dstLength, this->m_Utf32Length1 - 1);   // 終端文字分の長さは含まれない
    // 実際の変換では終端文字が付加される
    dstLength += 1;
    TypeParam* pDst = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * dstLength));

    result = nn::util::ConvertStringUtf8ToUtf32(pDst, dstLength, utf8str1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);

    this->CheckCorrespondence(pDst, this->m_Utf32Str1, dstLength);

    // 与える dstLength が足りないとき
    result = nn::util::ConvertStringUtf8ToUtf32(pDst, dstLength - 1, utf8str1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InsufficientLength);

    // 不正文字があるとき
    utf8str1[0] = 0xFFu;
    result = nn::util::ConvertStringUtf8ToUtf32(pDst, dstLength, utf8str1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);

    std::free(reinterpret_cast<void*>(pDst));
}

// サロゲートペア文字列の変換
TYPED_TEST(EncSmokeUtf32Test, Utf8ToUtf32Surrogate)
{
    char utf8str3[g_Utf8Length3];
    memcpy(utf8str3, g_Utf8Str3, g_Utf8Length3);
    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&dstLength,
        utf8str3);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);

    // 成功ケース
    ASSERT_EQ(dstLength, this->m_Utf32Length3 - 1);
    dstLength += 1;
    TypeParam* pDst = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * dstLength));
    result = nn::util::ConvertStringUtf8ToUtf32(pDst, dstLength, utf8str3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);

    this->CheckCorrespondence(pDst, this->m_Utf32Str3, dstLength);

    // dstLength が不足している場合
    result = nn::util::ConvertStringUtf8ToUtf32(pDst, dstLength - 1, utf8str3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InsufficientLength);

    // 不正文字が入り込んだ場合
    utf8str3[0] = 0xFFu;
    result = nn::util::ConvertStringUtf8ToUtf32(pDst, dstLength, utf8str3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);
    // InsufficientLength より InvalidFormat が優先される
    result = nn::util::ConvertStringUtf8ToUtf32(pDst, dstLength - 1, utf8str3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);

    std::free(reinterpret_cast<void*>(pDst));
}

// 冗長表現で InvalidFormat が返ることを確認
TYPED_TEST(EncSmokeUtf32Test, Utf8ToUtf32LengthyExpression)
{
    // ' ' (スペース)
    const char pSpace[] = { '\x20', '\0' };
    TypeParam pSpaceUtf32[] = { 0x00000020, '\0' };
    // ' ' の UTF-8 冗長表現
    const char pLengthySpace[3][5] = { {'\xC0', '\x20',   '\0',   '\0', '\0'},
                                       {'\xE0', '\x80', '\x20',   '\0', '\0'},
                                       {'\xF0', '\x80', '\x80', '\x20', '\0'} };

    // '/'
    const char pSlash[] = { '\x2F', '\0' };
    TypeParam pSlashUtf32[] = { 0x0000002F, '\0' };
    // '/' の UTF-8 冗長表現
    const char pLengthySlash[3][5] = { {'\xC0', '\xAF',   '\0',   '\0', '\0'},
                                       {'\xE0', '\x80', '\xAF',   '\0', '\0'},
                                       {'\xF0', '\x80', '\x80', '\xAF', '\0'} };

    // '~'
    const char pTilde[] = { '\x7E', '\0' };
    TypeParam pTildeUtf32[] = { 0x0000007E, '\0' };
    // '~' の UTF-8 冗長表現
    const char pLengthyTilde[3][5] = { {'\xC1', '\xBE',   '\0',   '\0', '\0'},
                                       {'\xE0', '\x81', '\xBE',   '\0', '\0'},
                                       {'\xF0', '\x80', '\x81', '\xBE', '\0'} };

    // 与えられた冗長表現で InvalidFormat が返ることをテスト
    auto CheckLengthyExpression = [](const char* pCorrect, TypeParam* pCorrectUtf32, const char pLengthy[][5])
    {
        int dstLength = 0;
        TypeParam* pDst = new TypeParam[5];
        nn::util::CharacterEncodingResult result;

        // 成功パターン
        result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&dstLength, pCorrect);
        EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
        EXPECT_EQ(dstLength, 1);

        dstLength++;
        result = nn::util::ConvertStringUtf8ToUtf32(pDst, dstLength, pCorrect);
        EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);
        EXPECT_EQ(pDst[0], pCorrectUtf32[0]);
        EXPECT_EQ(pDst[1], pCorrectUtf32[1]);

        // 失敗パターン
        for (int i = 0; i < 3; ++i)
        {
            // 冗長表現はどのパターンでもエラー
            result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&dstLength, pLengthy[i]);
            EXPECT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);
            result = nn::util::ConvertStringUtf8ToUtf32(pDst, 2, pLengthy[i]);
            EXPECT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);
        }
    };

    CheckLengthyExpression(pSpace, pSpaceUtf32, pLengthySpace);
    CheckLengthyExpression(pSlash, pSlashUtf32, pLengthySlash);
    CheckLengthyExpression(pTilde, pTildeUtf32, pLengthyTilde);
}

// 1 文字変換
TYPED_TEST(EncSmokeUtf32Test, Utf8ToUtf32Chacacter)
{
    char pSrc[4] = { 0 };
    TypeParam pDst;

    pSrc[0] = g_Utf8Str1[0];
    pSrc[1] = g_Utf8Str1[1];
    pSrc[2] = g_Utf8Str1[2];

    auto result = nn::util::ConvertCharacterUtf8ToUtf32(&pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    EXPECT_EQ(pDst, this->m_Utf32Str1[0]);

    // 不正文字の場合
    pSrc[0] = 0xFFu;
    result = nn::util::ConvertCharacterUtf8ToUtf32(&pDst, pSrc);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);
}

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

// nullptr を渡したときの挙動
TEST(EncSmokeUtf32DeathTest, Utf8ToUtf32Nullptr)
{
    const int dstLength = 10;
    uint32_t pDst[dstLength];
    uint32_t* pNullPtr = nullptr;
    char utf8str[4] = { '\xE3', '\x81', '\x82', '\0' };

    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf8ToUtf32(pNullPtr,
                                                                  0,
                                                                  utf8str), "");
    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf8ToUtf32(pDst,
                                                                  dstLength,
                                                                  nullptr), "");
}

// 不正な長さを渡した時の挙動
TEST(EncSmokeUtf32DeathTest, Utf8ToUtf32InvalidLength)
{
    const int dstLength = 10;
    uint32_t pDst[dstLength];
    char utf8str[4] = { '\xE3', '\x81', '\x82', '\0' };

    EXPECT_DEATH_IF_SUPPORTED(nn::util::ConvertStringUtf8ToUtf32(pDst,
                                                                 -1,
                                                                 utf8str), "");
}
#endif

// ---------------
// UTF-32 -> UTF-8
// ---------------

// UTF-32 の文字列を UTF-8 に変換
TYPED_TEST(EncSmokeUtf32Test, Utf32ToUtf8Simple)
{
    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf32ToUtf8(&dstLength,
                                                                  this->m_Utf32Str1);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);

    ASSERT_EQ(dstLength, g_Utf8Length1 - 1);   // 終端文字分の長さは含まれない
    // 実際の変換では終端文字が付加される
    dstLength += 1;
    char* pDst = reinterpret_cast<char*>(std::malloc(sizeof(char) * dstLength));

    result = nn::util::ConvertStringUtf32ToUtf8(pDst, dstLength, this->m_Utf32Str1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);

    this->CheckCorrespondence(pDst, g_Utf8Str1, dstLength);

    // 与える dstLength が足りないとき
    result = nn::util::ConvertStringUtf32ToUtf8(pDst, dstLength - 1, this->m_Utf32Str1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InsufficientLength);

    // 不正文字があるとき
    this->m_Utf32Str1[0] = 0x110000u;
    result = nn::util::ConvertStringUtf32ToUtf8(pDst, dstLength, this->m_Utf32Str1);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);

    std::free(reinterpret_cast<void*>(pDst));
}

// サロゲートペア文字列の変換
TYPED_TEST(EncSmokeUtf32Test, Utf32ToUtf8Surrogate)
{
    int dstLength = 0;
    auto result = nn::util::GetLengthOfConvertedStringUtf32ToUtf8(&dstLength,
                                                                  this->m_Utf32Str3);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);

    ASSERT_EQ(dstLength, g_Utf8Length3 - 1);
    dstLength += 1;
    char* pDst = reinterpret_cast<char*>(std::malloc(sizeof(char) * dstLength));
    result = nn::util::ConvertStringUtf32ToUtf8(pDst, dstLength, this->m_Utf32Str3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_Success);

    this->CheckCorrespondence(pDst, g_Utf8Str3, dstLength);

    result = nn::util::ConvertStringUtf32ToUtf8(pDst, dstLength - 1, this->m_Utf32Str3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InsufficientLength);

    this->m_Utf32Str3[0] = 0x110000u;
    result = nn::util::ConvertStringUtf32ToUtf8(pDst, dstLength, this->m_Utf32Str3);
    EXPECT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);

    std::free(reinterpret_cast<void*>(pDst));
}


// 1 文字変換
TYPED_TEST(EncSmokeUtf32Test, Utf32ToUtf8Chacacter)
{
    TypeParam src;
    char pDst[4];

    src = this->m_Utf32Str1[0];

    auto result = nn::util::ConvertCharacterUtf32ToUtf8(pDst, src);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    this->CheckCorrespondence(pDst, g_Utf8Str1, 3);

    // 不正文字の場合
    src = 0x110000u;
    result = nn::util::ConvertCharacterUtf32ToUtf8(pDst, src);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);
}

// nullptr を渡したときの挙動
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
TEST(EncSmokeUtf32DeathTest, Utf32ToUtf8Nullptr)
{
    const int dstLength = 10;
    char pDst[dstLength];
    uint32_t* pNullPtr = nullptr;
    uint32_t utf32Str[6] = { 0x00003042, 0x00003044, 0x00003046, 0x00003048, 0x0000304A, '\0' };

    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf32ToUtf8(nullptr,
                                                                  0,
                                                                  utf32Str), "");
    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf32ToUtf8(pDst,
                                                                  dstLength,
                                                                  pNullPtr), "");
}

// 不正な長さを渡した時の挙動
TEST(EncSmokeUtf32DeathTest, Utf32ToUtf8InvalidLength)
{
    const int dstLength = 10;
    char pDst[dstLength];
    uint32_t utf32Str[6] = { 0x00003042, 0x00003044, 0x00003046, 0x00003048, 0x0000304A, '\0' };

    EXPECT_DEATH_IF_SUPPORTED(nn::util::ConvertStringUtf32ToUtf8(pDst,
                                                                 -1,
                                                                 utf32Str), "");
}
#endif

// ---------------
// 1 文字切り出し
// ---------------

TYPED_TEST(EncSmokeUtf16Test, PickOut)
{
    const char* ptr = g_Utf8Str1;

    // size:                1 文字に必要なバッファサイズ
    // numberOfCharacters:  元の ptr に入っている文字数
    auto CheckPickingOut = [&](int size, int numberOfCharacters)
    {
        char pChar[4];
        for(int i = 0; i < numberOfCharacters; ++i)
        {
            char pPrevStr[4] = { 0 };
            std::memcpy(pPrevStr, ptr, size);
            auto result = nn::util::PickOutCharacterFromUtf8String(pChar, const_cast<const char **>(&ptr));
            ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
            this->CheckCorrespondence(pChar, pPrevStr, size);
        }
    };

    CheckPickingOut(3, 5);
    ptr = g_Utf8Str2;
    CheckPickingOut(1, 5);
    ptr = g_Utf8Str3;
    CheckPickingOut(4, 3);
}
