﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/os.h>
#include <nn/util/util_Utf8StringUtil.h>
#include <nn/util/util_StringUtil.h>
#include <nn/nn_Log.h>
#include <algorithm>

namespace nn { namespace util {

// ASCII
TEST(Utf8StringUtil, Basic1)
{
    const char message[] = {'A', 'B', 'C', 'D', 'E', 'F', '\0'};

    char buffer[128] = {};

    {
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(GetCodePointCountOfUtf8String(message, sizeof (message)), sizeof (message));

        for (int i = 1; i < sizeof (message); i++)
        {
            EXPECT_TRUE(VerifyUtf8String(message, i));
        }
    }

    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 1), 1u);
    EXPECT_EQ(std::memcmp(buffer, message, 1), 0);
    EXPECT_EQ(buffer[1], 0);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 2), 2u);
    EXPECT_EQ(std::memcmp(buffer, message, 2), 0);
    EXPECT_EQ(buffer[2], 0);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 3), 3u);
    EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
    EXPECT_EQ(buffer[3], 0);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 4), 4u);
    EXPECT_EQ(std::memcmp(buffer, message, 4), 0);
    EXPECT_EQ(buffer[4], 0);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 5), 5u);
    EXPECT_EQ(std::memcmp(buffer, message, 5), 0);
    EXPECT_EQ(buffer[5], 0);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 6), 6u);
    EXPECT_EQ(std::memcmp(buffer, message, 6), 0);
    EXPECT_EQ(buffer[6], 0);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 7), 7u);
    EXPECT_EQ(std::memcmp(buffer, message, 7), 0);
    EXPECT_EQ(buffer[7], 0);
}

// 「あいうえお」
TEST(Utf8StringUtil, Basic2)
{
    const char message[] =
    {
        '\xE3', '\x81', '\x82',
        '\xE3', '\x81', '\x84',
        '\xE3', '\x81', '\x86',
        '\xE3', '\x81', '\x88',
        '\xE3', '\x81', '\x8A',
        '\0'
    };

    char buffer[128] = {};

    {
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(GetCodePointCountOfUtf8String(message, sizeof (message)), 6);

        for (int i = 1; i < sizeof (message); i++)
        {
            // 3 バイト毎に正しい文字列であると判定されなければいけない。
            if (i % 3 == 0)
            {
                EXPECT_TRUE(VerifyUtf8String(message, i));
                EXPECT_EQ(GetCodePointCountOfUtf8String(message, i), i / 3);
            }
            else
            {
                EXPECT_FALSE(VerifyUtf8String(message, i));
                EXPECT_EQ(GetCodePointCountOfUtf8String(message, i), -1);
            }
        }
    }

    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 1), 3u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 2), 6u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 3), 9u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 4), 12u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 5), 15u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 6), 16u);

    // 3 バイト文字を含めることはできない。
    for (size_t i = 1; i <= 3; i++)
    {
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 1), 0u);
        EXPECT_EQ(buffer[0], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 2), 0u);
        EXPECT_EQ(buffer[0], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 3), 0u);
        EXPECT_EQ(buffer[0], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 4), 0u);
        EXPECT_EQ(buffer[0], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 5), 0u);
        EXPECT_EQ(buffer[0], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 6), 0u);
        EXPECT_EQ(buffer[0], 0);
    }

    // 3 バイト文字を 1 文字だけコピーできる。
    for (size_t i = 4; i <= 6; i++)
    {
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 1), 3u);
        EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
        EXPECT_EQ(buffer[3], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 2), 3u);
        EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
        EXPECT_EQ(buffer[3], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 3), 3u);
        EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
        EXPECT_EQ(buffer[3], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 4), 3u);
        EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
        EXPECT_EQ(buffer[3], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 5), 3u);
        EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
        EXPECT_EQ(buffer[3], 0);
        EXPECT_EQ(CopyUtf8String(buffer, i, message, sizeof (message), 6), 3u);
        EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
        EXPECT_EQ(buffer[3], 0);
    }
}

// 途中で切れている (「あ」+「い」＋「う」の 2 バイト目まで）
TEST(Utf8StringUtil, Basic3)
{
    const char message[] = {'\xE3', '\x81', '\x82', '\xE3', '\x81', '\x84', '\xE3', '\x81','\0'};

    char buffer[128] = {};

    // 「あ」までコピーできる。
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 1), 3u);
    EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
    EXPECT_EQ(buffer[3], 0);
    // 「い」までコピーできる。
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 2), 6u);
    EXPECT_EQ(std::memcmp(buffer, message, 6), 0);
    EXPECT_EQ(buffer[6], 0);
    // 「い」までコピーできる。
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 3), 6u);
    EXPECT_EQ(std::memcmp(buffer, message, 6), 0);
    EXPECT_EQ(buffer[6], 0);
}

TEST(Utf8StringUtil, Over1)
{
    const char message[] = {'A', 'B', 'C', '\0', 'X', 'Y', '\0', 'Z'};

    char buffer[128] = {};

    EXPECT_EQ(GetCodePointCountOfUtf8String(message, 3u), 3);
    EXPECT_EQ(GetCodePointCountOfUtf8String(message, 4u), 4);
    EXPECT_EQ(GetCodePointCountOfUtf8String(message, 5u), 5);
    EXPECT_EQ(GetCodePointCountOfUtf8String(message, 6u), 6);
    EXPECT_EQ(GetCodePointCountOfUtf8String(message, 7u), 7);
    EXPECT_EQ(GetCodePointCountOfUtf8String(message, 8u), 8);

    for (size_t l = 1; l <= sizeof (message); l++)
    {
        for (int c = 1; c <= sizeof (message) + 1; c++)
        {
            size_t expectedLength = std::min(l, static_cast<size_t>(c));

            std::memset(buffer, 0xFF, sizeof (buffer));
            EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, l, c), expectedLength);
            EXPECT_EQ(std::memcmp(buffer, message, expectedLength), 0);
            EXPECT_EQ(buffer[expectedLength], 0);
            EXPECT_EQ(static_cast<Bit8>(buffer[expectedLength + 1]), 0xFF);

            EXPECT_EQ(Strnlen(buffer, sizeof (buffer)), static_cast<int>(expectedLength <= 3 ? expectedLength : 3));
        }
    }

    EXPECT_EQ(GetCodePointCountOfUtf8String(message, Strnlen(message, sizeof (message))), 3);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, Strnlen(message, sizeof (message)), 100), 3u);
}

TEST(Utf8StringUtil, Over2)
{
    const char message[] = {'\xE3', '\x81', '\x82', '\0', '\xE3', '\x81', '\x84'};

    char buffer[128] = {};

    EXPECT_EQ(GetCodePointCountOfUtf8String(message, 3u), 1);
    EXPECT_EQ(GetCodePointCountOfUtf8String(message, 4u), 2);
    EXPECT_EQ(GetCodePointCountOfUtf8String(message, 7u), 3);

    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, 3u, 1), 3u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, 4u, 1), 3u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, 7u, 1), 3u);

    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, 3u, 2), 3u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, 4u, 2), 4u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, 7u, 2), 4u);

    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, 3u, 3), 3u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, 4u, 3), 4u);
    EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, 7u, 3), 7u);
}

TEST(Utf8StringUtil, BadString1)
{
    char buffer[128] = {};

    // 冗長な文字コード (2 バイト '/')
    {
        const char message[] = {'\xC0', '\xAF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 0u);
        EXPECT_EQ(buffer[0], 0);
    }
    // 冗長な文字コード (3 バイト '/')
    {
        const char message[] = {'\xE0', '\x80', '\xAF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 0u);
        EXPECT_EQ(buffer[0], 0);
    }
    // 冗長な文字コード (4 バイト '/')
    {
        const char message[] = {'\xF0', '\x80', '\x80', '\xAF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 0u);
        EXPECT_EQ(buffer[0], 0);
    }
    // 冗長な文字コード (5 バイト '/')
    {
        const char message[] = {'\xF8', '\x80', '\x80', '\x80', '\xAF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 0u);
        EXPECT_EQ(buffer[0], 0);
    }
    // シーケンス未完了 (「あ」の 1 バイト目まで）
    {
        const char message[] = {'\xE3', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 0u);
        EXPECT_EQ(buffer[0], 0);
    }
    // シーケンス未完了 (「あ」の 2 バイト目まで）
    {
        const char message[] = {'\xE3', '\x81', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 0u);
        EXPECT_EQ(buffer[0], 0);
    }
    // 不正なシーケンス (UTF-8 でないデータ列）
    {
        const char message[] = {'\xFD', '\xFE', '\xFF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 0u);
        EXPECT_EQ(buffer[0], 0);
    }
}

TEST(Utf8StringUtil, BadString2)
{
    char buffer[128] = {};

    // 冗長な文字コード ('/' + 2 バイト '/')
    {
        const char message[] = {'\x2F', '\xC0', '\xAF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 1u);
        EXPECT_EQ(std::memcmp(buffer, message, 1), 0);
        EXPECT_EQ(buffer[1], 0);
    }
    // 冗長な文字コード ('/' + 3 バイト '/')
    {
        const char message[] = {'\x2F', '\xE0', '\x80', '\xAF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 1u);
        EXPECT_EQ(std::memcmp(buffer, message, 1), 0);
        EXPECT_EQ(buffer[1], 0);
    }
    // 冗長な文字コード ('/' + 4 バイト '/')
    {
        const char message[] = {'\x2F', '\xF0', '\x80', '\x80', '\xAF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 1u);
        EXPECT_EQ(std::memcmp(buffer, message, 1), 0);
        EXPECT_EQ(buffer[1], 0);
    }
    // 冗長な文字コード ('/' + 5 バイト '/')
    {
        const char message[] = {'\x2F', '\xF8', '\x80', '\x80', '\x80', '\xAF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 1u);
        EXPECT_EQ(std::memcmp(buffer, message, 1), 0);
        EXPECT_EQ(buffer[1], 0);
    }
    // シーケンス未完了 (「あ」+「い」の 1 バイト目まで）
    {
        const char message[] = {'\xE3', '\x81', '\x82', '\xE3', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 3u);
        EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
        EXPECT_EQ(buffer[3], 0);
    }
    // シーケンス未完了 (「あ」+「い」の 2 バイト目まで）
    {
        const char message[] = {'\xE3', '\x81', '\x82', '\xE3', '\x81', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 3u);
        EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
        EXPECT_EQ(buffer[3], 0);
    }
    // 不正なシーケンス (「あ」+ UTF-8 でないデータ列）
    {
        const char message[] = {'\xE3', '\x81', '\x82', '\xFD', '\xFE', '\xFF', '\0'};

        std::memset(buffer, 0xFF, sizeof (buffer));
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
        EXPECT_EQ(CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), sizeof (message)), 3u);
        EXPECT_EQ(std::memcmp(buffer, message, 3), 0);
        EXPECT_EQ(buffer[3], 0);
    }
}

TEST(Utf8StringUtil, BoundaryValueTest)
{
    // 1 バイト
    {
        char message[] = {'\x0'};
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\x7F', '\x0'};
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    // 2 バイト
    {
        char message[] = {'\xC1', '\xBF', '\x0'}; // 冗長判定（1 バイトで表現可能）
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xC2', '\x80', '\x0'};
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xDF', '\xBF', '\x0'};
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    // 3 バイト
    {
        char message[] = {'\xE0', '\x9F', '\xBF', '\x0'}; // 冗長判定（2 バイトで表現可能）
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xE0', '\xA0', '\x80', '\x0'};
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xED', '\x9F', '\xBF', '\x0'}; // 0xD800 - 1
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xED', '\xA0', '\x80', '\x0'}; // 0xD800
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xED', '\xBF', '\xBF', '\x0'}; // 0xDFFF
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xEE', '\x80', '\x80', '\x0'}; // 0xDFFF + 1
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xEF', '\xBF', '\xBF', '\x0'};
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    // 4 バイト
    {
        char message[] = {'\xF0', '\x8F', '\xBF', '\xBF', '\x0'}; // 冗長判定（3 バイトで表現可能）
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xF0', '\x90', '\x80', '\x80', '\x0'};
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xF4', '\x8F', '\xBF', '\xBF', '\x0'}; // 0x10FFFF
        EXPECT_TRUE(VerifyUtf8String(message, sizeof (message)));
    }
    {
        char message[] = {'\xF4', '\x90', '\x80', '\x80', '\x0'}; // 0x10FFFF + 1
        EXPECT_FALSE(VerifyUtf8String(message, sizeof (message)));
    }
}

TEST(Utf8StringUtil, SpeedTest)
{
    const char message[] =
    {
        'A', 'B', 'C', 'D', 'E',
        '\xC2', '\x80',
        '\xE3', '\x81', '\x82',
        '\xE3', '\x81', '\x84',
        '\xE3', '\x81', '\x86',
        '\xE3', '\x81', '\x88',
        '\xE3', '\x81', '\x8A',
        '\xF4', '\x8F', '\xBF', '\xBF', '\0'
    };

    const int LoopCount = 100000;

    {
        nn::os::Tick start = nn::os::GetSystemTick();

        for (int i = 0; i < LoopCount; i++)
        {
            VerifyUtf8String(message, sizeof (message));
        }

        nn::TimeSpan elapsedTime = (nn::os::GetSystemTick() - start).ToTimeSpan();

        NN_LOG("!!!!! VerifyUtf8String loop = %d, time = %lld us\n", LoopCount, elapsedTime.GetMicroSeconds());
    }
    {
        char buffer[128] = {};

        nn::os::Tick start = nn::os::GetSystemTick();

        for (int i = 0; i < LoopCount; i++)
        {
            CopyUtf8String(buffer, sizeof (buffer), message, sizeof (message), 100);
        }

        nn::TimeSpan elapsedTime = (nn::os::GetSystemTick() - start).ToTimeSpan();

        NN_LOG("!!!!! CopyUtf8String loop = %d, time = %lld us\n", LoopCount, elapsedTime.GetMicroSeconds());
    }
}

}} // namespace nn::util
