﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <limits>
#include <cstring>
#include <nnt/nntest.h>

// NN_ALIGNAS には整数リテラルしか書けないので、const 定数ではなくマクロを使用する。
#define NN_DETAIL_TEST_UTIL_ALIGNMENT 16 // NOLINT(readability/define)

namespace {

#if NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC
// 明示的にパディングを挟むと NN_ALIGNAS のテストにならないのでここでは警告を無効化します。
#pragma warning( push )
#pragma warning( disable:4324 )
#endif

class NN_ALIGNAS( NN_DETAIL_TEST_UTIL_ALIGNMENT ) MacroTest
{
    NN_DISALLOW_COPY( MacroTest );
    NN_DISALLOW_MOVE( MacroTest );

public:

    static const int Alignment = NN_DETAIL_TEST_UTIL_ALIGNMENT;

    MacroTest() : m_Padding()
    {
    }

    NN_NORETURN NN_NOINLINE
    void ThrowException()
    {
#if NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC
        throw 0;
#else
        for (;;) {}
#endif
    }

    NN_FORCEINLINE
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return m_Padding != 0;
    }

private:
    char m_Padding;
};

struct AlignmentTest
{
    char padding;
    NN_ALIGNAS( NN_DETAIL_TEST_UTIL_ALIGNMENT ) int aligned;
};

#if NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC
#pragma warning( pop )
#endif


NN_DEFINE_STATIC_CONSTANT( const int MacroTest::Alignment );

// この関数自身の名前を NN_CURRENT_FUNCTION_NAME で取得します
const char* GetNameOfThisFunctionItself()
{
    return NN_CURRENT_FUNCTION_NAME;
}

void InvokeUnexpectedDefaultUnlessZero(int n)
{
    switch (n)
    {
    case 0:
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

TEST( Macro, LanguageMacro )
{
    NN_ALIGNAS( NN_DETAIL_TEST_UTIL_ALIGNMENT ) char aligned = 0;
    NN_UNUSED( aligned ); // 変数としては使用しないので未使用扱いとします。

    EXPECT_EQ( NN_DETAIL_TEST_UTIL_ALIGNMENT, NN_ALIGNOF( aligned ) );
    EXPECT_EQ( NN_DETAIL_TEST_UTIL_ALIGNMENT, NN_ALIGNOF( AlignmentTest ) );
    EXPECT_EQ( NN_DETAIL_TEST_UTIL_ALIGNMENT, NN_ALIGNOF( MacroTest ) );

    EXPECT_EQ( 16, sizeof( MacroTest ) );
    EXPECT_EQ( 128, NN_BITSIZEOF( MacroTest ) );

    // NN_CURRENT_FUNCTION_NAME の値に関数名が含まれていることを期待します。
    // 処理系定義なので厳密には保証がありませんが、
    // NN_CURRENT_FUNCTION_NAME の実装が誤っているにもかかわらずテストを通過することはありません。
    EXPECT_NE( nullptr, std::strstr(GetNameOfThisFunctionItself(), "GetNameOfThisFunctionItself") );
}

TEST( Macro, CompileAndRun )
{
    // コンパイル及び実行のテストです。
    // テストを完走することでしか結果を取得できません。

    // コピー禁止かつムーブ禁止のクラスです。
    MacroTest macroTest;

    // 修飾子付きの関数群のテストです。

#if NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC
    try
    {
        macroTest.ThrowException();
    }
    catch ( ... )
    {
    }
#endif

    auto  boolean = static_cast< bool >( macroTest );
    NN_UNUSED( boolean );

    // 各種マクロのテストです。

    do
    {
    } while ( NN_STATIC_CONDITION( 0 ) );

    // Windows.h で定義されている min マクロを回避します。
    auto min = std::numeric_limits< float >::min NN_PREVENT_MACRO_FUNC ();
    NN_UNUSED( min );

    NN_PRAGMA_PUSH_WARNINGS
    NN_DISABLE_WARNING_UNUSED_VAR
    NN_DISABLE_WARNING_LOCAL_STATIC

    static auto NN_MAKE_TEMPORARY_NAME( dummy ) = 0;
    static auto NN_MAKE_TEMPORARY_NAME( dummy ) = 1;

    NN_PRAGMA_POP_WARNINGS

    switch (0)
    {
    case 0:
        NN_FALL_THROUGH;
    default:
        break;
    }
}

TEST(MacroDeathTest, UnexpectedDefault)
{
    // NN_UNEXPECTED_DEFAULT で停止することのテストです。
    EXPECT_DEATH_IF_SUPPORTED( InvokeUnexpectedDefaultUnlessZero(-1), "" );
}

NN_PRAGMA_PUSH_WARNINGS
#if !defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_GCC)
NN_ENABLE_WARNING_IMPLICIT_PADDING_WITHIN_CLASS
#endif

TEST( Macro, PaddingMacro )
{
    // 各マクロのサイズとアライメントが正しいことの確認
    {
        struct PaddingSizeTest1
        {
            char a;
            NN_PADDING1;
        };
        EXPECT_EQ(std::alignment_of<PaddingSizeTest1>::value, 1);
        EXPECT_EQ(sizeof(PaddingSizeTest1), 2);

        struct PaddingSizeTest2
        {
            char a;
            NN_PADDING2;
        };
        EXPECT_EQ(std::alignment_of<PaddingSizeTest2>::value, 1);
        EXPECT_EQ(sizeof(PaddingSizeTest2), 3);

        struct PaddingSizeTest3
        {
            char a;
            NN_PADDING3;
        };
        EXPECT_EQ(std::alignment_of<PaddingSizeTest3>::value, 1);
        EXPECT_EQ(sizeof(PaddingSizeTest3), 4);

        struct PaddingSizeTest4
        {
            char a;
            NN_PADDING4;
        };
        EXPECT_EQ(std::alignment_of<PaddingSizeTest4>::value, 1);
        EXPECT_EQ(sizeof(PaddingSizeTest4), 5);

        struct PaddingSizeTest5
        {
            char a;
            NN_PADDING5;
        };
        EXPECT_EQ(std::alignment_of<PaddingSizeTest5>::value, 1);
        EXPECT_EQ(sizeof(PaddingSizeTest5), 6);

        struct PaddingSizeTest6
        {
            char a;
            NN_PADDING6;
        };
        EXPECT_EQ(std::alignment_of<PaddingSizeTest6>::value, 1);
        EXPECT_EQ(sizeof(PaddingSizeTest6), 7);

        struct PaddingSizeTest7
        {
            char a;
            NN_PADDING7;
        };
        EXPECT_EQ(std::alignment_of<PaddingSizeTest7>::value, 1);
        EXPECT_EQ(sizeof(PaddingSizeTest7), 8);

        struct PaddingSizeTest8
        {
            char a;
            NN_PADDING8;
        };
        EXPECT_EQ(std::alignment_of<PaddingSizeTest8>::value, 1);
        EXPECT_EQ(sizeof(PaddingSizeTest8), 9);
    }
}

//--------------------------------------------------------------------------------------------------
// NN_CHAR16LITERAL 用のテスト

TEST( Macro, StringLiteral )
{
    const uint16_t* pStringVal = reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("abcdefg"));
    EXPECT_EQ(0x63 /* c */, pStringVal[2]);
    EXPECT_EQ(0x64 /* d */, pStringVal[3]);
}

//--------------------------------------------------------------------------------------------------

//
// nnc/nn_Macro.h 用のテスト
//
extern "C" bool MacroForC_LanguageMacro( void );
extern "C" bool MacroForC_CompileAndRun( void );
extern "C" bool MacroForCDeathTest_UnexpectedDefault( void );
extern "C" NN_NORETURN void MacroForCDeathTest_ExpectedAssertFailure( void );
extern "C" NN_NORETURN void MacroForCDeathTest_ExpectedAbort( void );

// 現状、未対応
//extern "C" bool MacroForC_PaddingMacro( void );

TEST( MacroForC, LanguageMacro )
{
    EXPECT_TRUE( MacroForC_LanguageMacro() );
}

TEST( MacroForC, CompileAndRun )
{
    EXPECT_TRUE( MacroForC_CompileAndRun() );
}

TEST(MacroForCDeathTest, UnexpectedDefault)
{
    EXPECT_DEATH_IF_SUPPORTED( MacroForCDeathTest_UnexpectedDefault(), "" );
}

TEST( MacroForCDeathTest, AssertTest )
{
    EXPECT_DEATH_IF_SUPPORTED( MacroForCDeathTest_ExpectedAssertFailure(), "" );
}

TEST( MacroForCDeathTest, AbortTest )
{
    EXPECT_DEATH_IF_SUPPORTED( MacroForCDeathTest_ExpectedAbort(), "" );
}

// 現状、未対応
#if 0
TEST( MacroForC, PaddingMacro )
{
    EXPECT_TRUE( MacroForC_PaddingMacro() );
}
#endif

NN_PRAGMA_POP_WARNINGS

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW
int globalShadow; // NOLINT
struct SuppressShadowWarningTest
{
    int memberShadow;

    void Test(int memberShadow, int argumentShadow, int globalShadow)
    {
        int localShadow = argumentShadow;
        {
            int argumentShadow = localShadow;
            int localShadow = 10;
            this->memberShadow = memberShadow + argumentShadow + localShadow + globalShadow;
        }
    }
};
NN_PRAGMA_POP_WARNINGS

//--------------------------------------------------------------------------------------------------
// プラットフォーム情報マクロのテスト
#if defined(NN_BUILD_CONFIG_SPEC_GENERIC)

    #if !defined(NN_BUILD_CONFIG_SPEC_GENERIC)
        #error
    #endif

#elif defined(NN_BUILD_CONFIG_SPEC_CAFE)

    #if !defined(NN_BUILD_APISET_CAFE)
        #error
    #endif

#elif defined(NN_BUILD_CONFIG_SPEC_NX)

    #if !defined(NN_BUILD_APISET_NX)
        #error
    #endif

#else
    #error Unknown NN_BUILD_CONFIG_SPEC
#endif

#if defined(NN_BUILD_CONFIG_OS_WIN)

    #if !defined(NN_BUILD_TARGET_PLATFORM_WIN)
        #error
    #endif

    #if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
        #error
    #endif

#elif defined(NN_BUILD_CONFIG_OS_CAFE)

    #if !defined(NN_BUILD_TARGET_PLATFORM_CAFE)
        #error
    #endif

    #if !defined(NN_BUILD_TARGET_PLATFORM_OS_CAFE)
        #error
    #endif

#elif defined(NN_BUILD_CONFIG_OS_HORIZON)

    #if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        // NX

    #if !defined(NN_BUILD_TARGET_PLATFORM_NX)
        #error
    #endif

    #elif defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || \
        defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || \
        defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || \
        defined(NN_BUILD_CONFIG_HARDWARE_JUNO) || \
        defined(NN_BUILD_CONFIG_HARDWARE_KZMA9) || \
        defined(NN_BUILD_CONFIG_HARDWARE_SMMA53)
        // 汎用開発ボード

        // NN_BUILD_TARGET_PLATFORM は定義されない

    #else
        #error Unknown NN_BUILD_CONFIG_HARDWARE
    #endif

    #if !defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
        #error
    #endif

#elif defined(NN_BUILD_CONFIG_OS_BAREMETAL)
    // OS_BAREMETAL は SDK 利用者向けではないため、SDK 利用者向けのマクロは定義されない
#else
    #error Unknown NN_BUILD_CONFIG_OS
#endif


#if defined(NN_BUILD_TARGET_PLATFORM_ADDRESS_32)
NN_STATIC_ASSERT(sizeof(void*) == 4);
#elif defined(NN_BUILD_TARGET_PLATFORM_ADDRESS_64)
NN_STATIC_ASSERT(sizeof(void*) == 8);
#else
    #error Unknown NN_BUILD_TARGET_PLATFORM_ADDRESS
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_ENDIAN_LITTLE)
    #if !defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
        #error
    #endif
#elif defined(NN_BUILD_TARGET_PLATFORM_ENDIAN_BIG)
    #if !defined(NN_BUILD_CONFIG_ENDIAN_BIG)
        #error
    #endif
#else
    #error Unknown NN_BUILD_TARGET_PLATFORM_ENDIAN
#endif

TEST(TargetPlatformMacro, Endian)
{
    uint32_t value = 0x01020304;
    unsigned char bs[sizeof(uint32_t)];
    std::memcpy(bs, &value, sizeof(uint32_t));

#if defined(NN_BUILD_TARGET_PLATFORM_ENDIAN_LITTLE)
    ASSERT_EQ(bs[3], 0x01);
    ASSERT_EQ(bs[2], 0x02);
    ASSERT_EQ(bs[1], 0x03);
    ASSERT_EQ(bs[0], 0x04);
#elif defined(NN_BUILD_TARGET_PLATFORM_ENDIAN_BIG)
    ASSERT_EQ(bs[0], 0x01);
    ASSERT_EQ(bs[1], 0x02);
    ASSERT_EQ(bs[2], 0x03);
    ASSERT_EQ(bs[3], 0x04);
#else
    #error Unknown NN_BUILD_TARGET_PLATFORM_ENDIAN
#endif
}
