﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/util/util_BitUtil.h>

namespace nn { namespace util {

TEST( BitUtilTest, BitUtil  )
{
#if defined( NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC )
NN_PRAGMA_PUSH_WARNINGS
__pragma( warning( disable:4310 ) )
#endif

    typedef char type;

    EXPECT_EQ( type( 0xC8 ), rstls1b( type( 0xCC ) ) ); // 1100 1100 -> 1100 1000
    EXPECT_EQ( type( 0x00 ), rstls1b( type( 0x00 ) ) ); // 0000 0000 -> 0000 0000
    EXPECT_EQ( type( 0xFE ), rstls1b( type( 0xFF ) ) ); // 1111 1111 -> 1111 1110

    EXPECT_EQ( type( 0x37 ), setls0b( type( 0x33 ) ) ); // 0011 0011 -> 0011 0111
    EXPECT_EQ( type( 0x01 ), setls0b( type( 0x00 ) ) ); // 0000 0000 -> 0000 0001
    EXPECT_EQ( type( 0xFF ), setls0b( type( 0xFF ) ) ); // 1111 1111 -> 1111 1111

    EXPECT_EQ( type( 0x30 ), rstt1( type( 0x33 ) ) ); // 0011 0011 -> 0011 0000
    EXPECT_EQ( type( 0x00 ), rstt1( type( 0x00 ) ) ); // 0000 0000 -> 0000 0000
    EXPECT_EQ( type( 0x00 ), rstt1( type( 0xFF ) ) ); // 1111 1111 -> 0000 0000

    EXPECT_EQ( type( 0xCF ), sett0( type( 0xCC ) ) ); // 1100 1100 -> 1100 1111
    EXPECT_EQ( type( 0xFF ), sett0( type( 0x00 ) ) ); // 0000 0000 -> 1111 1111
    EXPECT_EQ( type( 0xFF ), sett0( type( 0xFF ) ) ); // 1111 1111 -> 1111 1111

    EXPECT_EQ( type( 0x04 ), isols1b( type( 0xCC ) ) ); // 1100 1100 -> 0000 0100
    EXPECT_EQ( type( 0x00 ), isols1b( type( 0x00 ) ) ); // 0000 0000 -> 0000 0000
    EXPECT_EQ( type( 0x01 ), isols1b( type( 0xFF ) ) ); // 1111 1111 -> 0000 0001

    EXPECT_EQ( type( 0x04 ), isols0b( type( 0x33 ) ) ); // 0011 0011 -> 0000 0100
    EXPECT_EQ( type( 0x01 ), isols0b( type( 0x00 ) ) ); // 0000 0000 -> 0000 0001
    EXPECT_EQ( type( 0x00 ), isols0b( type( 0xFF ) ) ); // 1111 1111 -> 0000 0000

    EXPECT_EQ( type( 0x03 ), maskt1( type( 0x33 ) ) ); // 0011 0011 -> 0000 0011
    EXPECT_EQ( type( 0x00 ), maskt1( type( 0x00 ) ) ); // 0000 0000 -> 0000 0000
    EXPECT_EQ( type( 0xFF ), maskt1( type( 0xFF ) ) ); // 1111 1111 -> 1111 1111

    EXPECT_EQ( type( 0x03 ), maskt0( type( 0xCC ) ) ); // 1100 1100 -> 0000 0011
    EXPECT_EQ( type( 0xFF ), maskt0( type( 0x00 ) ) ); // 0000 0000 -> 1111 1111
    EXPECT_EQ( type( 0x00 ), maskt0( type( 0xFF ) ) ); // 1111 1111 -> 0000 0000

    EXPECT_EQ( type( 0x07 ), maskt1ls0b( type( 0x33 ) ) ); // 0011 0011 -> 0000 0111
    EXPECT_EQ( type( 0x01 ), maskt1ls0b( type( 0x00 ) ) ); // 0000 0000 -> 0000 0001
    EXPECT_EQ( type( 0xFF ), maskt1ls0b( type( 0xFF ) ) ); // FFFF FFFF -> FFFF FFFF

    EXPECT_EQ( type( 0x07 ), maskt0ls1b( type( 0xCC ) ) ); // 1100 1100 -> 0000 0111
    EXPECT_EQ( type( 0xFF ), maskt0ls1b( type( 0x00 ) ) ); // 0000 0000 -> FFFF FFFF
    EXPECT_EQ( type( 0x01 ), maskt0ls1b( type( 0xFF ) ) ); // FFFF FFFF -> 0000 0001

    EXPECT_FALSE( ispow2( type( 0x11 ) ) );
    EXPECT_TRUE( ispow2( type( 0x10 ) ) );
    EXPECT_TRUE( ispow2( type( 0x01 ) ) );
    EXPECT_FALSE( ispow2( type( 0x00 ) ) );

    EXPECT_EQ( type( 0x10 ), floorp2( type( 0x11 ) ) );
    EXPECT_EQ( type( 0x01 ), floorp2( type( 0x01 ) ) );

    EXPECT_EQ( type( 0x20 ), ceilp2( type( 0x11 ) ) );
    EXPECT_EQ( type( 0x01 ), ceilp2( type( 0x01 ) ) );

    EXPECT_EQ( type( 0x88 ), reverse_bits( type( 0x11 ) ) );
    EXPECT_EQ( type( 0x11 ), reverse_bits( type( 0x88 ) ) );

    // 0xC5 = 1100 0101
    EXPECT_EQ( type( 0x3A ), reverse_bits( type( 0xC5 ), 1, 2 ) ); // reverse_bits(ABCDEFGHb, 1, 2) -> DCBAHGFEb, 1100 0101 -> 0011 1010
    EXPECT_EQ( type( 0xCA ), reverse_bits( type( 0xC5 ), 1, 4 ) ); // reverse_bits(ABCDEFGHb, 1, 4) -> BADCFEHGb, 1100 0101 -> 1100 1010
    EXPECT_EQ( type( 0xC5 ), reverse_bits( type( 0xC5 ), 1, 8 ) ); // reverse_bits(ABCDEFGHb, 1, 8) -> ABCDEFGHb, 1100 0101 -> 1100 0101
    EXPECT_EQ( type( 0x53 ), reverse_bits( type( 0xC5 ), 2, 1 ) ); // reverse_bits(ABCDEFGHb, 2, 1) -> GHEFCDABb, 1100 0101 -> 0101 0011
    EXPECT_EQ( type( 0x35 ), reverse_bits( type( 0xC5 ), 2, 2 ) ); // reverse_bits(ABCDEFGHb, 2, 2) -> CDABGHEFb, 1100 0101 -> 0011 0101
    EXPECT_EQ( type( 0x5C ), reverse_bits( type( 0xC5 ), 4, 1 ) ); // reverse_bits(ABCDEFGHb, 4, 1) -> EFGHABCDb, 1100 0101 -> 0101 1100

    EXPECT_EQ( 2, popcount( type( 0x11 ) ) );
    EXPECT_EQ( 2, popcount( type( 0x88 ) ) );
    EXPECT_EQ( 0, popcount( type( 0x00 ) ) );
    EXPECT_EQ( 8, popcount( type( 0xFF ) ) );

    EXPECT_EQ( 8, cntl0( type( 0x00 ) ) );
    EXPECT_EQ( 3, cntl0( type( 0x11 ) ) );
    EXPECT_EQ( 0, cntl0( type( 0x88 ) ) );
    EXPECT_EQ( 8, cntt0( type( 0x00 ) ) );
    EXPECT_EQ( 0, cntt0( type( 0x11 ) ) );
    EXPECT_EQ( 3, cntt0( type( 0x88 ) ) );

    EXPECT_EQ( uint8_t( 0x12 ), reverse_bytes( uint8_t( 0x12 ) ) );
    EXPECT_EQ( uint16_t( 0x3412 ), reverse_bytes( uint16_t( 0x1234 ) ) );
    EXPECT_EQ( uint32_t( 0x78563412 ), reverse_bytes( uint32_t( 0x12345678 ) ) );
    EXPECT_EQ( uint64_t( 0xF0DEBC9A78563412 ), reverse_bytes( uint64_t( 0x123456789ABCDEF0 ) ) );

    EXPECT_TRUE( is_aligned( uint8_t( 0xA8 ), 8 ) );
    EXPECT_TRUE( is_aligned( uint16_t( 0xAAA0 ), 32 ) );
    EXPECT_TRUE( is_aligned( uint32_t( 0xAAAAAAC0 ), 64 ) );
    EXPECT_TRUE( is_aligned( uint64_t( 0xAAAAAAAAAAAAAA80 ), 128 ) );

    EXPECT_TRUE( is_aligned( int8_t( 0xA8 ), 8 ) );
    EXPECT_TRUE( is_aligned( int16_t( 0xAAA0 ), 32 ) );
    EXPECT_TRUE( is_aligned( int32_t( 0xAAAAAAC0 ), 64 ) );
    EXPECT_TRUE( is_aligned( int64_t( 0xAAAAAAAAAAAAAA80 ), 128 ) );

    EXPECT_FALSE( is_aligned( uint8_t( 0xA1 ), 8 ) );
    EXPECT_FALSE( is_aligned( uint16_t( 0xAA10 ), 32 ) );
    EXPECT_FALSE( is_aligned( uint32_t( 0xAAAAAAA0 ), 64 ) );
    EXPECT_FALSE( is_aligned( uint64_t( 0xAAAAAAAAAAAAAAC0 ), 128 ) );

    EXPECT_FALSE( is_aligned( int8_t( 0xA1 ), 8 ) );
    EXPECT_FALSE( is_aligned( int16_t( 0xAA10 ), 32 ) );
    EXPECT_FALSE( is_aligned( int32_t( 0xAAAAAAA0 ), 64 ) );
    EXPECT_FALSE( is_aligned( int64_t( 0xAAAAAAAAAAAAAAC0 ), 128 ) );

    EXPECT_EQ( uint8_t( 0x18 ), align_up( uint8_t( 0x12 ), 8 ) );
    EXPECT_EQ( uint16_t( 0x1238 ), align_up( uint16_t( 0x1234 ), 8 ) );
    EXPECT_EQ( uint32_t( 0x12345678 ), align_up( uint32_t( 0x12345674 ), 8 ) );
    EXPECT_EQ( uint64_t( 0x123456789ABCDEF8 ), align_up( uint64_t( 0x123456789ABCDEF4 ), 8 ) );

    EXPECT_EQ( int8_t( 0xA8 ), align_up( int8_t( 0xA1 ), 8 ) );
    EXPECT_EQ( int16_t( 0xA128 ), align_up( int16_t( 0xA121 ), 8 ) );
    EXPECT_EQ( int32_t( 0xA1234568 ), align_up( int32_t( 0xA1234561 ), 8 ) );
    EXPECT_EQ( int64_t( 0xA123456789ABCDE8 ), align_up( int64_t( 0xA123456789ABCDE1 ), 8 ) );

    EXPECT_EQ( uint8_t( 0x10 ), align_down( uint8_t( 0x12 ), 8 ) );
    EXPECT_EQ( uint16_t( 0x1230 ), align_down( uint16_t( 0x1234 ), 8 ) );
    EXPECT_EQ( uint32_t( 0x12345670 ), align_down( uint32_t( 0x12345674 ), 8 ) );
    EXPECT_EQ( uint64_t( 0x123456789ABCDEF0 ), align_down( uint64_t( 0x123456789ABCDEF4 ), 8 ) );

    EXPECT_EQ( int8_t( 0xA0 ), align_down( int8_t( 0xA1 ), 8 ) );
    EXPECT_EQ( int16_t( 0xA120 ), align_down( int16_t( 0xA121 ), 8 ) );
    EXPECT_EQ( int32_t( 0xA1234560 ), align_down( int32_t( 0xA1234561 ), 8 ) );
    EXPECT_EQ( int64_t( 0xA1234567890ABCD0 ), align_down( int64_t( 0xA1234567890ABCD1 ), 8 ) );

#if defined( NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC )
NN_PRAGMA_POP_WARNINGS
#endif
}

}} // namespace nn::util
