﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include <nn/util/util_BitPack.h>

// コンパイルを通すために明示的インスタンス化を行います。
template struct nn::util::BitPack< uint8_t >;
template struct nn::util::BitPack< uint16_t >;
template struct nn::util::BitPack< uint32_t >;

namespace {

struct Sample
{
    typedef nn::util::BitPack32::Field< 0, 1, bool > Bit0;
    typedef nn::util::BitPack32::Field< Bit0::Next, 1, bool > Bit1;
    typedef nn::util::BitPack32::Field< Bit1::Next, 1, bool > Bit2;
    typedef nn::util::BitPack32::Field< Bit2::Next, 1, bool > Bit3;
    typedef nn::util::BitPack32::Field< Bit3::Next, 4, int > Value4;
    typedef nn::util::BitPack32::Field< 0, 32, int > AllBit;
};

} // anonymous namespace

TEST( BitType, Bit )
{
    nn::util::BitPack32 flag = {};

    flag.SetBit( Sample::Bit0::Pos, true );
    ASSERT_EQ( Sample::Bit0::Mask, flag.storage );

    flag.SetBit( Sample::Bit1::Pos, true );
    ASSERT_EQ( Sample::Bit0::Mask | Sample::Bit1::Mask, flag.storage );

    flag.SetBit( Sample::Bit2::Pos, true );
    ASSERT_EQ( Sample::Bit0::Mask | Sample::Bit1::Mask | Sample::Bit2::Mask, flag.storage );

    flag.SetBit( Sample::Bit1::Pos, false );
    ASSERT_EQ( Sample::Bit0::Mask | Sample::Bit2::Mask, flag.storage );

    flag.SetBit( Sample::Bit3::Pos, !flag.GetBit( Sample::Bit3::Pos ) );
    ASSERT_EQ( Sample::Bit0::Mask | Sample::Bit2::Mask | Sample::Bit3::Mask, flag.storage );

    ASSERT_TRUE( flag.GetBit( Sample::Bit0::Pos ) );
    ASSERT_FALSE( flag.GetBit( Sample::Bit1::Pos ) );

    flag.Clear();
    ASSERT_TRUE( flag.IsAllBitOff( Sample::AllBit::Mask ) );
}

TEST( BitType, MaskedBits )
{
    nn::util::BitPack32 flag = {};
    flag.storage = Sample::Bit1::Mask | Sample::Bit3::Mask;

    ASSERT_TRUE( flag.IsAllBitOn( Sample::Bit1::Mask | Sample::Bit3::Mask ) );
    ASSERT_FALSE( flag.IsAllBitOn( Sample::Bit1::Mask | Sample::Bit2::Mask ) );
    ASSERT_TRUE( flag.IsAnyBitOn( Sample::Bit1::Mask | Sample::Bit2::Mask ) );
    ASSERT_FALSE( flag.IsAnyBitOn( Sample::Bit0::Mask | Sample::Bit2::Mask ) );

    ASSERT_TRUE( flag.IsAllBitOff( Sample::Bit0::Mask | Sample::Bit2::Mask ) );
    ASSERT_FALSE( flag.IsAllBitOff( Sample::Bit1::Mask | Sample::Bit2::Mask ) );
    ASSERT_TRUE( flag.IsAnyBitOff( Sample::Bit1::Mask | Sample::Bit2::Mask ) );
    ASSERT_FALSE( flag.IsAnyBitOff( Sample::Bit1::Mask | Sample::Bit3::Mask ) );

    flag.SetAllBitOff( Sample::Bit1::Mask | Sample::Bit3::Mask );
    ASSERT_TRUE( flag.IsAllBitOff( Sample::Bit1::Mask | Sample::Bit3::Mask ) );

    flag.SetAllBitOn( Sample::Bit1::Mask | Sample::Bit3::Mask );
    ASSERT_TRUE( flag.IsAllBitOn( Sample::Bit1::Mask | Sample::Bit3::Mask ) );

    flag.SetMaskedBits( Sample::Bit2::Mask | Sample::Bit3::Mask, Sample::Bit2::Mask );
    ASSERT_EQ( Sample::Bit1::Mask | Sample::Bit2::Mask, flag.storage );

    flag.SetMaskedBits( Sample::Bit2::Mask | Sample::Bit3::Mask, 0 );
    ASSERT_EQ( Sample::Bit1::Mask, flag.storage );

    flag.SetMaskedBits( Sample::Bit2::Mask | Sample::Bit3::Mask,
        Sample::Bit2::Mask | Sample::Bit3::Mask );
    ASSERT_EQ( Sample::Bit1::Mask | Sample::Bit2::Mask | Sample::Bit3::Mask, flag.storage );

    flag.SetMaskedBits( Sample::Bit1::Mask | Sample::Bit2::Mask, Sample::Bit2::Mask );
    ASSERT_EQ( Sample::Bit2::Mask | Sample::Bit3::Mask, flag.storage );

    ASSERT_EQ( Sample::Bit2::Mask | Sample::Bit3::Mask, flag.GetMaskedBits(
        Sample::Bit0::Mask | Sample::Bit1::Mask | Sample::Bit2::Mask | Sample::Bit3::Mask ) );

}

TEST( BitType, Field )
{
    nn::util::BitPack32 flag = {};
    flag.storage = Sample::Bit1::Mask | Sample::Bit3::Mask;

    flag.Set< Sample::Value4 >( 0xC );
    ASSERT_EQ( 0xCA, flag.storage );
    ASSERT_EQ( 0xC, flag.Get< Sample::Value4 >() );

    ASSERT_TRUE( flag.Get< Sample::Bit1 >() );
    ASSERT_FALSE( flag.Get< Sample::Bit0 >() );
}
