﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include "UsbDiag_Theme.h"
#include "UsbDiag_UsbModel.h"
#include "UsbDiag_DeviceView.h"
#include "UsbDiag_DeviceListView.h"

namespace nnt {
namespace usb {

DeviceListView::DeviceListView(const glv::space_t width, const glv::space_t height) NN_NOEXCEPT
    : glv::Scroll(glv::Rect(width, height), DeviceListScrollBarWidth, 1, 1)
    , m_pTable(nullptr)
{
    disable(glv::Property::DrawBack       |
            glv::Property::DrawBorder     |
            glv::Property::FocusToTop     |
            glv::Property::FocusHighlight |
            glv::Property::Animate);

    mSliderX.disable (glv::Property::DrawBorder | glv::Property::FocusToTop);
    mSliderY.disable (glv::Property::DrawBorder | glv::Property::FocusToTop);
    mSliderXY.disable(glv::Property::DrawBorder | glv::Property::FocusToTop);

    addHandler(m_UpdateHandler);

    g_Model.attach(OnDeviceUpdate, Event_UpdateDevice, this);
}

DeviceListView::~DeviceListView() NN_NOEXCEPT
{
    g_Model.detach(OnDeviceUpdate, Event_UpdateDevice, this);
}

void DeviceListView::OnDeviceUpdate(const glv::Notification& n) NN_NOEXCEPT
{
    DeviceListView *self = n.receiver<DeviceListView>();
    const DeviceMap& deviceMap = *n.data<DeviceMap>();

    glv::Table *pTable = new glv::Table("<", 0, 0);

    for (auto& kv : deviceMap)
    {
        glv::View *spacer = new glv::View(glv::Rect(DeviceWidth, DeviceListGapY));
        DeviceView *deviceView = new DeviceView(kv.second);

        *pTable << deviceView << spacer;

        spacer->disable(glv::Property::DrawBorder   |
                        glv::Property::DrawBack     |
                        glv::Property::Controllable |
                        glv::Property::HitTest);
    }

    pTable->arrange();

    // set the anchor and stretch AFTER arrange()
    glv::View *cv = pTable->child;
    while (cv)
    {
        cv->anchor(0, 0).stretch(1, 0);
        cv = cv->sibling;
    }

    self->m_UpdateHandler.SetDeviceTable(pTable);
}

void DeviceListView::UpdateHandler::SetDeviceTable(glv::Table *p) NN_NOEXCEPT
{
    pTable = p;
    needsUpdate = true;
}

bool DeviceListView::UpdateHandler::onDraw(glv::View& v, glv::GLV& g) NN_NOEXCEPT
{
    DeviceListView *deviceList = static_cast<DeviceListView*>(&v);

    if (!needsUpdate)
    {
        return true;
    }

    needsUpdate = false;

    delete deviceList->m_pTable;

    *deviceList << pTable;

    deviceList->m_pTable = pTable;

    pTable->width(deviceList->width() - DeviceListGapX - DeviceListScrollBarWidth);
    pTable->anchor(0, 0).stretch(1, 0);

    return true;
}

} // ~usb
} // ~nnt
