﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/os/os_UserExceptionHandler.h>
#include <nnt/nntest.h>

#include <nn/usb/pd/usb_PdCradle.h>
#include <nnt/usbPdUtil/testUsbPd_util.h>

namespace {
    const int RepeatNum = 100;
}

void CheckCradleUsbHubVBusDetectResult( nn::Result result, nn::usb::pd::VdmMcuFwVersion mcuFwVersion, nn::usb::pd::Session* pSession )
{
    if ( result <= nn::usb::pd::ResultErrorReply() )
    {
        NN_SDK_LOG("ResultErrorReply from cradle\n");
    }
    else if ( result <= nn::usb::pd::ResultInactive() )
    {
        NN_SDK_LOG("ResultInactive\n");
    }
    else if ( result <= nn::usb::pd::ResultInvalidDevice() )
    {
        NN_SDK_LOG("ResultInvalidDevice\n");
    }
    else if ( result <= nn::usb::pd::ResultTimeout() )
    {
        NN_SDK_LOG("ResultTimeout\n");
        nn::usb::pd::Status status;
        nn::usb::pd::Notice notice;
        // クレードル挿抜未発生の確認のため Status -> Notice の順に読む必要あり
        nn::usb::pd::GetStatus( &status, pSession );
        nn::usb::pd::GetNotice( &notice, pSession );
        bool hasCradleBeenActive = nnt::usb::pd::IsCradleFamily( status.GetDeviceType() ) &&
                                   status.IsActive() &&
                                   !notice.IsActiveNotice();
        // クレードル DP1 接続継続中はタイムアウトしないはず
        if ( mcuFwVersion >= nn::usb::pd::CradleMcuFwRevisionDp1 &&
             hasCradleBeenActive
           )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    else if ( result.IsFailure() )
    {
        NN_SDK_LOG("fatal Result\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}

TEST( UsbPdCradleUsbHubVBusDetect, CradleUsbHubVBusDetect )
{
    NN_SDK_LOG("[testUsbPd_CradleUsbHubVBusDetect] Main\n");

    nn::Result result;

    nn::usb::pd::Initialize();
    nn::usb::pd::InitializeCradle();
    nn::usb::pd::Session session;
    nn::usb::pd::OpenSession( &session );
    nn::usb::pd::CradleSession cradleSession;
    nn::usb::pd::OpenCradleSession( &cradleSession );
    nn::usb::pd::Notice notice;
    // 通知フラグ空読みによるクリア
    nn::usb::pd::GetNotice( &notice, &session );
    nn::usb::pd::Vdo vdo;
    nn::usb::pd::GetCradleVdo( &vdo, &cradleSession, nn::usb::pd::CradleVdmCommand_McuFwVersion );
    nn::usb::pd::VdmMcuFwVersion mcuFwVersion = vdo.storage;
    for ( int i = 0; i < RepeatNum; i++ )
    {
        vdo.Clear();
        result = nn::usb::pd::SetCradleVdo( &cradleSession, &vdo, nn::usb::pd::CradleVdmCommand_UsbHubVBusDetect );
        if ( result.IsFailure() )
        {
            NN_SDK_LOG("SetCradleVdo is ");
        }
        else
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
            result = nn::usb::pd::GetCradleVdo( &vdo, &cradleSession, nn::usb::pd::CradleVdmCommand_UsbHubVBusDetect );
            if ( result.IsFailure() )
            {
                NN_SDK_LOG("GetCradleVdo is ");
            }
        }
        CheckCradleUsbHubVBusDetectResult( result, mcuFwVersion, &session );
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 200 ) );
    }
    nn::usb::pd::CloseCradleSession( &cradleSession );
    nn::usb::pd::CloseSession( &session );
    nn::usb::pd::FinalizeCradle();
    nn::usb::pd::Finalize();
}

