﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/nn_Windows.h>
#include <winusb.h>
#include <memory>
#pragma warning(disable: 4668)
#include <setupapi.h>

#include "../Parameter.h"
#include "../../Common/UsbTestUtil.h"
#include "../../Common/UsbTestPcUtil.h"

namespace nnt {
namespace usb {
namespace ds {

StopWatch watch;

UsbDsTestEnvironment* UsbTestEnv;
extern "C" void nninitStartup()
{
    NN_LOG("DsPerformance Environment Setup..\n");

    ::testing::Environment* TestEnv = ::testing::AddGlobalTestEnvironment(new UsbDsTestEnvironment);
    UsbTestEnv = static_cast<UsbDsTestEnvironment*>(TestEnv);
}

uint32_t seed = 1;

unsigned char sendBuffer[DataSize + 4096];
unsigned char recvBuffer[DataSize + 4096];

class DsPerformancePcHostTest : public ::testing::TestWithParam<uint32_t>
{
public:
    virtual void SetUp() override
    {
        memset(recvBuffer, 0, sizeof(recvBuffer));
        memset(sendBuffer, 0, sizeof(sendBuffer));
    }
};


INSTANTIATE_TEST_CASE_P(
    PerformanceTest, DsPerformancePcHostTest, testing::ValuesIn(BlockSize)
);


TEST_P(DsPerformancePcHostTest, BulkReadSync)
{
    int argc    = nnt::GetHostArgc();
    char **argv = nnt::GetHostArgv();

    for (int i = 0; i < argc; i++)
    {
        NN_LOG("%d %s\n", i, argv[i]);
    }

    UsbDevice *pDevice = UsbTestEnv->GetDevice();

    watch.Start();
    pDevice->ReceiveDataSync(recvBuffer, DataSize, DataSize);
    watch.Stop();

    // check data
    ASSERT_TRUE(CheckGaloisPattern(recvBuffer, DataSize, seed++));

    int elapsed = watch.GetElapsedMicroseconds();

    NN_LOG("[PERF:I:S] 0x%x\t0x%-10x\t%10d us\t%10.0f BPS\n", DataSize, GetParam(),
           elapsed, DataSize * (1000000.0 / elapsed));

    NN_LOG(
        "##teamcity[buildStatisticValue key='%s_Bandwidth(MB/s)_BulkReadSync_%dB' value='%f']\n",
        argc >= 2 ? argv[1] : "",
        GetParam(),
        DataSize / ((float)elapsed / (float)1000000) / (1024 * 1024)
        );
}

TEST_P(DsPerformancePcHostTest, BulkWriteSync)
{
    int argc    = nnt::GetHostArgc();
    char **argv = nnt::GetHostArgv();

    for (int i = 0; i < argc; i++)
    {
        NN_LOG("%d %s\n", i, argv[i]);
    }

    UsbDevice *pDevice = UsbTestEnv->GetDevice();

    // prepare data
    MakeGaloisPattern(sendBuffer, DataSize, seed++);

    watch.Start();
    pDevice->SendDataSync(sendBuffer, DataSize, DataSize);
    watch.Stop();

    int elapsed = watch.GetElapsedMicroseconds();

    NN_LOG("[PERF:O:S] 0x%x\t0x%-10x\t%10d us\t%10.0f BPS\n", DataSize, GetParam(),
           elapsed, DataSize * (1000000.0 / elapsed));

    NN_LOG(
            "##teamcity[buildStatisticValue key='%s_Bandwidth(MB/s)_BulkWriteSync_%dB' value='%f']\n",
            argc >= 2 ? argv[1] : "",
            GetParam(),
            DataSize / ((float)elapsed / (float)1000000) / (1024 * 1024)
            );
}

#ifdef ENABLE_ASYNC_TEST
TEST_P(DsPerformancePcHostTest, BulkReadAsync)
{
    int argc    = nnt::GetHostArgc();
    char **argv = nnt::GetHostArgv();

    for (int i = 0; i < argc; i++)
    {
        NN_LOG("%d %s\n", i, argv[i]);
    }

    UsbDevice *pDevice = UsbTestEnv->GetDevice();

    watch.Start();
    pDevice->ReceiveDataAsync(recvBuffer, DataSize, DataSize);
    watch.Stop();

    // check data
    ASSERT_TRUE(CheckGaloisPattern(recvBuffer, DataSize, seed++));

    int elapsed = watch.GetElapsedMicroseconds();

    NN_LOG("[PERF:I:A] 0x%x\t0x%-10x\t%10d us\t%10.0f BPS\n", DataSize, GetParam(),
           elapsed, DataSize * (1000000.0 / elapsed));

    NN_LOG(
            "##teamcity[buildStatisticValue key='%s_Bandwidth(MB/s)_BulkReadAsync_%dB' value='%f']\n",
            argc >= 2 ? argv[1] : "",
            GetParam(),
            DataSize / ((float)elapsed / (float)1000000) / (1024 * 1024)
            );
}

TEST_P(DsPerformancePcHostTest, BulkWriteAsync)
{
    int argc    = nnt::GetHostArgc();
    char **argv = nnt::GetHostArgv();

    for (int i = 0; i < argc; i++)
    {
        NN_LOG("%d %s\n", i, argv[i]);
    }

    UsbDevice *pDevice = UsbTestEnv->GetDevice();

    // prepare data
    MakeGaloisPattern(sendBuffer, DataSize, seed++);

    watch.Start();
    pDevice->SendDataAsync(sendBuffer, DataSize, DataSize);
    watch.Stop();

    int elapsed = watch.GetElapsedMicroseconds();

    NN_LOG("[PERF:O:A] 0x%x\t0x%-10x\t%10d us\t%10.0f BPS\n", DataSize, GetParam(),
           elapsed, DataSize * (1000000.0 / elapsed));

    NN_LOG(
            "##teamcity[buildStatisticValue key='%s_Bandwidth(MB/s)_BulkWriteAsync_%dB' value='%f']\n",
            argc >= 2 ? argv[1] : "",
            GetParam(),
            DataSize / ((float)elapsed / (float)1000000) / (1024 * 1024)
            );
}

#endif

} // end of namespace ds
} // end of namespace usb
} // end of namespace nnt
