﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <nnt.h>
#include <nn/nn_Result.h>
#include <nn/usb/usb_Device.h>

namespace nn    {
namespace usb  {
namespace test  {

using namespace nn;
using namespace nn::usb;

const int m_testMaxDescriptors = 8;

class UsbDsInterface
{
public:




    static DsEndpoint m_EndPoints[DsLimitMaxInterfacesPerConfigurationCount][UsbLimitMaxEndpointsCount];
    static DsClient m_DsClient;
    static DsInterface m_DsInterfaces[DsLimitMaxInterfacesPerConfigurationCount];
    static int m_numInterfaces;

    static Result InitializeClient(ComplexId complexId) NN_NOEXCEPT
    {
        NN_LOG("Initialize USB client..\n");

        UsbStringDescriptor languageStringDescriptor            = {4,   UsbDescriptorType_String, {0x0409}};
        UsbStringDescriptor manufacturerStringDescriptor        = {18,  UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o'}};
        UsbStringDescriptor productStringDescriptor             = {32,  UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o', 'S', 'd', 'k', 'T', 'e', 's', 't'}};
        UsbStringDescriptor serialNumberStringDescriptor        = {26,  UsbDescriptorType_String, {'S', 'e', 'r', 'i', 'a', 'l', 'N', 'u', 'm', 'b', 'e', 'r'}};
        UsbStringDescriptor interfaceStringDescriptor           = {20,  UsbDescriptorType_String, {'I', 'n', 't', 'e', 'r', 'f', 'a', 'c', 'e'}};

        UsbDeviceDescriptor fullSpeedDeviceDescriptor =
        {
            UsbDescriptorSize_Device,                   // bLength
            UsbDescriptorType_Device,                   // bDescriptorType
            0x0110,                                     // bcdUSB 1.1
            0x00,                                       // bDeviceClass
            0x00,                                       // bDeviceSubClass
            0x00,                                       // bDeviceProtocol
            0x40,                                       // bMaxPacketSize0
            0x057e,                                     // idVendor
            0x3001,                                     // idProduct
            0x0100,                                     // bcdDevice
            0x01,                                       // iManufacturer
            0x02,                                       // iProduct
            0x03,                                       // iSerialNumber
            0x01                                        // bNumConfigurations
        };


        UsbDeviceDescriptor highSpeedDeviceDescriptor =
        {
            UsbDescriptorSize_Device,                   // bLength
            UsbDescriptorType_Device,                   // bDescriptorType
            0x0200,                                     // bcdUSB 2.0
            0x00,                                       // bDeviceClass
            0x00,                                       // bDeviceSubClass
            0x00,                                       // bDeviceProtocol
            0x40,                                       // bMaxPacketSize0
            0x057e,                                     // idVendor
            0x3001,                                     // idProduct
            0x0100,                                     // bcdDevice
            0x01,                                       // iManufacturer
            0x02,                                       // iProduct
            0x03,                                       // iSerialNumber
            0x01                                        // bNumConfigurations
        };


        UsbDeviceDescriptor superSpeedDeviceDescriptor =
        {
            UsbDescriptorSize_Device,                   // bLength
            UsbDescriptorType_Device,                   // bDescriptorType
            0x0300,                                     // bcdUSB 3.0
            0x00,                                       // bDeviceClass
            0x00,                                       // bDeviceSubClass
            0x00,                                       // bDeviceProtocol
            0x09,                                       // bMaxPacketSize0, SS 512 or 2^9
            0x057e,                                     // idVendor
            0x3001,                                     // idProduct
            0x0100,                                     // bcdDevice
            0x01,                                       // iManufacturer
            0x02,                                       // iProduct
            0x03,                                       // iSerialNumber
            0x01                                        // bNumConfigurations
        };


        uint8_t binaryObjectStore[] =
        {
            0x05,                                       // bLength
            UsbDescriptorType_Bos,                      // bDescriptorType
            0x16,0x00,                                  // Length of this descriptor and all sub descriptors
            0x02,                                       // Number of device capability descriptors

            // USB 2.0 extension
            0x07,                                       // bLength
            UsbDescriptorType_DeviceCapability,         // bDescriptorType
            0x02,                                       // USB 2.0 extension capability type
            0x02,0x00,0x00,0x00,                        // Supported device level features: LPM support

            // SuperSpeed device capability
            0x0A,                                       // bLength
            UsbDescriptorType_DeviceCapability,         // bDescriptorType
            0x03,                                       // SuperSpeed device capability type
            0x00,                                       // Supported device level features
            0x0e,0x00,                                  // Speeds supported by the device : SS, HS
            0x03,                                       // Functionality support
            0x00,                                       // U1 Device Exit latency
            0x00,0x00                                   // U2 Device Exit latency
        };

        uint8_t stringIndex;

        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.Initialize(complexId));
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.ClearDeviceData());

        // Add string descriptors
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.AddUsbStringDescriptor(&stringIndex, &languageStringDescriptor));
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.AddUsbStringDescriptor(&stringIndex, &manufacturerStringDescriptor));
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.AddUsbStringDescriptor(&stringIndex, &productStringDescriptor));
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.AddUsbStringDescriptor(&stringIndex, &serialNumberStringDescriptor));
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.AddUsbStringDescriptor(&stringIndex, &interfaceStringDescriptor));

        // Add device descriptors
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.SetUsbDeviceDescriptor(&fullSpeedDeviceDescriptor,   UsbDeviceSpeed_Full));
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.SetUsbDeviceDescriptor(&highSpeedDeviceDescriptor,   UsbDeviceSpeed_High));
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.SetUsbDeviceDescriptor(&superSpeedDeviceDescriptor,  UsbDeviceSpeed_Super));

        // Binary Object Store descriptor is required for 3.0 host.
        NNT_EXPECT_RESULT_SUCCESS(m_DsClient.SetBinaryObjectStore(binaryObjectStore, sizeof(binaryObjectStore)));

        m_numInterfaces = 0;

        return ResultSuccess();;
    }

    // Call for each interface
    static Result InitializeDescriptors(
                                        UsbInterfaceDescriptor          *pUsbInterfaceDescriptor,
                                        UsbEndpointDescriptor           *pUsbEndpointDescriptorFullSpeed,
                                        UsbEndpointDescriptor           *pUsbEndpointDescriptorHighSpeed,
                                        UsbEndpointDescriptor           *pUsbEndpointDescriptorSuperSpeed,
                                        UsbEndpointCompanionDescriptor  *pUsbEndpointCompanionDescriptor
                                        ) NN_NOEXCEPT
    {
        Result result;

        if (pUsbInterfaceDescriptor->bNumEndpoints <= 0 || pUsbInterfaceDescriptor->bNumEndpoints > m_testMaxDescriptors)
        {
            result = ResultImplementationLimit();
        }

        // Build descriptors on the device
        DsInterface *pDsInterface = &m_DsInterfaces[m_numInterfaces];

        // Initialize DsIterface
        NNT_EXPECT_RESULT_SUCCESS(pDsInterface->Initialize(&m_DsClient, pUsbInterfaceDescriptor->bInterfaceNumber));

        // Full speed
        NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Full, pUsbInterfaceDescriptor, sizeof(UsbInterfaceDescriptor)));

        for (uint8_t i = 0; i < pUsbInterfaceDescriptor->bNumEndpoints; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Full, &pUsbEndpointDescriptorFullSpeed[i], sizeof(UsbEndpointDescriptor)));
        }

        // High speed
        NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_High, pUsbInterfaceDescriptor, sizeof(UsbInterfaceDescriptor)));

        for (uint8_t i = 0; i < pUsbInterfaceDescriptor->bNumEndpoints; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_High, &pUsbEndpointDescriptorHighSpeed[i], sizeof(UsbEndpointDescriptor)));
        }

        // Super speed
        NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Super, pUsbInterfaceDescriptor, sizeof(UsbInterfaceDescriptor)));

        for (uint8_t i = 0; i < pUsbInterfaceDescriptor->bNumEndpoints; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Super, &pUsbEndpointDescriptorSuperSpeed[i], sizeof(UsbEndpointDescriptor)));
            NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Super, &pUsbEndpointCompanionDescriptor[i], sizeof(UsbEndpointCompanionDescriptor)));
        }

        // Initialize the underlying USB interface.
        for (uint8_t i = 0; i < pUsbInterfaceDescriptor->bNumEndpoints; i++)
        {
            NN_LOG("Initialize USB endpoint..\n");
            NNT_EXPECT_RESULT_SUCCESS(m_EndPoints[m_numInterfaces][i].Initialize(&m_DsInterfaces[m_numInterfaces], pUsbEndpointDescriptorHighSpeed[i].bEndpointAddress));   // only end point address is used from pUsbEndpointDescriptorHighSpeed
        }

        m_numInterfaces++;

        return ResultSuccess();
    }

    static Result Finalize() NN_NOEXCEPT
    {
        Result result;

        // Disable USB device
        m_DsClient.DisableDevice();

        // Disable the USB interface.
        NN_LOG("Disable USB interface..\n");
        for (int i = 0; i < m_numInterfaces; i++)
        {
            result = m_DsInterfaces[i].Disable();
            if (result.IsFailure())
            {
                NN_LOG("Error: %d:%d \n", result.GetModule(), result.GetDescription());
            }
        }

        result = m_DsClient.Finalize();
        if (result.IsFailure())
        {
            NN_LOG("Error: %d:%d \n", result.GetModule(), result.GetDescription());
        }
        else
        {
            m_numInterfaces = 0;
        }

        return result;
    }

    // blocking call
    Result WaitForUsbConfigured()
    {
        Result result;
        nn::os::SystemEventType *pStateChangeEvent;

        NN_LOG("Enable USB interfaces..\n");
        for (int i = 0; i < m_numInterfaces; i++)
        {
            result = m_DsInterfaces[i].Enable();
            if (result.IsFailure())
            {
                NN_LOG("Error: %d:%d \n", result.GetModule(), result.GetDescription());
            }
        }

        // Enable USB device
        m_DsClient.EnableDevice();

        pStateChangeEvent = m_DsClient.GetStateChangeEvent();

        NN_LOG("Polling USB...\n");
        while (true)
        {
            UsbState usbState;

            nn::os::WaitSystemEvent(pStateChangeEvent);
            nn::os::ClearSystemEvent(pStateChangeEvent);

            result = m_DsClient.GetState(&usbState);

            switch (usbState)
            {
            case UsbState_Detached:
                break;

            case UsbState_Configured:
                NN_LOG("Configured\n");
                return result;

            default:
                break;
            }
        }
    }

};

}}}
