﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_Npad.h>

#include "Input.h"

namespace {
    // DebugPad か Npad を使用
    nn::hid::DebugPadButtonSet g_OldDebugPadButtonSet;
    nn::hid::DebugPadButtonSet g_CurrentDebugPadButtonSet;

    // Npad の場合は、No.1 コントローラか Handheld コントローラを使用
    nn::hid::NpadIdType g_NpadIds[] = {nn::hid::NpadId::No1, nn::hid::NpadId::Handheld};
    const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(nn::hid::NpadIdType);

    nn::hid::NpadButtonSet g_OldNpadButtonSet;
    nn::hid::NpadButtonSet g_CurrentNpadButtonSet;
}

namespace nnt { namespace usb { namespace pd { namespace input {
    namespace {
        template <typename T>
        bool IsDebugPadPushed()
        {
            nn::hid::DebugPadButtonSet pushed = (g_CurrentDebugPadButtonSet ^ g_OldDebugPadButtonSet & g_CurrentDebugPadButtonSet);
            return pushed.Test<T>();
        }

        template <typename T>
        bool IsNpadPushed()
        {
            nn::hid::NpadButtonSet pushed = (g_CurrentNpadButtonSet ^ g_OldNpadButtonSet & g_CurrentNpadButtonSet);
            return pushed.Test<T>();
        }
    }

    void Initialize()
    {
        nn::hid::InitializeDebugPad();
        nn::hid::InitializeNpad();

        // フルキーコンと Handheld コントローラをサポート
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
        nn::hid::SetSupportedNpadIdType(g_NpadIds, sizeof(g_NpadIds) / sizeof(nn::hid::NpadIdType));

        g_OldDebugPadButtonSet.Reset();
        g_CurrentDebugPadButtonSet.Reset();
        g_OldNpadButtonSet.Reset();
        g_CurrentNpadButtonSet.Reset();
    }

    void Finalize()
    {
    }

    void Update()
    {
        // current を old にコピー
        g_OldDebugPadButtonSet = g_CurrentDebugPadButtonSet;
        g_OldNpadButtonSet = g_CurrentNpadButtonSet;

        // current のリセット
        g_CurrentDebugPadButtonSet.Reset();
        g_CurrentNpadButtonSet.Reset();

        // DebugPad
        {
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            g_CurrentDebugPadButtonSet = state.buttons;
        }

        // Npad のコントローラを1つずつ調べて、最初に有効なもののデータを current に格納
        for(int i = 0; i < NpadIdCountMax; i++)
        {
            //現在有効な操作形態(NpadStyleSet)を取得
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);

            if (style.Test<nn::hid::NpadStyleFullKey>() == true)
            {
                nn::hid::NpadFullKeyState state;
                nn::hid::GetNpadState(&state, g_NpadIds[i]);

                g_CurrentNpadButtonSet = state.buttons;
                break;
            }
            if (style.Test<nn::hid::NpadStyleHandheld>() == true)
            {
                nn::hid::NpadHandheldState state;
                nn::hid::GetNpadState(&state, g_NpadIds[i]);

                g_CurrentNpadButtonSet = state.buttons;
                break;
            }
        }
    }

    // 上ボタンが押されたか
    bool IsUpPushed()
    {
        return IsDebugPadPushed<nn::hid::DebugPadButton::Up>() || IsNpadPushed<nn::hid::NpadButton::Up>();
    }

    // 下ボタンが押されたか
    bool IsDownPushed()
    {
        return IsDebugPadPushed<nn::hid::DebugPadButton::Down>() || IsNpadPushed<nn::hid::NpadButton::Down>();
    }

    // 決定ボタンが押されたか
    bool IsOkPushed()
    {
        return IsDebugPadPushed<nn::hid::DebugPadButton::A>() || IsNpadPushed<nn::hid::NpadButton::A>();
    }

    // キャンセルボタンが押されたか
    bool IsCancelPushed()
    {
        return IsDebugPadPushed<nn::hid::DebugPadButton::B>() || IsNpadPushed<nn::hid::NpadButton::B>();
    }
}}}}
